"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const iam = require("aws-cdk-lib/aws-iam");
// import * as defaults from '@aws-solutions-constructs/core';
require("@aws-cdk/assert/jest");
const iotTopicRuleProps = {
    topicRulePayload: {
        description: "sends data to kinesis streams",
        sql: "SELECT * FROM 'solutions/constructs'",
        actions: []
    }
};
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "sends data to kinesis streams",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        RetentionPeriodHours: 24,
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check existing kinesis stream', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {
        shardCount: 2,
        streamName: 'testexistingstream',
        retentionPeriod: cdk.Duration.hours(25)
    });
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 25,
        Name: 'testexistingstream'
    });
    expect(stack).not.toHaveResource('AWS::CloudWatch::Alarm');
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check new kinesis stream with override props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps,
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    const kinesisStream = construct.kinesisStream;
    expect(kinesisStream).toBeDefined();
});
test('check existing action in topic rule props', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const existingIamRole = new iam.Role(stack, 'existingRole', {
        assumedBy: new iam.ServicePrincipal('iot.amazonaws.com')
    });
    const existingKinesisActionProperty = {
        kinesis: {
            streamName: existingKinesisStream.streamName,
            roleArn: existingIamRole.roleArn
        }
    };
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: [existingKinesisActionProperty],
                ruleDisabled: true
            },
            ruleName: 'testiotrulename'
        },
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    // Check multiple actions in the Topic Rule
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                },
                {
                    Kinesis: {
                        RoleArn: {
                            "Fn::GetAtt": [
                                "existingRole3E995BBA",
                                "Arn"
                            ]
                        },
                        StreamName: {
                            Ref: "existingstream0A902451"
                        }
                    }
                }
            ],
            RuleDisabled: true,
            Sql: "SELECT * FROM 'solutions/constructs'"
        },
        RuleName: "testiotrulename"
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check name confict', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props);
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 2);
});
test('check construct chaining', () => {
    const stack = new cdk.Stack();
    const props1 = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props1);
    const props2 = {
        iotTopicRuleProps,
        existingStreamObj: construct.kinesisStream
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props2);
    expect(stack).toCountResources('AWS::Kinesis::Stream', 1);
});
test('check error when stream props and existing stream is supplied', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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