from __future__ import annotations
import socket
from typing import Optional as Opt


def findport() -> Opt[int]:
    """
    Find a free port (for UDP communication)

    Returns:
        the port number or None if no ports are available
    """
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
        s.bind(('', 0))
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        return s.getsockname()[1]
    except:
        s.close()
        return None


def udpsocket() -> socket.socket:
    """
    Creates a UDP socket

    Example::

        # send some data
        sock = udpsocket()
        sock.sendto(b"mymessage", ('192.168.1.3', 8888))
    """
    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return sock


def udpserver(port, ip='127.0.0.1') -> socket.socket:
    """
    Create a udp server

    Example::

        # receive some data
        sock = udpserver(8888)
        while True:
            # bufsize = 1024
            data, addr = sock.recvfrom(1024)

    https://wiki.python.org/moin/UdpCommunication
    """
    sock = udpsocket()
    sock.bind((ip, port))
    return sock 
