#![allow(non_snake_case)]

mod assert_arr;
mod original;
mod shared;
use assert_arr::EPS;

#[test]
fn test_alpha1() {
    let brine_properties = libdhe::FluidProperties {
        rho: 1037.0,
        c: 3905.0,
        lambda: 0.5,
        nu: 3.5e-06,
    };
    let Phi_m = 0.854;
    let D = 0.04;
    let thickness = 0.0037;
    let alpha = libdhe::alpha1(
        &brine_properties,
        Phi_m / brine_properties.rho,
        D,
        thickness,
    );
    assert_almost_eq!(alpha, 352.19673948754377);
}

#[test]
fn test_parametrize() {
    let dhe = [libdhe::DHE {
        D: 0.04,
        D_borehole: 0.135,
        L: 230.9,
        R1: 0.00136462160582245,
        Ra: 0.252072903027523,
        Rb: 0.0746663386837654,
        Phi_m: 0.854,
        brine_properties: libdhe::FluidProperties {
            c: 3905.0,
            lambda: 0.5,
            nu: 3.5e-06,
            rho: 1037.0,
        },
        T_soil_0_parameters: libdhe::model::TSoil0Parameters {
            d_DHE: 10.0,
            g_coefs: [4.82, 5.69, 6.29, 6.57, 6.6],
        },
        fill_properties: libdhe::model::MaterialProperties {
            c: 3040.0,
            lambda: 0.81,
            rho: 1180.0,
        },
        thickness: 0.0037,
        x: 0.0,
        y: 0.0,
    }];
    let g_method = libdhe::GConeParameters {};
    let env = libdhe::GlobalParameters {
        Gamma: 2.0,
        R: 2.5,
        adiabat: 0.0,
        dim_ax: 5,
        dim_rad: 5,
        dt: 3600.0,
        dt_boundary_refresh: 604800.0,
        n_steps_0: 100,
        optimal_n_steps_multiplier: 2.0,
        soil_layers: vec![
            libdhe::SoilLayerProperties {
                c: 1000.0,
                d: 150.0,
                lambda: 2.4,
                rho: 2600.0,
            },
            libdhe::SoilLayerProperties {
                c: 1000.0,
                d: 150.0,
                lambda: 2.4,
                rho: 2600.0,
            },
            libdhe::SoilLayerProperties {
                c: 1000.0,
                d: 150.0,
                lambda: 2.4,
                rho: 2600.0,
            },
            libdhe::SoilLayerProperties {
                c: 1000.0,
                d: 150.0,
                lambda: 2.4,
                rho: 2600.0,
            },
            libdhe::SoilLayerProperties {
                c: 1000.0,
                d: 150.0,
                lambda: 2.4,
                rho: 2600.0,
            },
        ],
        soil_parameters: libdhe::model::SoilParameters {
            T_grad: 0.03,
            T_soil_mean: 9.8,
        },
        t0: 0.0,
    };
    let t_boundary_refresh = libdhe::numerics::arange(
        env.dt_boundary_refresh,
        31532400. + env.dt_boundary_refresh,
        env.dt_boundary_refresh,
    );
    let statefull_dhe = dhe
        .iter()
        .map(|d| {
            d.with_state::<_, _, libdhe::TBrineCalcMethodDynamic>(
                &env,
                &g_method,
                &t_boundary_refresh,
            )
        })
        .collect::<Result<Vec<_>, _>>()
        .unwrap();
    let libdhe::StatefulDHE(dhe_0, state_0) = statefull_dhe.into_iter().next().unwrap();
    fn repeat(x: &[f64], n: usize) -> Vec<f64> {
        let mut out = Vec::<f64>::with_capacity(x.len() * n);
        for _x in x.iter() {
            for _ in 0..n {
                out.push(*_x);
            }
        }
        out
    }
    let g_ref = [
        0.0,
        0.0,
        0.12298132961836304,
        0.18687231578224872,
        0.24659878674334568,
        0.3015393054869499,
        0.3514677835375596,
        0.397341686373383,
        0.4396540639815625,
        0.47885588024528214,
        0.5153371071686327,
        0.549428699863836,
        0.5811330179559241,
        0.6112961463190627,
        0.6397726834800465,
        0.6667381059533435,
        0.6923423257696212,
        0.7167143416666374,
        0.7399658958045708,
        0.762194377043457,
        0.7834851469860719,
        0.803913419312799,
        0.8235457901991743,
        0.8424414938286691,
        0.8606534395440014,
        0.878229074214665,
        0.895211103684741,
        0.9116380998234758,
        0.9275450141059995,
        0.9429636143536583,
        0.9579228579370509,
        0.9724492121518603,
        0.9865669304424026,
        1.0002982915399943,
        1.013663807305105,
        1.0266824040401417,
        1.0393715812176725,
        1.0517475509041634,
        1.063825360618986,
        1.0756190019270826,
        1.0871415067013905,
        1.0984050326924384,
        1.1094209397951726,
        1.120199858197355,
        1.1307517494221526,
        1.141085961133583,
        1.1512112764523896,
        1.161135958427696,
        1.170867790223182,
        1.18041411150294,
        1.189781851439388,
        1.1989775587119929,
        1.2080074288194915,
    ];
    let dhe_ref = libdhe::DHECore {
        x: 0.0,
        y: 0.0,
        L: 230.9,
        R: 2.5,
        L1_on: 732.8038745197101,
        n_steps: 3,
        pump_dependent_parameters: [
            libdhe::TSoilParameters {
                L: vec![572.4040585563218; 5],
                T_soil_tensor: vec![
                    0.304487652605313,
                    0.3640060027135161,
                    0.32344915344741776,
                    0.008013154535698208,
                    4.3978913923256835e-05,
                    5.7729233389312885e-08,
                    5.489835750412128e-11,
                    0.00611112464251796,
                    0.027375857853027236,
                    0.918717336667827,
                    0.04753445283492464,
                    0.00026088522320431086,
                    3.4245283920529646e-07,
                    3.256599350317043e-10,
                    2.8704987619414383e-05,
                    0.00012858904157749473,
                    0.009012540345030906,
                    0.9799492538614852,
                    0.010866634035331717,
                    1.4264164264642657e-05,
                    1.356469059647744e-08,
                    3.514069640447911e-08,
                    1.5741893119513114e-07,
                    1.103316776502103e-05,
                    0.0024238573640479014,
                    0.9949437495770016,
                    0.0026186770668972692,
                    2.490264660832717e-06,
                    1.0942244197264381e-11,
                    4.901770774781598e-11,
                    3.4355498981761095e-09,
                    7.547499591775046e-07,
                    0.00062119320051083,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.304487652605313,
                    0.3640060027135161,
                    0.32344915344741776,
                    0.008013154535698208,
                    4.3978913923256835e-05,
                    5.7729233389312885e-08,
                    5.489835750412128e-11,
                    0.00611112464251796,
                    0.027375857853027236,
                    0.918717336667827,
                    0.04753445283492464,
                    0.00026088522320431086,
                    3.4245283920529646e-07,
                    3.256599350317043e-10,
                    2.8704987619414383e-05,
                    0.00012858904157749473,
                    0.009012540345030906,
                    0.9799492538614852,
                    0.010866634035331717,
                    1.4264164264642657e-05,
                    1.356469059647744e-08,
                    3.514069640447911e-08,
                    1.5741893119513114e-07,
                    1.103316776502103e-05,
                    0.0024238573640479014,
                    0.9949437495770016,
                    0.0026186770668972692,
                    2.490264660832717e-06,
                    1.0942244197264381e-11,
                    4.901770774781598e-11,
                    3.4355498981761095e-09,
                    7.547499591775046e-07,
                    0.00062119320051083,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.304487652605313,
                    0.3640060027135161,
                    0.32344915344741776,
                    0.008013154535698208,
                    4.3978913923256835e-05,
                    5.7729233389312885e-08,
                    5.489835750412128e-11,
                    0.00611112464251796,
                    0.027375857853027236,
                    0.918717336667827,
                    0.04753445283492464,
                    0.00026088522320431086,
                    3.4245283920529646e-07,
                    3.256599350317043e-10,
                    2.8704987619414383e-05,
                    0.00012858904157749473,
                    0.009012540345030906,
                    0.9799492538614852,
                    0.010866634035331717,
                    1.4264164264642657e-05,
                    1.356469059647744e-08,
                    3.514069640447911e-08,
                    1.5741893119513114e-07,
                    1.103316776502103e-05,
                    0.0024238573640479014,
                    0.9949437495770016,
                    0.0026186770668972692,
                    2.490264660832717e-06,
                    1.0942244197264381e-11,
                    4.901770774781598e-11,
                    3.4355498981761095e-09,
                    7.547499591775046e-07,
                    0.00062119320051083,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.304487652605313,
                    0.3640060027135161,
                    0.32344915344741776,
                    0.008013154535698208,
                    4.3978913923256835e-05,
                    5.7729233389312885e-08,
                    5.489835750412128e-11,
                    0.00611112464251796,
                    0.027375857853027236,
                    0.918717336667827,
                    0.04753445283492464,
                    0.00026088522320431086,
                    3.4245283920529646e-07,
                    3.256599350317043e-10,
                    2.8704987619414383e-05,
                    0.00012858904157749473,
                    0.009012540345030906,
                    0.9799492538614852,
                    0.010866634035331717,
                    1.4264164264642657e-05,
                    1.356469059647744e-08,
                    3.514069640447911e-08,
                    1.5741893119513114e-07,
                    1.103316776502103e-05,
                    0.0024238573640479014,
                    0.9949437495770016,
                    0.0026186770668972692,
                    2.490264660832717e-06,
                    1.0942244197264381e-11,
                    4.901770774781598e-11,
                    3.4355498981761095e-09,
                    7.547499591775046e-07,
                    0.00062119320051083,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.304487652605313,
                    0.3640060027135161,
                    0.32344915344741776,
                    0.008013154535698208,
                    4.3978913923256835e-05,
                    5.7729233389312885e-08,
                    5.489835750412128e-11,
                    0.00611112464251796,
                    0.027375857853027236,
                    0.918717336667827,
                    0.04753445283492464,
                    0.00026088522320431086,
                    3.4245283920529646e-07,
                    3.256599350317043e-10,
                    2.8704987619414383e-05,
                    0.00012858904157749473,
                    0.009012540345030906,
                    0.9799492538614852,
                    0.010866634035331717,
                    1.4264164264642657e-05,
                    1.356469059647744e-08,
                    3.514069640447911e-08,
                    1.5741893119513114e-07,
                    1.103316776502103e-05,
                    0.0024238573640479014,
                    0.9949437495770016,
                    0.0026186770668972692,
                    2.490264660832717e-06,
                    1.0942244197264381e-11,
                    4.901770774781598e-11,
                    3.4355498981761095e-09,
                    7.547499591775046e-07,
                    0.00062119320051083,
                    0.9974785206459719,
                    0.0018995279080481666,
                ],
                T_brine_method: libdhe::TBrineDynamicParameters {
                    kappa_ax: 0.0,
                    kappa_rad: vec![0.004970991690489825; 5],
                    lambda_brine: vec![1.9469525801235437; 5],
                    n_sub_steps: 147,
                },
            },
            libdhe::TSoilParameters {
                L: vec![732.8038745197101; 5],
                T_soil_tensor: vec![
                    0.3738618988190867,
                    0.308195799729907,
                    0.31021478140443165,
                    0.007685285171277033,
                    4.217945548379778e-05,
                    5.5367161501699346e-08,
                    5.2652114841202965e-11,
                    0.007503478854472309,
                    0.026255736548144108,
                    0.9184517203236889,
                    0.047527872434914,
                    0.0002608491077338854,
                    3.4240543197680523e-07,
                    3.256148525176088e-10,
                    3.5245111206146084e-05,
                    0.0001233276420692588,
                    0.009011292700069298,
                    0.9799492229522317,
                    0.010866633865691233,
                    1.4264164041962872e-05,
                    1.3564690384717231e-08,
                    4.314712721909028e-08,
                    1.5097791664975138e-07,
                    1.1031640395861435e-05,
                    0.0024238573262087386,
                    0.9949437495767938,
                    0.0026186770668969964,
                    2.4902646608324574e-06,
                    1.3435317189147695e-11,
                    4.701208004987731e-11,
                    3.4350743001364103e-09,
                    7.547499473950016e-07,
                    0.0006211932005107654,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.3738618988190867,
                    0.308195799729907,
                    0.31021478140443165,
                    0.007685285171277033,
                    4.217945548379778e-05,
                    5.5367161501699346e-08,
                    5.2652114841202965e-11,
                    0.007503478854472309,
                    0.026255736548144108,
                    0.9184517203236889,
                    0.047527872434914,
                    0.0002608491077338854,
                    3.4240543197680523e-07,
                    3.256148525176088e-10,
                    3.5245111206146084e-05,
                    0.0001233276420692588,
                    0.009011292700069298,
                    0.9799492229522317,
                    0.010866633865691233,
                    1.4264164041962872e-05,
                    1.3564690384717231e-08,
                    4.314712721909028e-08,
                    1.5097791664975138e-07,
                    1.1031640395861435e-05,
                    0.0024238573262087386,
                    0.9949437495767938,
                    0.0026186770668969964,
                    2.4902646608324574e-06,
                    1.3435317189147695e-11,
                    4.701208004987731e-11,
                    3.4350743001364103e-09,
                    7.547499473950016e-07,
                    0.0006211932005107654,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.3738618988190867,
                    0.308195799729907,
                    0.31021478140443165,
                    0.007685285171277033,
                    4.217945548379778e-05,
                    5.5367161501699346e-08,
                    5.2652114841202965e-11,
                    0.007503478854472309,
                    0.026255736548144108,
                    0.9184517203236889,
                    0.047527872434914,
                    0.0002608491077338854,
                    3.4240543197680523e-07,
                    3.256148525176088e-10,
                    3.5245111206146084e-05,
                    0.0001233276420692588,
                    0.009011292700069298,
                    0.9799492229522317,
                    0.010866633865691233,
                    1.4264164041962872e-05,
                    1.3564690384717231e-08,
                    4.314712721909028e-08,
                    1.5097791664975138e-07,
                    1.1031640395861435e-05,
                    0.0024238573262087386,
                    0.9949437495767938,
                    0.0026186770668969964,
                    2.4902646608324574e-06,
                    1.3435317189147695e-11,
                    4.701208004987731e-11,
                    3.4350743001364103e-09,
                    7.547499473950016e-07,
                    0.0006211932005107654,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.3738618988190867,
                    0.308195799729907,
                    0.31021478140443165,
                    0.007685285171277033,
                    4.217945548379778e-05,
                    5.5367161501699346e-08,
                    5.2652114841202965e-11,
                    0.007503478854472309,
                    0.026255736548144108,
                    0.9184517203236889,
                    0.047527872434914,
                    0.0002608491077338854,
                    3.4240543197680523e-07,
                    3.256148525176088e-10,
                    3.5245111206146084e-05,
                    0.0001233276420692588,
                    0.009011292700069298,
                    0.9799492229522317,
                    0.010866633865691233,
                    1.4264164041962872e-05,
                    1.3564690384717231e-08,
                    4.314712721909028e-08,
                    1.5097791664975138e-07,
                    1.1031640395861435e-05,
                    0.0024238573262087386,
                    0.9949437495767938,
                    0.0026186770668969964,
                    2.4902646608324574e-06,
                    1.3435317189147695e-11,
                    4.701208004987731e-11,
                    3.4350743001364103e-09,
                    7.547499473950016e-07,
                    0.0006211932005107654,
                    0.9974785206459719,
                    0.0018995279080481666,
                    0.3738618988190867,
                    0.308195799729907,
                    0.31021478140443165,
                    0.007685285171277033,
                    4.217945548379778e-05,
                    5.5367161501699346e-08,
                    5.2652114841202965e-11,
                    0.007503478854472309,
                    0.026255736548144108,
                    0.9184517203236889,
                    0.047527872434914,
                    0.0002608491077338854,
                    3.4240543197680523e-07,
                    3.256148525176088e-10,
                    3.5245111206146084e-05,
                    0.0001233276420692588,
                    0.009011292700069298,
                    0.9799492229522317,
                    0.010866633865691233,
                    1.4264164041962872e-05,
                    1.3564690384717231e-08,
                    4.314712721909028e-08,
                    1.5097791664975138e-07,
                    1.1031640395861435e-05,
                    0.0024238573262087386,
                    0.9949437495767938,
                    0.0026186770668969964,
                    2.4902646608324574e-06,
                    1.3435317189147695e-11,
                    4.701208004987731e-11,
                    3.4350743001364103e-09,
                    7.547499473950016e-07,
                    0.0006211932005107654,
                    0.9974785206459719,
                    0.0018995279080481666,
                ],
                T_brine_method: libdhe::TBrineDynamicParameters {
                    kappa_ax: 0.009993715395963878,
                    kappa_rad: vec![0.0010980088223843225; 5],
                    lambda_brine: vec![0.4300492221359801; 5],
                    n_sub_steps: 852,
                },
            },
        ],
        d_lambda_soil: vec![110.832, 110.832, 110.832, 110.832, 110.832],
        g: repeat(&g_ref, 5),
    };
    assert_eq!(
        (dhe_0.x, dhe_0.y, dhe_0.L, dhe_0.R, dhe_0.n_steps),
        (dhe_ref.x, dhe_ref.y, dhe_ref.L, dhe_ref.R, dhe_ref.n_steps)
    );
    assert_almost_eq!(dhe_0.L1_on, dhe_ref.L1_on);
    assert_almost_eq_arr!(dhe_0.d_lambda_soil, dhe_ref.d_lambda_soil);
    assert_almost_eq_arr!(dhe_0.g, dhe_ref.g);
    /*
        fn move_axis(X: &[f64], dims: (usize, usize, usize)) -> Vec<f64> {
        let mut Y = Vec::with_capacity(X.len());
        for i in 0..dims.0 {
            for j in 0..dims.1 {
            for k in 0..dims.2 {
                Y.push(X[j*dims.2*dims.0 + k*dims.0 + i])
            }
            }
        }
        Y
        }
    */
    for (p, p_ref) in dhe_0
        .pump_dependent_parameters
        .iter()
        .zip(dhe_ref.pump_dependent_parameters.iter())
    {
        assert_almost_eq_arr!(p.L, p_ref.L);
        //	assert_almost_eq_arr!(p.T_soil_tensor, move_axis(&p_ref.T_soil_tensor, (env.dim_ax, env.dim_rad, env.dim_rad + 2)));
        assert_almost_eq_arr!(p.T_soil_tensor, p_ref.T_soil_tensor);
        assert_eq!(
            p.T_brine_method.n_sub_steps,
            p_ref.T_brine_method.n_sub_steps
        );
        assert_almost_eq!(p.T_brine_method.kappa_ax, p_ref.T_brine_method.kappa_ax);
        assert_almost_eq_arr!(p.T_brine_method.kappa_rad, p_ref.T_brine_method.kappa_rad);
        assert_almost_eq_arr!(
            p.T_brine_method.lambda_brine,
            p_ref.T_brine_method.lambda_brine
        );
    }
    let T_U_ref = [
        10.492700000000001,
        11.8781,
        13.2635,
        14.648900000000001,
        16.0343,
        16.0343,
        14.648900000000001,
        13.2635,
        11.8781,
        10.492700000000001,
    ];
    let mut T_soil_ref = Vec::with_capacity(env.dim_rad * (env.dim_rad + 2));
    for _ in 0..env.dim_rad + 2 {
        for i in 0..env.dim_rad {
            T_soil_ref.push(T_U_ref[i]);
        }
    }
    assert_almost_eq_arr!(state_0.T_U, T_U_ref);
    assert_almost_eq_arr!(state_0.T_soil, T_soil_ref);
}
