#![allow(non_snake_case)]

use std::f64::NAN;

mod assert_arr;
use assert_arr::EPS;
mod shared;

const DIM_AX: usize = 1;
const DIM_RAD: usize = 5;
const DIM_T: usize = 3;

#[test]
fn test_calc_P() {
    let P = [500., 500., 500.];
    let U_brine = [1550., 1550., 1550.];
    let dim_ax = DIM_AX;
    let dim_rad = DIM_RAD;
    let dim_t = DIM_T;
    let n_boundary_refresh = 168;
    let precision = 0.050000000000000003;
    let T_soil_tensor_off = vec![
        0.59397874051346522,
        0.29734488581162227,
        0.10581642441735493,
        0.0028419850922479966,
        1.7936588314892776e-05,
        2.7546141825400331e-08,
        3.0852939451901944e-11,
        0.008657607331667273,
        0.018909603709712432,
        0.9205258518791879,
        0.051580894651949261,
        0.00032554191603941575,
        4.9995147526961115e-07,
        5.5996853182390063e-10,
        4.913393965609493e-05,
        0.00010731640879521776,
        0.010899431875253459,
        0.97645098670797781,
        0.012473952720527192,
        1.9156891195282716e-05,
        2.1456594824823218e-08,
        7.1956287394426816e-08,
        1.5716407858714374e-07,
        1.5962136517876929e-05,
        0.0028944976978436932,
        0.99402355649857899,
        0.003062324602995571,
        3.4299436980016677e-06,
        2.6668972346233207e-11,
        5.8249315208366271e-11,
        5.9160052970580311e-09,
        1.072780181624656e-06,
        0.00073903978389686911,
        0.99702312414521688,
        0.0022367572897808168,
    ];
    let T_soil_tensor_on = vec![
        0.4048048394700921,
        0.47189686458986663,
        0.12005355324199483,
        0.0032243615342685116,
        2.034987642134459e-05,
        3.1252352576230658e-08,
        3.5004065101952753e-11,
        0.0059002807795125369,
        0.021453806705801489,
        0.9207333668329849,
        0.051586468024971056,
        0.00032557709120542555,
        5.0000549558242791e-07,
        5.6002903704641342e-10,
        3.3485468752341334e-05,
        0.00012175535384016725,
        0.01090060957098705,
        0.97645101833816916,
        0.012473952920154458,
        1.9156891501860582e-05,
        2.1456595168204463e-08,
        4.9039218714098792e-08,
        1.78309805687369e-07,
        1.5963861244476553e-05,
        0.0028944977441658715,
        0.9940235564988712,
        0.0030623246029960194,
        3.4299436980021704e-06,
        1.8175278563197203e-11,
        6.6086501251410785e-11,
        5.9166445280085962e-09,
        1.0727801987929248e-06,
        0.00073903978389697742,
        0.99702312414521688,
        0.0022367572897808168,
    ];

    let L_off = vec![4523.7301988752424];
    let L_on = vec![2717.3748643622798];
    let prm_off = libdhe::TBrineDynamicParameters {
        n_sub_steps: 26,
        kappa_ax: 0.,
        kappa_rad: vec![0.12081393309520569],
        lambda_brine: vec![86.994811516831589],
    };
    let prm_on = libdhe::TBrineDynamicParameters {
        n_sub_steps: 16,
        kappa_ax: 0.13453503219940435,
        kappa_rad: vec![0.11792971447575207],
        lambda_brine: vec![84.917964511321244],
    };

    let T_soil = vec![
        11.300000000000001,
        11.300000000000001,
        11.300000000000001,
        11.300000000000001,
        11.300000000000001,
        11.300000000000001,
        11.300000000000001,
    ];
    let T_U = vec![0.87918606690479428, 0.87918606690479428];
    let mut sum_Q0 = [0.5];
    let mut Q_wall = [0.];
    let Q = vec![0., 0.00027482746658799145, 8.6957128100106682e-05];
    let d_lambda_soil = vec![200.];
    let g = vec![
        0.22507230641451292,
        0.34866754443584563,
        0.45005589230285686,
    ];
    let mut out_T_sink = vec![NAN; dim_t];
    let mut out_T_source = vec![NAN; dim_t];
    let mut out_T_soil = vec![NAN; dim_t * dim_ax * (dim_rad + 2)];
    let dhe = [libdhe::DHECore {
        d_lambda_soil,
        g,
        L: NAN,
        R: NAN,
        x: NAN,
        y: NAN,
        pump_dependent_parameters: [
            libdhe::TSoilParameters {
                T_soil_tensor: T_soil_tensor_off,
                L: L_off,
                T_brine_method: prm_off,
            },
            libdhe::TSoilParameters {
                T_soil_tensor: T_soil_tensor_on,
                L: L_on,
                T_brine_method: prm_on,
            },
        ],
        L1_on: 2717.3748643622798,
        n_steps: 6,
    }];
    let mut dhe_states = [libdhe::DHEState {
        T_soil,
        T_U,
        Q,
        T_sink: NAN,
    }];
    let mut out = [libdhe::CalcPOutputRefMut {
        T_sink: &mut out_T_sink,
        T_source: &mut out_T_source,
        T_soil: &mut out_T_soil,
    }];
    libdhe::calc_P_core(
        &P,
        &U_brine,
        dim_ax,
        dim_rad,
        &dhe,
        &mut dhe_states,
        &mut sum_Q0,
        &mut Q_wall,
        n_boundary_refresh,
        precision,
        &mut out,
    );

    let T_sink_ref = [
        9.300160858675422304,
        9.723971476460661734,
        9.819883852268217694,
    ];
    let T_source_ref = [
        9.622741503836712340,
        10.04655212162195177,
        10.14246449742950773,
    ];
    let T_soil_ref: [[f64; DIM_AX * (DIM_RAD + 2)]; DIM_T] = [
        [
            9.81202450206,
            10.13546875734,
            11.14439970418,
            11.29489380946,
            11.29997040295,
            11.29999996671,
            11.3,
        ],
        [
            10.01618410089,
            10.25686527103,
            11.04916163617,
            11.28212369308,
            11.29977835342,
            11.29999949003,
            11.3,
        ],
        [
            10.09481789394,
            10.30753833297,
            10.99858288058,
            11.26659668638,
            11.29934322798,
            11.29999764987,
            11.3,
        ],
    ];
    assert_almost_eq_arr!(out[0].T_sink, T_sink_ref);
    assert_almost_eq_arr!(out[0].T_source, T_source_ref);
    assert_almost_eq_arr!(
        out[0].T_soil,
        flatten_matrix_raw!(DIM_T, DIM_AX * (DIM_RAD + 2))(&T_soil_ref)
    );
}
