'''
# Crow API

Crow API lets you build an API intuitively based on the file structure of a project. Provide API Gateway and Lambda function configurations and crow will build out the appropriate paths and methods to the API Gateway. All created resources are available after initialization. `lambdaFunctions` will expose all Lambda functions created for further operations like adding environment variables and providing permissions.

| `crow-api` version | `aws-cdk` version | Notes                   |
| ------------------ | ----------------- | ----------------------- |
| 0                  | 1                 |                         |
| 1                  | 2                 | Not recommended for use |
| 2                  | 2                 |                         |

Contents:

* [Getting Started](#getting-started)
* [Example File Structure](#example-file-structure)
* [Crow API Props](#crow-api-props)

  * [`sourceDirectory`](#sourcedirectory)
  * [`sharedDirectory`](#shareddirectory)
  * [`useAuthorizerLambda`](#useauthorizerlambda)
  * [`authorizerDirectory`](#authorizerdirectory)
  * [`authorizerLambdaConfiguration`](#authorizerlambdaconfiguration)
  * [`tokenAuthorizerConfiguration`](#tokenauthorizerconfiguration)
  * [`createApiKey`](#createapikey)
  * [`logRetention`](#logretention)
  * [`apiGatewayName`](#apigatewayname)
  * [`apiGatewayConfiguration`](#apigatewayconfiguration)
  * [`lambdaConfigurations`](#lambdaconfigurations)
  * [`lambdaIntegrationOptions`](#lambdaintegrationoptions)
  * [`models`](#models)
  * [`requestValidators`](#requestValidators)
  * [`methodConfigurations`](#methodconfigurations)
* [Properties](#properties)

  * [`authorizerLambda`](#authorizerlambda)
  * [`gateway`](#gateway)
  * [`lambdaLayer`](#lambdalayer)
  * [`lambdaFunctions`](#lambdafunctions)
  * [`models`](#models)
  * [`requestValidators`](#requestValidators)

## Getting Started

[Start your application as a normal CDK app](https://docs.aws.amazon.com/cdk/v2/guide/getting_started.html)

```sh
npm install -g aws-cdk
cdk bootstrap # If this is your first cdk app, you will need to bootstrap your AWS account
cdk init app --language typescript
```

Next, install the Crow API package

```sh
npm install --save crow-api
```

In the `lib/` folder generated by the `cdk`, there should be a single file named `<your-app>-stack.js`. Create your Crow API construct inside of that file like so

```python
# Example automatically generated from non-compiling source. May contain errors.
from aws_cdk import Stack, StackProps
from constructs import Construct
from crow_api import CrowApi, ICrowApiProps

class YourAppStack(Stack):
    def __init__(self, scope, id, props):
        super().__init__(scope, id, props)

        {
              crowApiProps,
            } = props

        api = CrowApi(self, "api", {
            (SpreadAssignment ...crowApiProps
              crow_api_props)
        })
```

Your API will start to take shape as you create folders to define paths and methods (see Example File Structure below). To deploy your API, simply run `cdk synth` and `cdk deploy`. Follow the instructions as they are prompted, and you will end up receiving a URL where your API now lives.

## Example File Structure

```
|-- src/
    |-- authorizer/
        |-- index.js
    |-- v1/
        |-- book/
            |-- get/
                |-- index.js
            |-- post/
                |-- index.js
        |-- chapters/
            |-- get/
                |-- index.js
        |-- authors/
            |-- get/
                |-- index.js
            |-- post/
                |-- index.js
```

The preceding file structure will create an API with the following routes:

* GET /v1/book
* POST /v1/book
* GET /v1/book/chapters
* GET /v1/authors
* POST /v1/authors

There needs to be an `index.js` file inside of a folder named after an HTTP method in order for a path to be created. The `index.js` file needs to export a `handler` method that will process the payload and return like the following.

```javascript
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
```

## Crow API Props

Crow API takes in a few props to help you customize away from defaults.

#### `sourceDirectory`

By default, Crow walks through the `src` directory in the root of the repository to determine routes and methods, but you can change the top level directory by passing in the `sourceDirectory` prop. The string passed in should not start with or end with a slash (`/`). For example, `src`, `api/src`, or `source` are all valid options to pass in through that prop.

#### `sharedDirectory`

By default, Crow creates a Lambda layer out of the `shared` directory in the source directory of the repository, but you can change the name of the shared directory by passing in the `sharedDirectory` prop. The string passed in should not start with or end with a slash (`/`) and must be a direct child of the source directory. For example, `common` or `utils` are valid but `shared/utils` is not.

The Lambda layer created will be prepended to any the of the layers passed in through `lambdaConfigurations` and added to all Lambda functions created.

#### `useAuthorizerLambda`

Crow will create and attach an authorizer Lambda to specific methods if requested. The `useAuthorizerLambda` prop tells the `CrowApi` Construct that it should create an authorizer Lambda and accepts a boolean value. This is `false` by default.

#### `authorizerDirectory`

Crow will allow for a Lambda authorizer to be created and used by specific methods if requested. The `authorizerDirectory` prop tells Crow where to find the code for the Lambda authorizer **within the source directory which can be specified in the `sourceDirectory` prop**. It expects to find an `index.js` file that exports a `handler` function within the `authorizerDirectory`.

By default, Crow expects to find a directory called `src/authorizer` containing the authorizer Lambda source if the `useAuthorizerLambda` prop is `true`. If a different directory within the source directory should be looked at for this code, it should be specified by passing in a string to the `authorizerDirectory` prop. The string passed in should not start with nor end with a slash (`/`). For example, `auth` or `authLambdaSrc` are valid.

#### `authorizerLambdaConfiguration`

The `authorizerLambdaConfiguration` prop is passed directly to the Lambda functions which will be in charge of your API's authorization. The configuration allowed is exactly the same as the [Lambda Function props](https://docs.aws.amazon.com/cdk/api/v2//docs/aws-cdk-lib.aws_lambda.Function.html).

#### `tokenAuthorizerConfiguration`

The `tokenAuthorizerConfiguration` prop is passed directly to the `APIGateway.TokenAuthorizer` construct which will be in charge of your API's authorization. Anything available in the [class constructor for the `TokenAuthorizer`](https://docs.aws.amazon.com/cdk/api/v2//docs/aws-cdk-lib.aws_apigateway.TokenAuthorizer.html) can be overridden.

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK, API Gateway, and Lambda.

#### `createApiKey`

By default, Crow does not create an API key associated with the API. If an API key is desired, pass in the `createApiKey` prop as `true`.

#### `logRetention`

By default, Crow creates log groups for resources it creates and sets the log retention to one week. If a different retention is desired pass in the `logRetention` prop of [enum type `RetentionDays`](https://docs.aws.amazon.com/cdk/api/v2//docs/aws-cdk-lib.aws_logs.RetentionDays.html).

#### `apiGatewayConfiguration`

This props allows for more complex overrides to the API Gateway that fronts your API. The configuration allowed is exactly the same as the [RestApi props](https://docs.aws.amazon.com/cdk/api/v2//docs/aws-cdk-lib.aws_apigateway.RestApi.html).

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and API Gateway.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
import aws_cdk.aws_apigateway as apigateway
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    api_gateway_configuration={
        "endpoint_configuration": {
            "types": [apigateway.EndpointType.REGIONAL]
        }
    }
)
```

#### `apiGatewayName`

This is a simple prop that names the API Gateway. This is how the API will be identified in the AWS console. The value should be a string without spaces and defaults to `crow-api`.

#### `lambdaConfigurations`

This props allows for more complex overrides to Lambda functions. The prop is an object with keys corresponding to the API path of a Lambda function and a value corresponding to the configuration that should be applied to the Lambda. The configuration allowed is exactly the same as the [Lambda Function props](https://docs.aws.amazon.com/cdk/api/v2//docs/aws-cdk-lib.aws_lambda.Function.html).

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and Lambda.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    lambda_configurations={
        "/v1/book/get": {
            "timeout": cdk.Duration.seconds(5)
        }
    }
)
```

#### `lambdaIntegrationOptions`

This prop is an object with keys corresponding to the API path of a Lambda function and a value corresponding to the configuration that should be applied to the Lambda Integration. The configuration allowed is exactly the same as the [LambdaIntegrationOptions](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_apigateway.LambdaIntegrationOptions.html).

#### `models`

This prop helps set up the `Model`s used in `methodConfiguration` above. It is an array of `CrowModelOptions` which are the same as [`MethodOptions`](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_apigateway.ModelOptions.html) except that the `modelName` is required. The `Model`s will receive an ID equal to its `modelName` which is why that prop is required. The `IModel` can then be referenced in `methodConfigurations` using its `modelName`.

#### `requestValidators`

This prop helps set up the `RequestValidator`s used in `methodConfiguration` above. It is an array of `CrowRequestValidatorOptions` which are the same as [`RequestValidatorOptions`](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_apigateway.RequestValidatorOptions.html) except that the `requestValidatorName` is required. The `RequestValidator`s will receive an ID equal to its `requestValidatorName` which is why that prop is required. The `IRequestValidator` can then be referenced in `methodConfigurations` using its `requestValidatorName`.

#### `methodConfigurations`

This prop allows for more complex overrides to individual methods. The prop is an object with keys corresponding to the API path of a method and a value corresponding to the configuration that should be applied to the method as well as the key `useAuthorizerLambda` which will invoke the authorizer Lambda whenever the method is called. The configuration allowed is almost exactly the same as [`MethodOptions`](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_apigateway.MethodOptions.html) plus the `useAuthorizerLambda` boolean.

The differences between `MethodOptions` and Crow's `CrowMethodConfiguration` (the type for this prop) is that any value referencing `{ [string]: IModel }` (`MethodOptions.requestModels` and `MethodResponse.responseModels`) has been changed to `{ [string]: string }` and similarly `requestValidator` has been changed from `IRequestValidator` to `string`. The strings that are passed should correspond with the `modelName`s or `requestValidatorName`s used in the [`models`](#models) and [`requestValidators`](#requestvalidators) props (see next sections).

**Note:**

If `createApiKey` is `true`, then the `apiKeyRequired` parameter will need to be set for the methods needing the API key.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    models=[{
        "model_name": "authorsPost",
        "schema": {
            "schema": apigateway.JsonSchemaVersion.DRAFT4,
            "title": "/v1/authors/post",
            "type": apigateway.JsonSchemaType.OBJECT,
            "required": ["name"],
            "properties": {
                "name": {
                    "type": apigateway.JsonSchemaType.STRING
                }
            }
        }
    }
    ],
    method_configurations={
        "/v1/authors/post": {
            "api_key_required": True,
            "request_models": {
                "application/json": "authorsPost"
            }
        },
        "/v1/book/get": {
            "use_authorizer_lambda": True
        },
        "/v1/book/post": {
            "api_key_required": True
        }
    }
)
```

## Properties

A `CrowApi` construct will give full access to all of the resources it created.

#### `authorizerLambda`

This is the `lambda.Function` that authorizes API Gateway requests.

#### `gateway`

This is the `apigateway.RestApi` that all of the created Lambda functions sit behind.

#### `lambdaLayer`

If the `sharedDirectory` is populated, this is the `lambda.LayerVersion` created for that code. If the `sharedDirectory` is not populated, then this is `undefined`.

#### `lambdaFunctions`

This is an object with keys being the API paths and the values being the `lambda.Function`s sitting being them. Continuing off of the example file structure from above, the following would be an example of referencing `GET` `/v1/book/chapters`.

```python
# Example automatically generated from non-compiling source. May contain errors.
from aws_cdk import Stack, StackProps
from constructs import Construct
from crow_api import CrowApi, ICrowApiProps

class YourAppStack(Stack):
    def __init__(self, scope, id, props):
        super().__init__(scope, id, props)

        {
              crowApiProps,
            } = props

        api = CrowApi(self, "api", {
            (SpreadAssignment ...crowApiProps
              crow_api_props)
        })

        lambda_ = api.lambda_functions["/v1/book/chapters/get"]
        lambda_.add_environment("FOO", "bar")
```

#### `models`

This is an object with keys being the `modelName`s and values being the `IModel`s created.

#### `requestValidators`

This is an object with keys being the `requestValidatorName`s and values being the `IRequestValidator`s created.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.aws_apigateway
import aws_cdk.aws_lambda
import aws_cdk.aws_logs
import constructs


class CrowApi(
    constructs.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="crow-api.CrowApi",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        api_gateway_configuration: typing.Any = None,
        api_gateway_name: typing.Optional[builtins.str] = None,
        authorizer_directory: typing.Optional[builtins.str] = None,
        authorizer_lambda_configuration: typing.Any = None,
        create_api_key: typing.Optional[builtins.bool] = None,
        lambda_configurations: typing.Optional["CrowLambdaConfigurations"] = None,
        lambda_integration_options: typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.LambdaIntegrationOptions]] = None,
        log_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        method_configurations: typing.Optional["CrowMethodConfigurations"] = None,
        models: typing.Optional[typing.Sequence["CrowModelOptions"]] = None,
        request_validators: typing.Optional[typing.Sequence["CrowRequestValidatorOptions"]] = None,
        shared_directory: typing.Optional[builtins.str] = None,
        source_directory: typing.Optional[builtins.str] = None,
        token_authorizer_configuration: typing.Any = None,
        use_authorizer_lambda: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param api_gateway_configuration: 
        :param api_gateway_name: 
        :param authorizer_directory: 
        :param authorizer_lambda_configuration: 
        :param create_api_key: 
        :param lambda_configurations: 
        :param lambda_integration_options: 
        :param log_retention: 
        :param method_configurations: 
        :param models: 
        :param request_validators: 
        :param shared_directory: 
        :param source_directory: 
        :param token_authorizer_configuration: 
        :param use_authorizer_lambda: 

        :stability: experimental
        '''
        props = CrowApiProps(
            api_gateway_configuration=api_gateway_configuration,
            api_gateway_name=api_gateway_name,
            authorizer_directory=authorizer_directory,
            authorizer_lambda_configuration=authorizer_lambda_configuration,
            create_api_key=create_api_key,
            lambda_configurations=lambda_configurations,
            lambda_integration_options=lambda_integration_options,
            log_retention=log_retention,
            method_configurations=method_configurations,
            models=models,
            request_validators=request_validators,
            shared_directory=shared_directory,
            source_directory=source_directory,
            token_authorizer_configuration=token_authorizer_configuration,
            use_authorizer_lambda=use_authorizer_lambda,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerLambda")
    def authorizer_lambda(self) -> aws_cdk.aws_lambda.Function:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_lambda.Function, jsii.get(self, "authorizerLambda"))

    @authorizer_lambda.setter
    def authorizer_lambda(self, value: aws_cdk.aws_lambda.Function) -> None:
        jsii.set(self, "authorizerLambda", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gateway")
    def gateway(self) -> aws_cdk.aws_apigateway.RestApi:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_apigateway.RestApi, jsii.get(self, "gateway"))

    @gateway.setter
    def gateway(self, value: aws_cdk.aws_apigateway.RestApi) -> None:
        jsii.set(self, "gateway", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaFunctions")
    def lambda_functions(self) -> "LambdasByPath":
        '''
        :stability: experimental
        '''
        return typing.cast("LambdasByPath", jsii.get(self, "lambdaFunctions"))

    @lambda_functions.setter
    def lambda_functions(self, value: "LambdasByPath") -> None:
        jsii.set(self, "lambdaFunctions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="models")
    def models(self) -> typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel], jsii.get(self, "models"))

    @models.setter
    def models(
        self,
        value: typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel],
    ) -> None:
        jsii.set(self, "models", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="requestValidators")
    def request_validators(
        self,
    ) -> typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IRequestValidator]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IRequestValidator], jsii.get(self, "requestValidators"))

    @request_validators.setter
    def request_validators(
        self,
        value: typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IRequestValidator],
    ) -> None:
        jsii.set(self, "requestValidators", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaLayer")
    def lambda_layer(self) -> typing.Optional[aws_cdk.aws_lambda.LayerVersion]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[aws_cdk.aws_lambda.LayerVersion], jsii.get(self, "lambdaLayer"))

    @lambda_layer.setter
    def lambda_layer(
        self,
        value: typing.Optional[aws_cdk.aws_lambda.LayerVersion],
    ) -> None:
        jsii.set(self, "lambdaLayer", value)


@jsii.data_type(
    jsii_type="crow-api.CrowApiProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_gateway_configuration": "apiGatewayConfiguration",
        "api_gateway_name": "apiGatewayName",
        "authorizer_directory": "authorizerDirectory",
        "authorizer_lambda_configuration": "authorizerLambdaConfiguration",
        "create_api_key": "createApiKey",
        "lambda_configurations": "lambdaConfigurations",
        "lambda_integration_options": "lambdaIntegrationOptions",
        "log_retention": "logRetention",
        "method_configurations": "methodConfigurations",
        "models": "models",
        "request_validators": "requestValidators",
        "shared_directory": "sharedDirectory",
        "source_directory": "sourceDirectory",
        "token_authorizer_configuration": "tokenAuthorizerConfiguration",
        "use_authorizer_lambda": "useAuthorizerLambda",
    },
)
class CrowApiProps:
    def __init__(
        self,
        *,
        api_gateway_configuration: typing.Any = None,
        api_gateway_name: typing.Optional[builtins.str] = None,
        authorizer_directory: typing.Optional[builtins.str] = None,
        authorizer_lambda_configuration: typing.Any = None,
        create_api_key: typing.Optional[builtins.bool] = None,
        lambda_configurations: typing.Optional["CrowLambdaConfigurations"] = None,
        lambda_integration_options: typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.LambdaIntegrationOptions]] = None,
        log_retention: typing.Optional[aws_cdk.aws_logs.RetentionDays] = None,
        method_configurations: typing.Optional["CrowMethodConfigurations"] = None,
        models: typing.Optional[typing.Sequence["CrowModelOptions"]] = None,
        request_validators: typing.Optional[typing.Sequence["CrowRequestValidatorOptions"]] = None,
        shared_directory: typing.Optional[builtins.str] = None,
        source_directory: typing.Optional[builtins.str] = None,
        token_authorizer_configuration: typing.Any = None,
        use_authorizer_lambda: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param api_gateway_configuration: 
        :param api_gateway_name: 
        :param authorizer_directory: 
        :param authorizer_lambda_configuration: 
        :param create_api_key: 
        :param lambda_configurations: 
        :param lambda_integration_options: 
        :param log_retention: 
        :param method_configurations: 
        :param models: 
        :param request_validators: 
        :param shared_directory: 
        :param source_directory: 
        :param token_authorizer_configuration: 
        :param use_authorizer_lambda: 

        :stability: experimental
        '''
        if isinstance(lambda_configurations, dict):
            lambda_configurations = CrowLambdaConfigurations(**lambda_configurations)
        if isinstance(method_configurations, dict):
            method_configurations = CrowMethodConfigurations(**method_configurations)
        self._values: typing.Dict[str, typing.Any] = {}
        if api_gateway_configuration is not None:
            self._values["api_gateway_configuration"] = api_gateway_configuration
        if api_gateway_name is not None:
            self._values["api_gateway_name"] = api_gateway_name
        if authorizer_directory is not None:
            self._values["authorizer_directory"] = authorizer_directory
        if authorizer_lambda_configuration is not None:
            self._values["authorizer_lambda_configuration"] = authorizer_lambda_configuration
        if create_api_key is not None:
            self._values["create_api_key"] = create_api_key
        if lambda_configurations is not None:
            self._values["lambda_configurations"] = lambda_configurations
        if lambda_integration_options is not None:
            self._values["lambda_integration_options"] = lambda_integration_options
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if method_configurations is not None:
            self._values["method_configurations"] = method_configurations
        if models is not None:
            self._values["models"] = models
        if request_validators is not None:
            self._values["request_validators"] = request_validators
        if shared_directory is not None:
            self._values["shared_directory"] = shared_directory
        if source_directory is not None:
            self._values["source_directory"] = source_directory
        if token_authorizer_configuration is not None:
            self._values["token_authorizer_configuration"] = token_authorizer_configuration
        if use_authorizer_lambda is not None:
            self._values["use_authorizer_lambda"] = use_authorizer_lambda

    @builtins.property
    def api_gateway_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        result = self._values.get("api_gateway_configuration")
        return typing.cast(typing.Any, result)

    @builtins.property
    def api_gateway_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("api_gateway_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("authorizer_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_lambda_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        result = self._values.get("authorizer_lambda_configuration")
        return typing.cast(typing.Any, result)

    @builtins.property
    def create_api_key(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("create_api_key")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def lambda_configurations(self) -> typing.Optional["CrowLambdaConfigurations"]:
        '''
        :stability: experimental
        '''
        result = self._values.get("lambda_configurations")
        return typing.cast(typing.Optional["CrowLambdaConfigurations"], result)

    @builtins.property
    def lambda_integration_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.LambdaIntegrationOptions]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("lambda_integration_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.LambdaIntegrationOptions]], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        '''
        :stability: experimental
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[aws_cdk.aws_logs.RetentionDays], result)

    @builtins.property
    def method_configurations(self) -> typing.Optional["CrowMethodConfigurations"]:
        '''
        :stability: experimental
        '''
        result = self._values.get("method_configurations")
        return typing.cast(typing.Optional["CrowMethodConfigurations"], result)

    @builtins.property
    def models(self) -> typing.Optional[typing.List["CrowModelOptions"]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("models")
        return typing.cast(typing.Optional[typing.List["CrowModelOptions"]], result)

    @builtins.property
    def request_validators(
        self,
    ) -> typing.Optional[typing.List["CrowRequestValidatorOptions"]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_validators")
        return typing.cast(typing.Optional[typing.List["CrowRequestValidatorOptions"]], result)

    @builtins.property
    def shared_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("shared_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("source_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def token_authorizer_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        result = self._values.get("token_authorizer_configuration")
        return typing.cast(typing.Any, result)

    @builtins.property
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("use_authorizer_lambda")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowApiProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowLambdaConfigurations",
    jsii_struct_bases=[],
    name_mapping={},
)
class CrowLambdaConfigurations:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowLambdaConfigurations(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowMethodConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "api_key_required": "apiKeyRequired",
        "authorization_scopes": "authorizationScopes",
        "authorization_type": "authorizationType",
        "authorizer": "authorizer",
        "method_responses": "methodResponses",
        "operation_name": "operationName",
        "request_models": "requestModels",
        "request_parameters": "requestParameters",
        "request_validator": "requestValidator",
        "request_validator_options": "requestValidatorOptions",
        "use_authorizer_lambda": "useAuthorizerLambda",
    },
)
class CrowMethodConfiguration:
    def __init__(
        self,
        *,
        api_key_required: typing.Optional[builtins.bool] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[aws_cdk.aws_apigateway.AuthorizationType] = None,
        authorizer: typing.Optional[aws_cdk.aws_apigateway.IAuthorizer] = None,
        method_responses: typing.Optional[typing.Sequence["CrowMethodResponse"]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.bool]] = None,
        request_validator: typing.Optional[builtins.str] = None,
        request_validator_options: typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions] = None,
        use_authorizer_lambda: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param api_key_required: 
        :param authorization_scopes: 
        :param authorization_type: 
        :param authorizer: 
        :param method_responses: 
        :param operation_name: 
        :param request_models: 
        :param request_parameters: 
        :param request_validator: 
        :param request_validator_options: 
        :param use_authorizer_lambda: 

        :stability: experimental
        '''
        if isinstance(request_validator_options, dict):
            request_validator_options = aws_cdk.aws_apigateway.RequestValidatorOptions(**request_validator_options)
        self._values: typing.Dict[str, typing.Any] = {}
        if api_key_required is not None:
            self._values["api_key_required"] = api_key_required
        if authorization_scopes is not None:
            self._values["authorization_scopes"] = authorization_scopes
        if authorization_type is not None:
            self._values["authorization_type"] = authorization_type
        if authorizer is not None:
            self._values["authorizer"] = authorizer
        if method_responses is not None:
            self._values["method_responses"] = method_responses
        if operation_name is not None:
            self._values["operation_name"] = operation_name
        if request_models is not None:
            self._values["request_models"] = request_models
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_validator is not None:
            self._values["request_validator"] = request_validator
        if request_validator_options is not None:
            self._values["request_validator_options"] = request_validator_options
        if use_authorizer_lambda is not None:
            self._values["use_authorizer_lambda"] = use_authorizer_lambda

    @builtins.property
    def api_key_required(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("api_key_required")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("authorization_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def authorization_type(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.AuthorizationType]:
        '''
        :stability: experimental
        '''
        result = self._values.get("authorization_type")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.AuthorizationType], result)

    @builtins.property
    def authorizer(self) -> typing.Optional[aws_cdk.aws_apigateway.IAuthorizer]:
        '''
        :stability: experimental
        '''
        result = self._values.get("authorizer")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.IAuthorizer], result)

    @builtins.property
    def method_responses(self) -> typing.Optional[typing.List["CrowMethodResponse"]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("method_responses")
        return typing.cast(typing.Optional[typing.List["CrowMethodResponse"]], result)

    @builtins.property
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("operation_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_models(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_models")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.bool]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.bool]], result)

    @builtins.property
    def request_validator(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_validator")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_validator_options(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions]:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_validator_options")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions], result)

    @builtins.property
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("use_authorizer_lambda")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowMethodConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowMethodConfigurations",
    jsii_struct_bases=[],
    name_mapping={},
)
class CrowMethodConfigurations:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowMethodConfigurations(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowMethodResponse",
    jsii_struct_bases=[],
    name_mapping={
        "status_code": "statusCode",
        "response_models": "responseModels",
        "response_parameters": "responseParameters",
    },
)
class CrowMethodResponse:
    def __init__(
        self,
        *,
        status_code: builtins.str,
        response_models: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        response_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.bool]] = None,
    ) -> None:
        '''
        :param status_code: 
        :param response_models: 
        :param response_parameters: 

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "status_code": status_code,
        }
        if response_models is not None:
            self._values["response_models"] = response_models
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters

    @builtins.property
    def status_code(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("status_code")
        assert result is not None, "Required property 'status_code' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def response_models(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("response_models")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.bool]]:
        '''
        :stability: experimental
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.bool]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowMethodResponse(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowModelOptions",
    jsii_struct_bases=[],
    name_mapping={
        "model_name": "modelName",
        "schema": "schema",
        "content_type": "contentType",
        "description": "description",
    },
)
class CrowModelOptions:
    def __init__(
        self,
        *,
        model_name: builtins.str,
        schema: aws_cdk.aws_apigateway.JsonSchema,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param model_name: 
        :param schema: 
        :param content_type: 
        :param description: 

        :stability: experimental
        '''
        if isinstance(schema, dict):
            schema = aws_cdk.aws_apigateway.JsonSchema(**schema)
        self._values: typing.Dict[str, typing.Any] = {
            "model_name": model_name,
            "schema": schema,
        }
        if content_type is not None:
            self._values["content_type"] = content_type
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def model_name(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("model_name")
        assert result is not None, "Required property 'model_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def schema(self) -> aws_cdk.aws_apigateway.JsonSchema:
        '''
        :stability: experimental
        '''
        result = self._values.get("schema")
        assert result is not None, "Required property 'schema' is missing"
        return typing.cast(aws_cdk.aws_apigateway.JsonSchema, result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowModelOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowRequestValidatorOptions",
    jsii_struct_bases=[],
    name_mapping={
        "request_validator_name": "requestValidatorName",
        "validate_request_body": "validateRequestBody",
        "validate_request_parameters": "validateRequestParameters",
    },
)
class CrowRequestValidatorOptions:
    def __init__(
        self,
        *,
        request_validator_name: builtins.str,
        validate_request_body: typing.Optional[builtins.bool] = None,
        validate_request_parameters: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param request_validator_name: 
        :param validate_request_body: 
        :param validate_request_parameters: 

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "request_validator_name": request_validator_name,
        }
        if validate_request_body is not None:
            self._values["validate_request_body"] = validate_request_body
        if validate_request_parameters is not None:
            self._values["validate_request_parameters"] = validate_request_parameters

    @builtins.property
    def request_validator_name(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("request_validator_name")
        assert result is not None, "Required property 'request_validator_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def validate_request_body(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("validate_request_body")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def validate_request_parameters(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        result = self._values.get("validate_request_parameters")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowRequestValidatorOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.LambdasByPath",
    jsii_struct_bases=[],
    name_mapping={},
)
class LambdasByPath:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdasByPath(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "CrowApi",
    "CrowApiProps",
    "CrowLambdaConfigurations",
    "CrowMethodConfiguration",
    "CrowMethodConfigurations",
    "CrowMethodResponse",
    "CrowModelOptions",
    "CrowRequestValidatorOptions",
    "LambdasByPath",
]

publication.publish()
