"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, lambdaIntegrationOptions = {}, models = [], requestValidators = [], methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.RestApi(this, apiGatewayName, {
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        const createdModels = {};
        models.forEach((model) => {
            // modelName is used as ID and can now be used for referencing model in method options
            createdModels[model.modelName] = gateway.addModel(model.modelName, model);
        });
        const createdRequestValidators = {};
        requestValidators.forEach((requestValidator) => {
            // requestValidatorName is used as ID and can now be used for referencing model in method options
            createdRequestValidators[requestValidator.requestValidatorName] = gateway.addRequestValidator(requestValidator.requestValidatorName, requestValidator);
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value and the tweaked model values
                    const { useAuthorizerLambda: authorizerLambdaConfigured = false, requestModels: crowRequestModels, methodResponses: crowMethodResponses, requestValidator: requestValidatorString, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    let bundledMethodConfiguration = {
                        ...userMethodConfiguration,
                    };
                    // Map models
                    const requestModels = {};
                    if (crowRequestModels) {
                        Object.entries(crowRequestModels).forEach(([contentType, modelName]) => {
                            requestModels[contentType] = createdModels[modelName];
                        });
                    }
                    const methodResponses = [];
                    if (crowMethodResponses && crowMethodResponses.length > 0) {
                        crowMethodResponses.forEach((crowMethodResponse) => {
                            const responseModels = {};
                            if (crowMethodResponse.responseModels) {
                                const crowResponseModels = crowMethodResponse.responseModels;
                                Object.entries(crowResponseModels).forEach(([contentType, modelName]) => {
                                    responseModels[contentType] = createdModels[modelName];
                                });
                            }
                            const { statusCode, responseParameters, } = crowMethodResponse;
                            methodResponses.push({
                                statusCode,
                                responseParameters,
                                responseModels,
                            });
                        });
                    }
                    // Find request validator
                    if (requestValidatorString
                        && createdRequestValidators[requestValidatorString]) {
                        bundledMethodConfiguration.requestValidator = createdRequestValidators[requestValidatorString];
                    }
                    bundledMethodConfiguration.requestModels = requestModels;
                    bundledMethodConfiguration.methodResponses = methodResponses;
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        bundledMethodConfiguration.authorizationType = apigateway.AuthorizationType.CUSTOM;
                        bundledMethodConfiguration.authorizer = tokenAuthorizer;
                    }
                    const integrationOptions = lambdaIntegrationOptions[newApiPath] || {};
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda, integrationOptions), bundledMethodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
        this.models = createdModels;
        this.requestValidators = createdRequestValidators;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.3.1" };
//# sourceMappingURL=data:application/json;base64,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