"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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