# lightweb4py - легкий фреймворк на Python

# Общая информация

Данный фреймворк реализован в рамках курса по паттернам проектирования 
в объектно-ориентированных языках программирования для демонстрации применения таких паттернов 
при реализации фреймворка, аналогичного Flask и прочим, 
предоставляющим базовый функционал для быстрой разработки web-приложений.

В рамках разработки фреймворка не ставилось задачи 
запрограммировать исчерпывающую реализацию какой-либо функциональности. 
Напротив, в условиях ограниченного времени акцент ставился на реализации минимального жизнеспособного продукта (MVP),
с одной стороны, демонстрирующего требуемый по ТЗ функционал, 
а с другой - показывающего эффективное применение паттернов проектирования для написания удобного в использовании, 
расширяемого по функциональности кода фреймворка, удовлетворяющего принципам DRY и прочим принятым в отрасли критериям 
"хорошего" кода.

Фреймворк реализует следующую функциональность (базовая реализация для демонстрации паттернов проектирования):
- Routing (маршрутизация) - распаковка и маршрутизация HTTP-запросов (requests) по дереву ссылок (URL) приложения, 
упаковка HTTP-ответов (response) (framework.py, request.py, response.py); маршрутизация возможна как по полному URL, 
так и по wildcard (например, /someurl/*).
- CBV (Class-Based Views, Классы-обработчики запросов) - иерархия классов для обработки HTTP-запросов 
и визуализации данных пользователя в web-приложениях с использованием шаблонизатора Jinja2 
и механизма наследования шаблонов (views.py).
- ORM (Object-Relational Mapping, объектно-реляционное отображение данных) - библиотека для сохранения и загрузки 
объектов Python в хранилища данных; реализованы интерфейсы JSON и SQLite (persistence.py, persistence_sqlite.py).
- Logger (логирование) - библиотека, реализующая функциональность создания произвольного количества именованных логгеров 
и логирования в различные системы хранения и отображения логов; реализовано отображение на консоль и сохранение 
в текстовый файл CSV. 

Для демонстрации функций фреймворка было реализовано демонстрационное приложение (папка demo/), 
демонстрирующее минимизацию фреймворком усилий программиста для разработки простейших web-приложений.
Оно реализует минимальную функциональность административного интерфейса учебного портала:
создание/редактирование/удаление категорий учебных курсов, 
самих курсов с присвоением им категорий и студентов с регистрацией их на курс. 
Также доступна функциональность посылки пользователем сообщений обратной связи и просмотра таких сообщений, в том числе
с помощью микросервисного приложения, предназначенного для демонстрации концепции разделения монолитного приложения 
на микросервисы.
Документация по демонстрационному приложению доступна в папке demo/docs.

# Краткий обзор состава фреймворка

Ниже дан краткий обзор состава фреймворка - папок, основных файлов и классов.

Состав фреймворка:
- demo/ - папка с демонстрационным приложением - административным интерфейсом образовательного портала;
- demo/_app.py_ - основной файл демонстрационного приложения;
- demo/_app_student.py_ - основной файл демонстрационного микросервисного приложения;
- demo/docs/ - папка с документацией демонстрационного приложения; 
- docs/ - папка с документацией фреймворка;
- docs/_README.md_ - этот файл - описание фреймворка;
- docs/examples/ - папка с примерами к документации фреймворка;
- docs/examples/templates/ - папка с HTML-шаблонами для примеров к документации фреймворка;
- templates/ - папка с HTML-шаблонами фреймворка, в т.ч. страницы 404 и административного интерфейса
- templates/baseview/ - папка с HTML-шаблонами класса BaseView (см. детальное описание фреймворка)
- _framework.py_ - основной файл фреймворка, в котором реализованы:
  - класс Framework, чей Singleton-объект передается WSGI-серверу (gunicorn) как точка входа в приложение;
  данный класс отвечает за первичную обработку запроса (front controller), его маршрутизацию в соответствующий 
  класс-обработчик и возвращение результата WSGI-серверу;
  - декоратор @url для добавления класса-обработчика в приложение.
- _logger.py_ - библиотека Logger - класс Logger и прочие, реализующие функциональность логгера, и декоратор @debug 
для логирования начала и конца выполнения методов (инициализации классов).
- _persistence.py_ - библиотека Persistence - класс Persistence и прочие, реализующие функциональность ORM 
(объектно-реляционного отображения данных) с сохранением данных в файл JSON и возможностью расширения списка типов 
хранилищ данных.
- _persistence_sqlite.py_ - класс PersistenceSQLite, реализующий расширение библиотеки Persistence в части хранения 
данных в БД SQLite (sqlite3).
- _request.py_ - класс Request, чей объект содержит параметры переданного WSGI-сервером окружения.
- _response.py_ - класс Response, чей объект содержит результаты работы класса-обработчика, возвращаемый WSGI-серверу.
- _settings.py_ - константы общего пользования, а также класс Environ, инициализирующий окружение шаблонизатора Jinja2.
- _singleton.py_ - абстрактный класс Singleton, реализующий соответствующую функциональность. 
- _urls.py_ - класс Url для хранения информации маршрутизации и объект urls с URL административного интерфейса.
- _views.py_ - класс View, базовый для всех классов-обработчиков, а также ряд полезных классов-обработчиков:
    - Minimal - для отображения информации без использования файлов-шаблонов, отображающий данные пользователя
    с минимальными заголовками HTML;
    - Render - для отображения параметризованных шаблонов из файлов с использованием шаблонизатора Jinja2;
    - BaseView - для отображения иерархического шаблона с настраиваемыми боковыми панелями, заголовком и футером;
    - View404 - для отображения страницы кода 404 по умолчанию;
    - Admin - административный интерфейс для отображающее параметров окружения WSGI и запросов HTTP и тестирования
    запросов GET и POST с параметрами.

# Описание фреймворка

В данном разделе дается детальное описание фреймворка.

## Обработка HTTP-запроса, маршрутизация, классы-обработчики запросов, формирование HTTP-ответа.

Фреймворк предоставляет пользователю средства для написания web-приложения, обрабатывающего запросы WSGI-сервера 
и формирующего ответы на эти запросы.

Для обработки запроса приложение пользователя создает Singleton-объект класса Framework (framework.py) 
и передает в его инициализатор список объектов класса Url (urls.py), содержащих обрабатываемые приложением URL 
и соответствующие им параметризуемые классы-обработчики запросов (CBV, Class-Based Views) - 
подклассы класса View (views.py). 
При поступлении HTTP-запроса объект класса Framework, являющийся для WSGI-сервера точкой входа в приложение 
пользователя, передает в соответствующий класс-обработчик объект класса Request (request.py) с параметрами WSGI-запроса, 
а класс-обработчик в свою очередь должен вернуть объект класса Response (response.py), 
содержащий страницу HTML для отображения и/или другую необходимую информацию.
Маршрутизация запросов в классы-обработчики возможна как по полному URL, например, "/contact", 
так и по URL - регулярному выражению (regular expression, wildcard), например, "/edit/*", что дает возможность
приложению пользователя реализовывать собственные алгоритмы маршрутизации запросов на отдельных ветках приложения.
Объект класса-обработчика запроса (CBV) инициализируется перед вызовом его методов для обработки 
закрепленного за ним URL, в его инициализатор можно передать параметры, указав их в соответствующем поле класса Url 
и переопределив метод __init__() класса-обработчика. Для обработки запроса вызывается метод run() класса-обработчика, 
который должен быть реализован пользователем.
Возможна также привязка URL к классу-обработчику с помощью параметризуемого декоратора @url (framework.py).

В случае использования приложением пользователя встроенных классов-обработчиков запросов (CBV) (views.py),
в том числе, административного интерфейса фреймворка - класса AdminView (views.py), 
необходимо также создать singleton-объект класса Environ (settings.py). Встроенные CBV фреймворка используют 
библиотеку шаблонизатора Jinja2 для загрузки (load) и рендеринга (render) шаблонов. В случае если приложение 
пользователя также использует данный шаблонизатор, в инициализатор объекта  класса Environ можно передать список
папок шаблонов приложения пользователя для их корректной загрузки - в этом случае шаблоны пользователя доступны для 
загрузки начиная с префикса пути "app", например, "app/index.html", если файл "index.html" расположен в папке,
переданный объекту класса Environ при инициализации.

Ниже приведен минимальный работоспособный фрагмент кода приложения с 2 страницами - главной и контактами, иллюстрирующий
описанную выше функциональность. Оно использует CBV SimpleView для рендеринга Jinja-2 шаблона. 
Приложение вызывается командой (программный файл можно найти в папке приложения):
```
gunicorn app_test:app
```
Текст приложения:
```
from framework.framework import Framework, url
from framework.urls import Url
from framework.views import View, SimpleView
from framework.request import Request
from framework.response import Response
from framework.settings import Environ


class MainView(View):

    def __init__(self, *args, template_name: str, **kwargs):
        self.template_name = template_name

    def run(self, request: Request, *args, **kwargs) -> Response:
        return SimpleView(template_name=self.template_name).run(request)


@url('/contact', name="Обратная связь", in_menu=True)
class ContactView(View):
    TEMPLATE_NAME = "app/contact.html"

    def run(self, request: Request, *args, **kwargs) -> Response:
        return SimpleView(template_name=self.TEMPLATE_NAME).run(request)


app_urls = [
    Url('/', MainView, {'template_name': 'app/index.html'}, "Главная страница", True),
]

app = Framework(app_urls, use_admin=True)           # Создать объект приложения
env = Environ('app/app_test')                       # Создать объект окружения
```
В рабочем приложении рекомендуется выделять в отдельные файлы:
- в файл urls.py - URL приложения,
- в файл views.py - классы-обработчики запросов (CBV).

В последующих разделах приводится детальное описание используемых приложением библиотек.

### Класс Request (request.py)

Объект класса Request создается методом __call__() singleton-объекта класса Framework каждый раз при вызове данного 
метода сервером WSGI при получении HTTP-запроса. Для доступа к нему необходимо использовать атрибут 
Framework().request , например:
```
app = Framework(app_urls, use_admin=True)           # Создать объект приложения
...
# Где-то в коде класса или функции, где недоступна переменная app, при обработке запроса:
if Framework().request.method == 'GET':             # Какой вызван HTTP-метод?
```
Объект данного класса предоставляет приложению пользователя удобный доступ к параметрам HTTP-запроса и переменным 
окружения WSGI. Инициализатору данного класса передается словарь environ, передаваемый пользовательскому приложению
сервером WSGI при получении последним HTTP-запроса, например:
```
# Фрагмент кода метода класса Framework
from framework.request import Request
...
def __call__(self, environ: dict, start_response, *args, **kwargs):
    self.request = Request(environ)     # Создать объект класса Request и инициализировать его словарем окружения WSGI
```
Приложению пользователя доступны следующие свойства объекта данного класса:

| Свойство     | Ключ словаря environ                       | Описание                                                                                                                                               |
|--------------|--------------------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------|
| path_array   | 'PATH_INFO'                                | список токенов в адресной строке URL запроса между косыми чертами ('/'); всегда содержит хотя бы один токен (пустой, если запрошен корневой URL сайта) |
| path         | 'PATH_INFO'                                | адресной строке URL запроса без финальной косой черты ('/'); косая черта ('/') - для корневого URL сайта                                               |
| method       | 'REQUEST_METHOD'                           | метод HTTP-запроса ('GET', 'POST', ...)                                                                                                                |
| query_params | 'QUERY_STRING' ('GET'),<br/> body ('POST') | словарь параметров запроса (например, '?a=1&b=2') для методов 'GET' и 'POST'                                                                           |
| headers      | 'HTTP_*'                                   | словарь всех параметров HTTP                                                                                                                           |

### Класс Response (response.py)

Объект класса Response используется во фреймворке для упаковки информации, которую требуется отправить серверу WSGI 
по окончании обработки запроса. Каждый класс-обработчик должен возвращать такой объект как результат работы 
метода run() , например:
```
def run(self, request: Request, *args, **kwargs) -> Response:
    return Response(status_code="200 OK", body="Это тестовая HTML-страница")
```
Инициализатору данного класса могут быть передан следующие параметры:

| Параметр    | Тип         | Значение по умолчанию                        | Описание                                                                       |
|-------------|-------------|----------------------------------------------|--------------------------------------------------------------------------------|
| status_code | str или int | 404                                          | код HTTP ответа - целочисленный или строка, например, '307 Temporary redirect' |
| body        | str         | '' (пустая строка)                           | тело ответа, например, HTML-страница для отображения                           |
| headers     | dict        | {'Content-Type': 'text/html; charset=UTF-8'} | словарь заголовков                                                             |

### Класс Url (urls.py)

Класс Url предназначен для описания дерева URL приложения пользователя и присвоения обработчиков его элементам. 
Для начала работы с фреймворком приложение пользователя создает singleton-объект класса Framework (framework.py) 
и передает в его инициализатор список объектов класса Url, содержащих обрабатываемые приложением URL 
и соответствующие им параметризуемые классы-обработчики запросов (CBV, Class-Based Views) - 
подклассы класса View (views.py).
Пример списка объектов Url приведен ниже:
```
app_urls = [
    Url('/', Index, {}, "Главная страница", True),
    Url('/course/edit/*', CourseEditView,
        {'element_class': Course,
         'element_html_class_name': "Course",
         'element_title': "Учебный курс",
         'html_template_form': settings.TEMPLATES_APP + "/element_edit.html",
         'editor_field_template_names': (settings.TEMPLATES_APP + "/element_edit_name_field.html",
                                         settings.TEMPLATES_APP + "/element_edit_category_field.html", ),
         },
        "Курсы", False),
]
```
В данном примере приложение имеет 2 URL:
- Главная страница, запрос к которой обрабатывается классом Index, инициализатор которого не имеет параметров; ссылку
на данный URL планируется отображать в меню навигации с подписью "Главная страница" 
(см. описание свойств name и inMenu ниже), и
- Курсы, обрабатываемый классом CourseEditView, которому передаются на обработку все URL, начинающиеся с /course/edit/ ,
инициализатору которого передаются параметры element_class, element_title и прочие из словаря; ссылку
на данный URL не планируется отображать в меню навигации.

Подробное описание свойств класса Url (параметров его инициализатора) приведено ниже.

| Свойство   | Тип    | Значение по<br/>умолчанию | Описание                                                                                                                                                                                                                                                                                                                                                             |
|------------|--------|---------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| url        | str    | - (обязательный параметр) | URL, за которым закрепляется класс-обработчик. Может быть полным URL, например, "/contact", или регулярным выражением (regular expression, wildcard), например, "/edit/*", в случае чего обработчику передаются все URL, начинающиеся с "/edit/", например, "/edit", "/edit/item". Wildcard может заменять только токен целиком, то есть, URL "/edit*" не сработает. |
| viewClass  | object | - (обязательный параметр) | идентификатор класса-обработчика данного URL - подкласс класса View, используемый для создания экземпляра данного класса при поступлении в маршрутизатор класса Framework запроса на отображение закрепленного за данным классом URL.                                                                                                                                |
| viewParams | dict   | - (обязательный параметр) | словарь параметров, передаваемый инициализатору класса-обработчика при создании его объекта; метод __init__() класса-обработчика должен поддерживать прием указанных здесь параметров.                                                                                                                                                                               |
| name       | str    | '' (пустая строка)        | название (текст ссылки) для данного URL в меню навигации приложения; поле введено в класс для удобства с прицелом на реализацию в будущем стандартной функциональности карты сайта                                                                                                                                                                                   |
| inMenu     | bool   | False                     | флаг, показывающий, нужно ли включать данный URL в список URL, возвращаемый вызовом метода Framework().get_clean_urls(for_menu_only=True), который можно использовать для построения меню навигации приложения (сайта)                                                                                                                                               |

#### Декоратор url (framework.py)

Другим способом добавить классы-обработчики к дереву маршрутизации можно при помощи декоратора @url. Декоратор добавляет 
декорируемый класс как обработчик для указанного в его параметре URL при помощи функции Framework().set_url() 
с параметром replace=True (см. описание класса Framework). Пример сценария использования декоратора приведен ниже:
```
from framework.framework import url
...
@url('/contact', name="Обратная связь", in_menu=True)
class ContactView(View):
    TEMPLATE_NAME = "app/contact.html"

    def run(self, request: Request, *args, **kwargs) -> Response:
        return SimpleView(template_name=self.TEMPLATE_NAME).run(request)
```
Ниже дано описание полей декоратора (их семантика идентична соответствующим полям класса Url):

| Свойство | Тип  | Значение по<br/>умолчанию | Описание                                                                                                                                                                                                                                                                                                                                                             |
|----------|------|---------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| url      | str  | - (обязательный параметр) | URL, за которым закрепляется класс-обработчик. Может быть полным URL, например, "/contact", или регулярным выражением (regular expression, wildcard), например, "/edit/*", в случае чего обработчику передаются все URL, начинающиеся с "/edit/", например, "/edit", "/edit/item". Wildcard может заменять только токен целиком, то есть, URL "/edit*" не сработает. |
| name     | str  | '' (пустая строка)        | название (текст ссылки) для данного URL в меню навигации приложения; поле введено в класс для удобства с прицелом на реализацию в будущем стандартной функциональности карты сайта                                                                                                                                                                                   |
| inMenu   | bool | False                     | флаг, показывающий, нужно ли включать данный URL в список URL, возвращаемый вызовом метода Framework().get_clean_urls(for_menu_only=True), который можно использовать для построения меню навигации приложения (сайта)                                                                                                                                               |
| **kwargs | dict | - (обязательный параметр) | любые именованные параметры, следующие за вышеуказанными, воспринимаются как словарь viewParams объекта Url: словарь параметров, передаваемый инициализатору класса-обработчика при создании его объекта; метод __init__() класса-обработчика должен поддерживать прием указанных здесь параметров.                                                                  |

### Класс View (views.py)

Класс View - это интерфейс (абстрактный класс), который должны наследовать все классы-обработчики URL в приложении 
пользователя.
```
class View(ABC):

    @abstractmethod
    def run(self, request: Request, *args, **kwargs) -> Response:
        pass
```
Классы-наследники View должны реализовать метод run(), который вызывается объектом класса Framework для обработки 
URL, привязанного к данному классу-обработчику. В качестве параметра методу передается объект класса Request
с параметрами HTTP-запроса и WSGI-окружения. Метод должен возвращать объект класса Response с требуемым содержимым.
В случае если объект Url задает для класса-обработчика параметры в свойстве viewParams, обработчик также должен 
реализовать метод __init__(), принимающий все эти параметры.

Фреймворк содержит также классы-обработчики, реализующие расширенную функциональность и облегчающие разработку 
приложений. Эти классы описаны ниже в разделе, посвященном Class-Based Views (CBV, классы-обработчики запросов).

### Класс Singleton (singleton.py)

Класс Singleton - абстрактный класс, унаследовав который, класс-наследник будет гарантированно иметь не более одного
экземпляра. Для выполнения этого условия класс реализует метод __new__(), в связи с чем любой класс-наследник, 
переопределяющий метод __new__(), должен вызывать метод __new__() класса Singleton. Какая-либо другая функциональность 
в данном классе отсутствует.

Фреймфорк использует класс Singleton для реализации ряда своих ключевых классов (объектов), в числе которых:

| Класс          | Описание                                                                                                                                                                                        |
|----------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| Framework      | Хранит конфигурацию и осуществляет первичную распаковку и маршрутизацию HTTP-запросов в приложении                                                                                              |
| Environ        | Хранит конфигурацию Jinja2 - ссылку на загрузчик и прочие параметры                                                                                                                             |
| LoggerFabric   | Хранит конфигурацию логгеров приложения                                                                                                                                                         |
| BaseViewLayout | Хранит конфигурацию и предоставляет интерфейс управления для класса BaseView, предоставляющего пользователю иерархический Jinja2-шаблон HTML-страницы с боковыми панелями, заголовком и футером |

### Класс Environ (settings.py)

Необходимо создать объект класса Environ (окружения) в приложении пользователя, в случае если выполняется 
любое из двух условий:
- приложение пользователя использует любые дополнительные классы из библиотеки views.py, кроме базового класса View, 
в том числе, административный интерфейс фреймворка - класс AdminView, или
- объект класса Framework был инициализирован с параметром use_admin=True, 
что включает в дерево URL приложения административный интерфейс.

Создать объекта класса Environ необходимо до использования дополнительных классов из библиотеки views.py, 
кроме базового класса View, но можно создать его после объекта класса Framework:
```
env = Environ(template_path='app/templates')            # Создать объект окружения
```
Инициализатор класса принимает единственный необязательный параметр - путь к шаблонам приложения. В случае если параметр
задан, все классы библиотеки views.py будут искать шаблоны, заданные пользователем, в указанном каталоге 
и его подкаталогах. Если параметр опущен, корневым каталогом для поиска шаблона пользователя будет считаться текущий
каталог на момент вызова инициализатора, зависящий от системы, в которой разрабатывается и запускается приложение.

После инициализации, например, в параметрах классов библиотеки views.py путь к конкретным файлам шаблонов пользователя 
необходимо указывать с префиксом 'app/', например, в случае если при инициализации Environ был задан вышеуказанный путь:

| Путь к шаблону в параметре класса-обработчика | Класс-обработчик загрузит шаблон, используя путь |
|-----------------------------------------------|--------------------------------------------------|
| app/index.html                                | app/templates/index.html                         |
| app/someview/baseview.html                    | app/templates/someview/baseview.html             |

Таким образом, не изменяя относительных ссылок, можно изменить местонахождение всего дерева шаблонов приложения 
пользователей, изменив параметр инициализации объекта класса Environ.

#### Замечания по использованию шаблонизатора Jinja2
Также желательно создать объект класса Environ, если приложение пользователя использует шаблонизатор Jinja2. 
В этом случае приложению в свойстве Environ().jinja_env доступен объект класса Jinja2.Environment (см. объяснение ниже),
что позволяет, например, использовать загрузчик (Loader) Jinja2 для загрузки шаблонов:
```
source_html, file_name, up_to_date = Environ().jinja_env.loader.get_source(Environ().jinja_env, 'app/sometemplate.html')
```
При инициализации объекта класса Environ для окружения (Environment) Jinja2 задается загрузчик PrefixLoader с отдельными
FileSystemLoader для шаблонов фреймворка (с префиксом пути 'fw/') и приложения пользователя ('app/') 
с кодированием шаблонов UTF-8.

Информацию по использованию объектов класса Environment смотрите в документации Jinja2. 

### Класс Framework (framework.py)

Singleton-объект класса Framework должен быть создан в приложении пользователя, если в нем планируется использовать 
средства маршрутизации фреймворка:
```
app = Framework(app_urls, use_admin=True)             # Создать объект приложения
```
Инициализатор объекта имеет следующие параметры:

| Параметр  | Тип   | Значение по<br/>умолчанию | Описание                                                                                                                                                                                            |
|-----------|-------|---------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| urls      | [Url] | None                      | список объектов класса Url, закрепляющий обработчики за деревом URL приложения (см. описание класса Url)                                                                                            |
| use_admin | bool  | False                     | флаг, указывающий, необходимо ли включать в дерево URL приложения административный интерфейс фреймворка; если use_admin==True, необходимо также создать объект класса Environ (смю описание класса) |

Инициализатор создает свойство объекта urls - список, в который добавляет переданные в параметре urls объекты Url 
в порядке их следования в списке (см. описание класса Url). В процессе создания Затем, если параметр use_admin==True, 
в список свойства urls добавляются объекты  Url административного интерфейса фреймворка, 
если их URL уже не заняты приложением пользователя:

| URL       | Класс-обработчик   | Описание                                                                                                         |
|-----------|--------------------|------------------------------------------------------------------------------------------------------------------|
| /admin/*  | Admin (views.py)   | Административный интерфейс фреймворка - анализ параметров HTTP и WSGI, тест запросов GET и POST (см. описание)   |
| /notfound | View404 (views.py) | Страница, вызываемая при отсутствии требуемой страницы приложения (не требуется для функционирования приложения) |

После создания объекта класса Framework, приложение пользователя можно запустить как приложение WSGI-сервера; 
если имя файла Python, где создан объект класса Framework - application.py, а имя переменной, содержащей объект класса
Framework - app, команда запуска приложения будет выглядеть следующим образом (квадратные скобки означают, что параметр 
не является обязательным):
```
gunicorn [-b 127.0.0.1:8000] application:app 
```
При поступлении HTTP-запроса сервер WSGI вызывает метод __call__() объекта класса Framework, 
в котором выполняются следующие действия:
- свойству объекта Framework().request присваивается объект класса Request (см. описание класса),
инициализированный параметром environ - словарем параметров запроса, переданным WSGI-сервером методу __call__();
- производится маршрутизация HTTP-запроса: анализируются объекты Url списка в свойстве объекта Framework().urls
в порядке их следования (см. описание класса Url); маршрутизация производится как по полному URL, например, "/contact", 
так и по URL - регулярному выражению (regular expression, wildcard), например, "/edit/*", что дает возможность
приложению пользователя реализовывать собственные алгоритмы маршрутизации запросов на отдельных ветках приложения;
- если целевой URL (или wildcard-URL) найден, создается экземпляр закрепленного за ним класса-обработчика, 
в его инициализатор передаются параметры, сохраненные в объекте класса Url, 
а затем вызывается метод класса-обработчика run() , которому передается объект класса Request данного HTTP-запроса;
от класса-обработчика ожидается результат в виде объекта класса Response; 
- если целевой URL не найден, или если класс-обработчик не вернул объект Response, 
производится вызов класса-обработчика View404().run(), и дальнейшая работа производится с возвращаемым им 
объектом класса Response;
- вызывается функция start_response, переданная методу __call__() сервером WSGI, и в нее передаются свойства
status_code - код ответа HTTP - и headers - заголовки HTTP объекта класса Response, возвращенного классом-обработчиком 
URL (см. описание класса Response);
- серверу WSGI возвращается тело ответа - свойство body объекта класса Response, возвращенного классом-обработчиком 
URL, предварительно преобразованное в массив байтов в кодировке UTF-8.

Приложению пользователя доступны следующие свойства объекта класса Framework:

| Свойство | Тип     | Доступно после...                           | Описание                                                                                  |
|----------|---------|---------------------------------------------|-------------------------------------------------------------------------------------------|
| urls     | [Url]   | ...инициализации объекта класса Framework   | список объектов класса Url для маршрутизации запросов HTTP (см. описание класса свойства) |
| request  | Request | ...поступления HTTP-запроса от сервера WSGI | распакованные параметры запроса HTTP и окружения WSGI (см. описание класса свойства)      |

Приложению пользователя доступны следующие методы объекта класса Framework:

| Метод                                             | Описание                                                                                                                                                                                                                                                                                    |
|---------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| set_url(self, url: Url, replace: bool = False)    | Устанавливает обработчик, определенный параметром url, заменяя существующий обработчик для данного URL, если он есть, в случае если replace==True; дает приложению возможность менять таблицу маршрутизации "на ходу"                                                                       |
| get_clean_urls(self, for_menu_only: bool = False) | Возвращает "чистый" список URL приложения: убирает символы wildcard и косую черту перед ними ('/*') из всех wildcard-URL; возвращает только объекты Url, где свойство inMenu==True, если параметр for_menu_only==True; можно использовать как источник данных для меню навигации приложения |

## Встроенные Классы-обработчики запросов (Class-Based Views, CBV) (views.py)

Во фреймворке реализован ряд классов-обработчиков общего назначения, помогающих пользователю быстро получить 
работоспособный код. Все они размещены в библиотеке views.py и наследуют класс View, то есть, реализуют интерфейс View,
их метод run() возвращает объект класса Response, что означает, что эти классы можно непосредственно использовать для 
обработки запросов к URL приложения.

**!ВАЖНО!** Так как все встроенные классы-обработчики, кроме MinimalView, используют шаблонизатор Jinja2 для обработки
шаблонов, перед их использованием необходимо создать объект класса Environ (см. описание класса) и задать путь к 
шаблонам приложения, а при именовании файлов шаблонов для загрузки следовать рекомендациям в описании класса Environ.

### MinimalView - CBV без использования файлов и шаблонизатора

Данный класс-обработчик предназначен для того, чтобы вернуть в ответ на запрос пользователя 
хоть какую-то информацию в условиях сбоя - недоступности дисков, файлов, баз данных или сетевых ресурсов, 
сбоя шаблонизатора и прочих ситуаций, когда доступ к шаблонам страниц невозможен. Данный CBV возвращает переданное 
пользователем содержимое тела web-страницы в минимальную необходимую разметку HTML для ее корректного вывода в браузере:
```
<!DOCTYPE html><html lang="ru">
<head><meta charset="UTF-8">
</head>
<body>
  ... (сюда помещаются переданные пользователем данные)
</body></html>
```
Этот класс удобно использовать для прототипирования приложение в самом начале его создания, когда еще не принято решение
о структуре каталогов и программных файлов приложения. Например, данный фрагмент кода определяет дерево приложения с 
главной страницей и 4 страницами по темам без какого-либо программирования и без необходимости во внешних файлах 
шаблонов и представляет собой законченное приложение с использованием фреймворка:
```
from framework.urls import Url
from framework.framework import Framework
from framework.views import MinimalView

# !ВАЖНО! Рекомендуется список app_urls выносить в файл приложения urls.py и импортировать его в основное приложение
app_urls = [
    Url('/', MinimalView,
        {'body': 'Это главная страница приложения.<br>' + \
                 '<a href="/about">О компании</a><br>' + \
                 '<a href="/sales">Продажи</a><br>' + \
                 '<a href="/service">Обслуживание</a><br>' + \
                 '<a href="/contact">Контакты</a>',
         'status_code': '200 OK'
         }, "Главная страница", True),
    Url('/about', MinimalView,
        {'body': 'Это страница информации о компании.<br>' + \
                 '<a href="/">Домой</a>',
         'status_code': '200 OK'
         }, "О компании", True),
    Url('/sales', MinimalView,
        {'body': 'Это страница подразделения продаж.<br>' + \
                 '<a href="/">Домой</a>',
         'status_code': '200 OK'
         }, "Продажи", True),
    Url('/service', MinimalView,
        {'body': 'Это страница подразделения обслуживания.<br>' + \
                 '<a href="/">Домой</a>',
         'status_code': '200 OK'
         }, "Обслуживание", True),
    Url('/contact', MinimalView,
        {'body': 'Это страница Контактов.<br>' + \
                 '<a href="/">Домой</a>',
         'status_code': '200 OK'
         }, "Контакты", True),
]

app = Framework(app_urls, use_admin=False)  # Create application object
```
Данное приложение вызывается командой (программный файл app_test_minimal.py можно найти в папке приложения):
```
gunicorn app_test_minimal:app
```
Другой сценарий использования данного класса приведен ниже. 
Здесь подразумевается, что метод run() какого-то CBV возвращает сообщение об ошибке, 
вызывая MinimalView, в случае если, например, шаблон данного CBV не найден.
```
from framework.views import View, MinimalView
from framework.request import Request
from framework.response import Response
...
class SomeView(View):
...
  def run(self, request: Request, *args, **kwargs) -> Response:
    ...
    catch FileNotFoundError:
        return MinimalView(f"Template file {file_name} not found.", status_code="404 Not found").run(request)
```
Из фрагмента видно, что в приложении пользователя необязательно наследовать встроенный класс-обработчик, а можно
наследовать интерфейс View, а затем использовать различные классы-обработчики для выполнения различных задач при 
обработке запроса. 

Инициализатор класса принимает следующие параметры:

| Параметр    | Тип | Значение по умолчанию     | Описание                                            |
|-------------|-----|---------------------------|-----------------------------------------------------|
| body        | str | - (обязательный параметр) | текст или HTML-страница для отображения             |
| status_code | int | 200                       | код HTTP ответа, например, 307 (Temporary redirect) |

### SimpleView - простейший CBV с использованием файлов шаблонов и шаблонизатора Jinja2

Данный класс-обработчик загружает шаблон HTML-страницы с заданным пользователем именем файла и обрабатывает этот шаблон 
средствами шаблонизатора Jinja2, которому передаются параметры, указанные пользователем в инициализаторе класса.
Обработанный таким образом шаблон помещается в тело HTTP-ответа. Пользователь также может указать код HTTP-ответа,
возвращаемый в случае успеха, если его не устраивает код по умолчанию. В случае если шаблон не найден, CBV вызывает
встроенный CBV View404 (см. описание ниже) или, по желанию пользователя, возвращает исключение (exception) 
FileNotFoundError, чтобы приложение пользователя могло обработать это событие в соответствии со своей логикой.

Например, следующий шаблон HTML/Jinja2 с именем 'template404.html':
```
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>{{ title }}</title>
</head>
<body>
    <h1>{{ title }}</h1>
    <p> {{ title }} : {{ parameter }}. </p>
</body>
</html>
```
с параметрами title и parameter обрабатывается с помощью фрагмента кода:
```
from framework.views import View, SimpleView
from framework.request import Request
from framework.response import Response
...
class SomeView(View):
  def run(self, request: Request, *args, **kwargs) -> Response:
    ...
    return SimpleView(status_code=404, 
                      template_name='fw/template404.html', 
                      raise_filenotfound=True,
                      title="Страница не найдена",
                      parameter='app/somefile.html').run(request)
```
Аргументы title и parameter, передаваемые инициализатору класса SimpleView (фрагмент кода класса SimpleView 
библиотеки views.py), описанному следующим образом:
```
def __init__(self, template_name: str, status_code: int=200, raise_filenotfound: bool = False, **kwargs):
```
не являются аргументами инициализатора и поэтому передаются шаблонизатору Jinja2 в словаре 
дополнительных именованных параметров **kwargs для обработки шаблона 
(фрагмент кода класса SimpleView библиотеки views.py):
```
self.body = Environ().jinja_env.get_template(template_name).render(**kwargs)
```
Шаблон содержит поля с этими именами, вместо которых шаблонизатор подставит переданные пользователем значения. 
В результате после обработки поле response.body будет содержать следующую строку:
```
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Страница не найдена</title>
</head>
<body>
    <h1>Страница не найдена</h1>
    <p> Страница не найдена : somefile.html. </p>
</body>
</html>
```
Передать данному CBV параметры можно также, задав его как обработчик URL в объекте класса Url (см. описание класса). 
Например, ниже в объекте класса Url передаются те же параметры, что и при непосредственном вызове инициализатора класса 
выше (только для демонстрации возможности, так как в данном случае это не имеет смысла - мы не можем заранее знать, 
какой файл не будет найден):
```
app_urls = [
    Url('/filenotfound', SimpleView, 
        {'status_code': 404, 
         'template_name': 'template404.html', 
         'raise_filenotfound: True,
         'title': "Страница не найдена",
         'parameter': 'somefile.html'
        }, "Главная страница", True),
]
```
Дополнительные примеры использования класса даны в разделе "Обработка HTTP-запроса, маршрутизация, 
классы-обработчики запросов, формирование HTTP-ответа." выше и в файле примера app_test_cbv.py.

Инициализатор класса принимает следующие параметры:

| Параметр           | Тип  | Значение по умолчанию     | Описание                                                                                                             |
|--------------------|------|---------------------------|----------------------------------------------------------------------------------------------------------------------|
| template_name      | str  | - (обязательный параметр) | имя файла шаблона HTML-страницы                                                                                      |
| status_code        | int  | 200                       | код HTTP ответа, например, 307 (Temporary redirect)                                                                  |
| raise_filenotfound | bool | False                     | флаг, указывающий, надо ли возвращать исключение (exception) FileNotFoundError в случае если файл шаблона не найден. |

### View404 - CBV на базе SimpleView/MinimalView для отображения информации о не найденном URL

Данный класс-обработчик может быть использован для отображения для пользователя приложения информации об ошибке 
в случае, например, отсутствия файла шаблона или другого события, когда уместно вернуть код HTTP 404. 
Его инициализатор не принимает параметров. Возможны два сценария использования данного CBV:
1. Вызвать его для формирования страницы ответа для текущего URL; в этом случае URL, об ошибке в котором сообщает 
страница, берется из свойства request.path данного HTTP-запроса, и в строке браузера отображается запрошенный 
пользователем URL:
```
from framework.views import View, SimpleView
from framework.request import Request
from framework.response import Response
...
class SomeView(View):
  def run(self, request: Request, *args, **kwargs) -> Response:
    ...
    catch FileNotFoundError:
      return View404().run(self.request)
```
2. Перенаправить (redirect) запрос на URL административного интерфейса /notfound, а в параметре page HTTP-запроса 
указать URL, в котором возникла ошибка; важно убедиться, что административный интерфейс фреймворка включен в дерево 
приложения при инициализации объекта класса Framework (см. код):
```
from framework.views import View
from framework.request import Request
from framework.response import Response
...
class SomeView(View):
  def run(self, request: Request, *args, **kwargs) -> Response:
    ...
    catch FileNotFoundError:
      return Response(status_code="307 Temporary redirect", headers={'Location': f'/notfound?page={request.path}"})
...
app = Framework(app_urls, use_admin=True)           # Создать объект приложения и включить административный интерфейс
```
Данный CBV используется классом Framework для отображения информации при ошибках маршрутизации.

### BaseView - CBV на базе иерархических шаблонов Jinja2 с боковыми панелями, заголовком и футером

Данный класс-обработчик используется совместно с рядом связанных классов для реализации иерархического шаблона страницы
приложения пользователя с боковыми панелями, заголовком и футером. Он минимизирует усилия по разработке подобных
визуальных объектов, так как не требует никакой дополнительной предварительной конфигурации по сравнению с любыми
другими CBV фреймворка - все его секции имеют наполнение по умолчанию, которое приложение пользователя может заменять
в любой момент в процессе работы приложения. Его интерфейс полностью аналогичен классу SimpleView (см. описание класса),
и поэтому в данном разделе описаны только его специфические особенности.

Вызов BaseView из приложения полностью аналогичен SimpleView (часть параметров здесь опущена для простоты, так как
параметры по умолчанию устраивают):
```
from framework.views import View, BaseView
from framework.request import Request
from framework.response import Response
...
class SomeView(View):
  def run(self, request: Request, *args, **kwargs) -> Response:
    ...
    return BaseView(template_name='app/someview.html', 
                    some_jinja_parameter='напишем что-нибудь').run(request)
```
В базовой конфигурации класс BaseView выводит шаблон пользователя в оформлении по умолчанию с пустым содержимым в
боковых панелях, заголовке и футере страницы. Изменить оформление базового шаблона BaseView можно, используя интерфейс
singleton-класса BaseViewLayout (см. описание ниже).

В подразделах ниже также описываются другие особенности работы с данным классом-обработчиком.

#### Формат файла шаблона

Файла шаблона для BaseView не является самостоятельным HTML-файлом - это дочерний шаблон (child template) в терминологии 
Jinja2 (см. документацию фреймворка Jinja2), загружаемый в секцию baseview_body базового шаблона 
(см. описание в следующих секциях). В базовом шаблоне текст данного шаблона размещается в разделе body файла HTML, 
поэтому шаблон пользователя не должен содержать HTML-заголовков и разметку раздела body. 
Ниже приведен пример минимального рабочего файла для приведенного выше вызова данного CBV:
```
{% extends base_view_template %}
<!-- Здесь можно разместить какие-либо комментарии, например, описание передаваемых странице параметров Jinja2 -->
<!-- Параметры: some_jinja_parameter - что-то крайне важное
-->
{% block baseview_body %}
<!-- Здесь начинается содержимое секции body страницы пользователя, без заголовков HTML и разметки <body></body> --> 
Вот что мы напишем: {{some_jinja_parameter}}
<!-- Здесь заканчивается содержимое секции body страницы пользователя --> 
{% endblock baseview_body %}
```
В файле шаблона необходима следующая разметка Jinja2:

| Разметка                         | Описание                                                                                                                              |
|----------------------------------|---------------------------------------------------------------------------------------------------------------------------------------|
| {% extends base_view_template %} | Должна быть первой строкой в файле шаблона. Дает шаблонизатору Jinja2 указание включить этот файл в базовый шаблон base_view_template |
| {% block baseview_body %}        | Указывает, что здесь начинается блок текста шаблона, который необходимо разместить в блоке baseview_body базового шаблона             |
| {% endblock baseview_body %}     | Указывает, что здесь заказчивается блок текста шаблона, который необходимо разместить в блоке baseview_body базового шаблона          |

Описание базового шаблона и его секций даны в следующих разделах.
В теле блока может присутствовать любой текст, который возможен в разделе body файла HTML и в шаблоне Jinja2. 
Однако, не следует использовать идентификаторы Jinja2, перечисленные ниже в секции 
"Зарезервированные идентификаторы Jinja2", во избежание непредсказуемого поведения шаблонизатора - данные идентификаторы
должны использоваться строго в соответствии с описанием класса BaseView.

#### Класс BaseViewLayout - конфигуратор базового шаблона.

**Не рекомендуется изменять содержимое базового шаблона BaseView по умолчанию - при обновлении фреймворка все такие 
изменения будут утеряны.** В связи с этим, предусмотрен конфигуратор - singleton-класс BaseViewLayout, который 
предоставляет интерфейс управления конфигурацией BaseView. Из того факта, что данный класс является наследником класса 
Singleton, следует, что в приложении может существовать только одна конфигурация BaseView.

Объект класса BaseViewLayout необязательно создавать в приложении - он создается автоматически при первом вызове
класса-обработчика BaseView. Однако, если требуется кастомизировать базовый шаблон BaseView, можно создать объект класса 
BaseViewLayout вручную до первого вызова BaseView для обработки URL:
```
layout = BaseViewLayout()                                   # Создать объект класса BaseViewLayout
```
Его инициализатор имеет следующие необязательные параметры:

| Параметр              | Тип  | Значение<br/>по<br/>умолчанию | Описание                                                                                                                                                                                                                                                                             |
|-----------------------|------|-------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| set_default_templates | bool | True                          | Указывает, нужно ли загружать базовый шаблон по умолчанию. Даже если указано, что его загружать не надо, базовый шаблон тела страницы будет загружен, чтобы дочерние шаблоны пользователя могли корректно отображаться, но боковые секции, заголовок и футер останутся пустыми       |
| raise_filenotfound    | bool | False                         | Указывает, нужно ли возвращать исключение FileNotFoundError в случае если какой-либо из шаблонов по умолчанию не мог быть загружен; приложение пользователя может также анализировать свойства класса, чтобы определить, какие шаблоны не были загружены (см. описание свойств ниже) |

Приложение пользователя может проанализировать следующие свойства класса BaseViewLayout, чтобы определить, какие шаблоны 
не были загружены при инициализации:

| Свойство       | Тип | Описание                                                                                                   |
|----------------|-----|------------------------------------------------------------------------------------------------------------|
| status_code    | int | 200, если шаблоны загружены успешно, 404 - если при загрузке произошла ошибка                              |
| status_message | str | 'OK', если шаблоны загружены успешно, в противном случае - имя файла шаблона, который не удалось загрузить |

Например, следующий код инициализирует объект класса BaseViewLayout и, в случае если какой-либо шаблон не загружен,
выводит на консоль сообщение об ошибке, не прерывая выполнения приложения:
```
layout = BaseViewLayout()                                   # Создать объект класса BaseViewLayout
if layout.status_code != 200:
    print("Ошибка BaseViewLayout: {}: шаблон {} не найден".format(layout.status_code, layout.status_message))
```

#### Классы BaseViewLayout и BaseViewSection - модификация и загрузка базовых шаблонов пользователя

Класс BaseViewLayout предоставляет интерфейс для загрузки любого из 5 базовых шаблонов BaseView - тела, боковых секций, 
заголовка и футера. Все базовые шаблоны могут иметь Jinja2-разметку. Дополнительно для всех шаблонов, кроме тела,
доступна загрузка второго шаблона, по одному на секцию, обработка (рендеринг, render) которых шаблонизатором Jinja2
производится в момент загрузки шаблона, а не в момент обработки запроса HTTP. Это может быть необходимо для передачи 
в базовый шаблон данных, которые невозможно или нецелесообразно передавать при обработке URL - например, меню навигации 
или другую информацию, имеющую динамическую природу, но не изменяющуюся в зависимости от обрабатываемого HTTP-запроса.

Все базовые шаблоны BaseView по умолчанию размещены в папке framework/templates/baseview/. Их имена файлов приведены 
в таблице ниже.

| Имя файла базового шаблона по умолчанию | Назначение шаблона                                                                                         |
|-----------------------------------------|------------------------------------------------------------------------------------------------------------|
| base_view_template.html                 | Основной базовый шаблон, содержащий разметку для тела (body) страницы, боковых панелей, заголовка и футера |
| header_template.html                    | Базовый шаблон заголовка страницы                                                                          |
| footer_template.html                    | Базовый шаблон футера страницы                                                                             |
| left_sidebar_template.html              | Базовый шаблон левой панели страницы                                                                       |
| right_sidebar_template.html             | Базовый шаблон правой панели страницы                                                                      |

**Не рекомендуется изменять содержимое базовых 
шаблонов BaseView по умолчанию - при обновлении фреймворка все такие изменения будут утеряны.** Вместо этого, базовые 
шаблоны по умолчанию могут быть использованы как основа для создания базовых шаблонов приложения, которые можно затем
загрузить вместо шаблонов по умолчанию с помощью конфигуратора BaseViewLayout, например: 
```
from framework.views import BaseViewLayout, BaseViewSection
...
layout = BaseViewLayout()                                   # Создать объект класса BaseViewLayout
# Предварительно обработать и загрузить пользовательский шаблон левой панели с параметром и удалить оттуда 
# базовый шаблон по умолчанию
layout.render_html_include(BaseViewSection.left_sidebar, "app/app_left_sidebar.html",
                           clear_template=True, template_param="ну вот что-то такое тут слева")
```
##### Формат файла базового шаблона для боковых панелей, заголовка или футера 

Файл загружаемого пользователем базового шаблона для боковых панелей, заголовка или футера может содержать любой 
текст, который может находиться в теле файла HTML (разделе body), включая разметку Jinja2, и может выглядеть, например, 
следующим образом:
```
<p>{{template_param}}</p>
```
##### Формат файла базового шаблона для тела страницы 

Файл загружаемого пользователем базового шаблона для тела страницы представляет собой полноценную HTML-страницу
со всеми необходимыми заголовками и другой служебной информацией, а также обязательными элементами Jinja2-разметки,
необходимыми для функционирования иерархических шаблонов, например (приводится фрагмент базового шаблона тела 
по умолчанию): 
```
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <style>
        body {
            background-color: lightGray;
            font-family: sans-serif;
        }
    </style>
</head>
<body>
<table width="100%">
    <tr>
        <td class="header" colspan="3">{% include header_template ignore missing %}{{header_html_include}}</td>
    </tr>
    <tr>
        <td class="left_sidebar">{% include left_sidebar_template ignore missing %}{{left_sidebar_html_include}}</td>
        <td class="body" width="100%">
            {% block baseview_body scoped %} There should be user app body here. {% endblock baseview_body %}
        </td>
        <td class="right_sidebar">{% include right_sidebar_template ignore missing %}{{right_sidebar_html_include}}</td>
    </tr>
    <tr>
        <td class="footer" colspan="3">{% include footer_template ignore missing %}{{footer_html_include}}</td>
    </tr>
</table>
</body>
</html>
```
Обязательные элементы разметки Jinja2 на вышеприведенной странице служат для загрузки загрузчиком Jinja2 
(см. документацию фреймворка) двух типов шаблонов:
1. шаблонов, чья обработка шаблонизатором Jinja2 осуществляется каждый раз при вызове метода run() класса BaseView 
для обработки URL, вместе с передаваемым в BaseView шаблоном пользователя - эти элементы разметки имеют суффикс 
_template - и
2. шаблонов, чья обработка шаблонизатором Jinja2 осуществляется однократно, заранее, в момент загрузки их конфигуратором 
BaseViewLayout - эти элементы разметки имеют суффикс _include.
Также на странице имеется разметка блока baseview_body, куда помещается шаблон пользователя, передаваемый в BaseView
для обработки URL.

Ниже в таблицах приводится перечень обязательных элементов разметки Jinja2, необходимой для корректного функционирования
BaseView и конфигуратора BaseViewLayout:

| Элемент разметки                                              | Описание                                                                                                                                       |
|---------------------------------------------------------------|------------------------------------------------------------------------------------------------------------------------------------------------|
| {% block baseview_body scoped %} {% endblock baseview_body %} | Границы блока для размещения шаблона страницы пользователя, передаваемого инициализатору элемента класса BaseView                              |
| {% include header_template ignore missing %}                  | Место включения шаблона заголовка, обрабатываемого шаблонизатором Jinja2 каждый раз вместе со страницей пользователя                           |
| {% include left_sidebar_template ignore missing %}            | Место включения шаблона левой боковой панели, обрабатываемого шаблонизатором Jinja2 каждый раз вместе со страницей пользователя                |                                                                            |
| {% include right_sidebar_template ignore missing %}           | Место включения шаблона правой боковой панели, обрабатываемого шаблонизатором Jinja2 каждый раз вместе со страницей пользователя               |
| {% include footer_template ignore missing %}                  | Место включения шаблона футера, обрабатываемого шаблонизатором Jinja2 каждый раз вместе со страницей пользователя                              |
| {{header_html_include}}                                       | Место включения шаблона заголовка, обрабатываемого шаблонизатором Jinja2 однократно при загрузке его конфигуратором BaseViewLayout             |
| {{left_sidebar_html_include}}                                 | Место включения шаблона левой боковой панели, обрабатываемого шаблонизатором Jinja2 однократно при загрузке его конфигуратором BaseViewLayout  |
| {{right_sidebar_html_include}}                                | Место включения шаблона правой боковой панели, обрабатываемого шаблонизатором Jinja2 однократно при загрузке его конфигуратором BaseViewLayout |
| {{footer_html_include}}                                       | Место включения шаблона футера, обрабатываемого шаблонизатором Jinja2 однократно при загрузке его конфигуратором BaseViewLayout                |

Другое наполнение страницы базового шаблона тела - стили, таблицы и другие элементы оформления и данные - оставляется
на усмотрение пользователя. Необходимо понимать, что при изменении структуры таблицы, содержащей обязательные элементы 
разметки Jinja2, взаимное размещение шаблонов на экране изменится, и их названия могут перестать соответствовать
позициям на экране.

##### Загрузка определяемого пользователем базового шаблона и возврат к шаблону по умолчанию

Загрузка определяемых пользователем базовых шаблонов осуществляется с помощью следующих методов объекта класса
BaseViewLayout:
1. set_template - для загрузки шаблонов, чья обработка шаблонизатором Jinja2 осуществляется каждый раз при вызове метода 
run() класса BaseView для обработки URL, вместе с передаваемым в BaseView шаблоном пользователя; с помощью этого
метода можно заменить любой из шаблонов по умолчанию, в том числе и шаблон тела страницы, а также вернуть для указанного
раздела базовый шаблон по умолчанию: 
```
def set_template(self, section: BaseViewSection,
                 template_file_name: str = None, template_path: str = None,
                 clear_html_include: bool = False, raise_filenotfound: bool = False) -> bool:
```
Параметры метода описаны в следующей таблице:

| Параметр           | Тип             | Значение по умолчанию     | Описание                                                                                                                                                                                                                                      |
|--------------------|-----------------|---------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| section            | BaseViewSection | - (обязательный параметр) | Раздел базового шаблона, который необходимо заменить, например, BaseViewSection.left_sidebar - левая боковая секция; описание значений типа BaseViewSection дано в таблице ниже                                                               |
| template_file_name | str             | None                      | Имя файла шаблона; может содержать, кроме имени, также и путь к файлу шаблона; если параметр не указан, функция заменяет текущий шаблон секции на шаблон по умолчанию для данной секции; имена шаблонов по умолчанию приведены в таблице выше |
| template_path      | str             | None                      | Для удобства здесь можно отдельно указать путь к шаблону; если указано имя файла шаюлона, содержимое данного параметра будет подставлено перед именем файла (конкатенация строк)                                                              |
| clear_html_include | bool            | False                     | Флаг, указывающий, нужно ли очищать содержимое шаблона, обрабатываемого Jinja2 при его загрузке (см. суффикс _include выше), для этого раздела                                                                                                |
| raise_filenotfound | bool            | False                     | Флаг, указывающий, надо ли возвращать исключение (exception) FileNotFoundError в случае если файл шаблона не найден                                                                                                                           |

Значения класса BaseViewSection соответствуют следующим секциям базового шаблона:

| Значение      | Секция базового шаблона |
|---------------|-------------------------|
| base_view     | тело страницы           |
| header        | заголовок               |
| footer        | футер                   |
| left_sidebar  | левая боковая секция    |
| right_sidebar | правая боковая секция   |

Метод возвращает True, если шаблон успешно загружен, и False, если произошла ошибка загрузки и 
raise_filenotfound==True - в этом случае можно анализировать свойства класса status_code и status_message, как это
указано для инициализатора класса. Например:
```
from framework.views import BaseViewLayout, BaseViewSection
...
layout = BaseViewLayout()                                   # Создать объект класса BaseViewLayout
# Загрузить пользовательский шаблон левой панели и удалить базовый шаблон _include, если он есть
if not layout.set_template(BaseViewSection.left_sidebar, "app/app_left_sidebar.html", clear_html_include=True):
  print("Ошибка BaseViewLayout.set_template: {}: шаблон {} не найден".format(layout.status_code, layout.status_message))
```
2. render_html_include - для загрузки шаблонов, чья обработка шаблонизатором Jinja2 осуществляется однократно, заранее, 
в момент загрузки их конфигуратором BaseViewLayout; с помощью этого метода можно установить любой из заранее 
обрабатываемых шаблонов, кроме шаблона тела страницы, который всегда обрабатывается вместе с шаблоном пользователя: 
```
def render_html_include(self, section: BaseViewSection,
                        template_file_name: str, template_path: str = None,
                        clear_template: bool = False, raise_filenotfound: bool = False, **kwargs) -> bool:
```
Параметры метода описаны в следующей таблице:

| Параметр           | Тип                                  | Значение по умолчанию     | Описание                                                                                                                                                                                                                                                                                      |
|--------------------|--------------------------------------|---------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| section            | BaseViewSection                      | - (обязательный параметр) | Раздел базового шаблона, который необходимо заменить, например, BaseViewSection.left_sidebar - левая боковая секция; описание значений типа BaseViewSection дано в таблице выше; значение BaseViewSection.base_view, хоть и не вызовет ошибки, не приведет к отображению загруженного шаблона |
| template_file_name | str                                  | - (обязательный параметр) | Имя файла шаблона; может содержать, кроме имени, также и путь к файлу шаблона                                                                                                                                                                                                                 |
| template_path      | str                                  | None                      | Для удобства здесь можно отдельно указать путь к шаблону; содержимое данного параметра, если указано, будет подставлено перед именем файла (конкатенация строк)                                                                                                                               |
| clear_template     | bool                                 | False                     | Флаг, указывающий, нужно ли очищать содержимое шаблона, обрабатываемого Jinja2 вместе с шаблоном пользователя (см. суффикс _template выше), для этого раздела                                                                                                                                 |
| raise_filenotfound | bool                                 | False                     | Флаг, указывающий, надо ли возвращать исключение (exception) FileNotFoundError в случае если файл шаблона не найден                                                                                                                                                                           |
| **kwargs           | (произвольные именованные параметры) | (отсутствуют)             | В случае наличия в загружаемом шаблоне параметров Jinja2, их можно передать шаблонизатору после всех обязательных и опциональных параметров метода                                                                                                                                            |

Значения класса BaseViewSection описаны в таблице выше.

Метод возвращает True, если шаблон успешно загружен, и False, если произошла ошибка загрузки и 
raise_filenotfound==True - в этом случае можно анализировать свойства класса status_code и status_message, как это
указано для инициализатора класса. Например:
```
from framework.views import BaseViewLayout, BaseViewSection
...
layout = BaseViewLayout()                                   # Создать объект класса BaseViewLayout
# Предварительно обработать и загрузить пользовательский шаблон левой панели с параметром и удалить оттуда 
# базовый шаблон по умолчанию
if not layout.render_html_include(BaseViewSection.left_sidebar, "app/app_left_sidebar.html",
                                  clear_template=True, template_param="ну вот что-то такое тут слева")
  print(f"Ошибка BaseViewLayout.render_html_include: {layout.status_code}: шаблон {layout.status_message} не найден")
```

#### Зарезервированные идентификаторы Jinja2

В шаблонах приложения, предназначенных для класса-обработчика BaseView, не следует использовать идентификаторы Jinja2, 
перечисленные ниже, во избежание непредсказуемого поведения шаблонизатора - данные идентификаторы
должны использоваться строго в соответствии с описанием класса BaseView:

| Идентификаторы (a-f) | Идентификаторы (h-l)      | Идентификаторы (r)         |
|----------------------|---------------------------|----------------------------|
| baseview_body        | header_html_include       | right_sidebar_html_include |
| base_view_template   | header_template           | right_sidebar_template     |
| footer_html_include  | left_sidebar_html_include |                            |
| footer_template      | left_sidebar_template     |                            |

### AdminView - CBV на базе SimpleView/MinimalView для диагностики 

Данный класс-обработчик реализует интерфейс базовой диагностики сервера WSGI и маршрутизации фреймворка.
Он отображает страницу, где в табличном виде представлено содержимое словаря environ окружения WSGI-сервера,
передаваемого им при вызове приложение для обработки HTTP-запроса: параметры HTTP-запроса и другие переменные
окружения WSGI-сервера. Также пользователь может протестировать отправку и декодирование двух произвольных текстовых 
параметров запроса методами GET и POST.

## Библиотека Persistence - Хранение (Объектно-реляционное отображение) данных (ORM) (persistence*.py)

Данная библиотека фреймворка реализует базовую функциональность ORM (Object-Relational Mapping) - сохранения и загрузки 
объектов Python в хранилища данных. Реализовано отображение объектов в файлы JSON (persistence.py) и таблицы баз данных 
SQLite (persistence_sqlite.py). Предусмотрены интерфейсы для добавления пользователем классов, реализующих отображение 
объектов в другие хранилища данных без необходимости модификации кода фреймворка - примером такого класса является
модуль SQLite, реализованный как внешний по отношению к фреймворку и регистрирующийся в нем.

Библиотека не требует написания функций для отображения объекта класса в объект хранения, однако, она тестировалась 
только на классах, имеющих атрибуты - простые типы данных - int, str - а также типы datetime.datetime и uuid.UUID; 
тестирование библиотеки на классах, имеющих атрибуты-коллекции, объекты с собственными атрибутами и прочие структуры 
данных, не производилось. 

Предполагается, что каждый объект сохраняемого класса имеет свойство id класса uuid.UUID, но 
некоторые методы хранилищ данных (PersistentEngine), такие как get() без параметров и append(), будут корректно работать 
и без поля id, что делает возможным, например, реализации логгера с использованием библиотеки Persistence - для этого 
пришлось бы только написать реализацию хранилища CSV. 
Ссылки на объекты других классов также производятся с использованием полей класса uuid.UUID; 
реляционная целостность связанных таким образом данных не поддерживается; уникальность полей id и любых других 
не проверяется; задание значений полей по умолчанию, автозаполняемых полей и обязательных полей не предусмотрено.

Чтобы библиотека могла корректно работать с классом пользователя, он должен быть подклассом класса 
PersistenceSerializable - в этом случае кодеки хранилищ данных смогут распознать
класс пользователя и надлежащим образом закодировать его свойства, которые не могут быть непосредственно сохранены в
данном хранилище. Например, хранилища JSON и SQLite не могут непосредственно хранить поля классов datetime.datetime и 
uuid.UUID, которые предварительно преобразуются кодеками в строку дата/время и шестнадцатеричное число, соответственно.

Каждый отдельный класс может быть сохранен в своем формате и в отдельном файле - например, один класс может быть
сохранен в хранилище JSON, а другой - в отдельную таблицу базы данных SQLite вместе с какими-либо другими классами в
других таблицах этой базы. Для этого каждый класс должен быть зарегистрирован в соответствующем хранилище.

В разделах ниже дается описание:
- в разделе "Создание, регистрация, сохранение и загрузка классов пользователя" - работы с существующими реализациями 
хранилищ данных,
- в разделе "Реализация дополнительного хранилища данных" - расширения списка поддерживаемых средствами библиотеки 
хранилищ.

### Создание, регистрация, сохранение и загрузка классов пользователя

Чтобы иметь возможность работать с объектами класса пользователя с помощью библиотеки Persistence, необходимо:
1. создать класс пользователя, сделав его наследником класса PersistenceSerializable;
2. если класс пользователя необходимо хранить во внешнем по отношению к библиотеке хранилище, таком как SQLite,
зарегистрировать хранилище в библиотеке (метод PersistenceSQLite.register() для SQLite);
3. зарегистрировать класс пользователя в хранилище данных, указав тип хранилища и место его размещения 
(метод Persistence.register_class());
4. воспользоваться функциями библиотеки для добавления/изменения/чтения объектов класса в/из хранилища.

Ниже дан пример использования библиотеки Persistence для работы с классом пользователя (файл app_test_persistence.py):
```
import uuid
from datetime import datetime
from dataclasses import dataclass
from framework.persistence_sqlite import PersistenceSQLite, ENGINE_SQLITE
from framework.persistence import Persistence, PersistenceSerializable


# Класс пользователя наследует PersistenceSerializable для надлежащей обработки библиотекой Persistence
@dataclass
class Personnel(PersistenceSerializable):
    id: uuid.UUID
    name: str
    score: int
    birth_date: datetime
    supervisor_id: uuid.UUID

    @staticmethod                 # Зарегистрировать класс пользователя в библиотеке Persistence
    def register() -> bool:
        return Persistence.register_class(Personnel, ENGINE_SQLITE, 'database.sqlite')


PersistenceSQLite.register()  # Зарегистрировать хранилище данных SQLite в библиотеке Persistence
if not Personnel.register():
    print("Не удалось зарегистрировать класс Personnel в базе данных")
else:
    # Создать объект класса пользователя
    person = Personnel(id=uuid.uuid4(), 
                       name="Мовлатшо Довлатшоевич", score=0, birth_date=datetime.now(), supervisor_id=None)
    Persistence.engine(Personnel).set(person)       # Сохранить его
    print(Persistence.engine(Personnel).get())      # Напечатать весь список объектов в хранилище
```
Ниже приводится детальное объяснение приведенных выше шагов и дается описание классов и методов в библиотеке Persistence 
для работы с классами пользователя.

#### PersistenceSerializable - требования к типам данных пользователя. 

Библиотека ориентирована на максимальное упрощение и унификацию операций по сохранению и загрузке из внешних хранилищ
типичных данных web-приложений - контактных данных пользователей, информации о товарах и услугах и т.п. Атрибутами 
таких классов данных обычно являются т.н. простые типы данных - строковые (str) и числовые (int) поля, 
логические значения (bool), дата и время (datetime.datetime), уникальные идентификаторы записей (uuid.UUID). 
Библиотека тестировалась только на подобных "одноуровневых" классах данных, и сохранение объектов классов, 
хранящих в своих свойствах коллекции, деревья и прочие вложенные структуры данных, может работать 
некорректно.

Библиотека не требует от пользователя написания функций для отображения объекта класса в объект хранения (ORM-функций). 
Однако, кодек
библиотеки json, например, не имеет обработчика по умолчанию для пользовательских классов. Чтобы преобразовать класс
пользователя в понятные библиотеке json данные, кодек библиотеки Persistence распознает наследников класса
PersistenceSerializable и преобразует данные пользователя для их корректной обработки библиотекой json. Таким образом,
во избежание несовместимости, классы пользователя, чьи объекты планируется обрабатывать с помощью библиотеки 
Persistence, следует объявлять наследниками класса PersistenceSerializable.

Также для работы большинства методов библиотеки требуется, чтобы каждый объект сохраняемого класса имел свойство id 
класса uuid.UUID; в отсутствии такого свойства, будут корректно работать только методы PersistentEngine.get() 
(без параметров) и .append(). Ссылки на объекты других классов также следует хранить с использованием полей класса 
uuid.UUID, так как вложенность объектов не поддерживается.

Таким образом, для корректной работы с библиотекой Persistence классы пользователя должны удовлетворять следующим 
требованиям:
1. Наследовать класс PersistenceSerializable;
2. Иметь только атрибуты - "простые" типы данных: int, bool, str и подобные, а также datetime.datetime и uuid.UUID;
3. Иметь свойство id класса uuid.UUID, хранящее уникальный идентификатор объекта класса;
4. Ссылаться на другие объекты данного класса и объекты других классов по идентификатору класса uuid.UUID, а не хранить
их как вложенные объекты (следует из пп.2 и 3).

Ниже приведен пример поддерживаемого класса пользователя:
```
from framework.persistence PersistenceSerializable

@dataclass
class Personnel(PersistenceSerializable):
    id: uuid.UUID
    name: str
    score: int
    birth_date: datetime
    supervisor_id: uuid.UUID
```

#### Регистрация хранилища данных SQLite и других внешних хранилищ

Библиотека "из коробки" поддерживает сохранение данных в файлах JSON.
Также реализован дополнительный модуль для сохранения данных в базах данных SQLite. 
Если планируется хранить данные в
SQLite, необходимо зарегистрировать данный модуль в библиотеке Persistence:
```
from framework.persistence_sqlite import PersistenceSQLite

PersistenceSQLite.register()  # Зарегистрировать хранилище данных SQLite в библиотеке Persistence
```
В случае использования другого внешнего хранилища, поддержка которого реализована силами третьих лиц, обратитесь к
документации такого хранилища для получения инструкций по регистрации его в библиотеке Persistence.

#### Регистрация классов данных пользователя

Чтобы библиотека Persistence знала, где хранить данные пользователя, каждый класс, объекты которого требуется хранить
во внешнем хранилище, необходимо зарегистрировать в библиотеке. 

Класс регистрируется в библиотеке с помощью статического метода Persistence.register_class(). При регистрации 
пользователь указывает тип хранилища и место размещения файла (базы) хранилища. Например, следующий код регистрирует 
объявленный пользователем класс Personnel для хранения в файле JSON 'personnel.json':
```
from framework.persistence import Persistence, PersistenceSerializable, ENGINE_JSON

class Personnel(PersistenceSerializable):
    ...                 # Свойства и методы класса пользователя

if not Persistence.register_class(Personnel, ENGINE_JSON, 'personnel.json'):
    print("Не удалось зарегистрировать класс Personnel")
```
В приложении-примере реализован другой подход к регистрации класса - когда регистрация производится вызовом метода
самого регистрируемого класса. Хотя было принято решение не навязывать разработчику этот подход, определяя метод 
register() в интерфейсе PersistenceSerializable, он видится разработчикам более изящным, так как избавляет код
основного приложения от подробностей регистрации классов:
```
from framework.persistence import Persistence, PersistenceSerializable, ENGINE_JSON

class Personnel(PersistenceSerializable):
    ...                 # Свойства и методы класса пользователя

    @staticmethod                 # Зарегистрировать класс пользователя в библиотеке Persistence
    def register() -> bool:
        return Persistence.register_class(Personnel, ENGINE_JSON, 'personnel.json')
        
if not Personnel.register():      # Вызвать метод регистрируемого класса
    print("Не удалось зарегистрировать класс Personnel")
```
Метод Persistence.register_class() имеет следующие параметры:

| Параметр          | Тип                    | Описание                                                                                                                                                                                                                                                   |
|-------------------|------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| class_to_register | (идентификатор класса) | Идентификатор регистрируемого класса                                                                                                                                                                                                                       |
| engine_type       | str                    | Тип хранилища данных; рекомендуется использовать константы, объявленные в библиотеках, реализующих интерфейс Persistence к хранилищам данных; текущий список констант приведен в таблице ниже                                                              |
| data_source       | str                    | Путь к файлу хранилища данных; каждый файл JSON может хранить данные только одного класса пользователя; файл SQLite может хранить данные разных классов в разных таблицах; ограничения реализаций других хранилищ смотрите в документации таких реализаций |

 В библиотеке определены следующие константы типов хранилища данных engine_type (см. таблицу ниже). Обратите внимание, 
 что каждая константа
 определена в модуле, где размещена реализация соответствующего хранилища - JSON, SQLite - и что ее необходимо 
 импортировать из соответствующего модуля. Также для использования любого хранилища, кроме JSON, хранилище необходимо 
 предварительно зарегистрировать, как это описано в разделе выше.
 
| Константа     | Модуль                       | Описание                                                                                                                                                                                                                                                                                                                                                                                         |
|---------------|------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| ENGINE_JSON   | framework.persistence        | Хранилище JSON; один файл хранит объекты одного класса как массив словарей (list of dict)                                                                                                                                                                                                                                                                                                        |
| ENGINE_SQLITE | framework.persistence_sqlite | Хранилище SQLite; для использования данного хранилища необходимо предварительно зарегистрировать его с помощью метода PersistenceSQLite.register(); один файл может хранить объекты разных классов, по одному в таблице; имя таблицы совпадает с именем хранимого в ней класса; каждый объект класса хранится в отдельной строке таблицы, каждое свойство объекта хранится в собственной колонке |

#### Работа с данными - сохранение, добавление, загрузка, удаление

После регистрации внешних хранилищ данных и регистрации классов пользователя в библиотеке Persistence, можно приступать
к работе с данными. Для этого в библиотеке предусмотрены 4 метода, функционирующие одинаково для всех хранилищ, если
не указано иное. Пример использующего эти методы кода приведен ниже:
```
# Создать объект класса пользователя
person = Personnel(id=uuid.uuid4(), 
                   name="Мовлатшо Довлатшоевич", score=0, birth_date=datetime.now(), supervisor_id=None)
Persistence.engine(Personnel).set(person)       # Сохранить его
print(Persistence.engine(Personnel).get())      # Напечатать весь список объектов в хранилище
```
Вызов Persistence.engine(Personnel) возвращает объект подкласса PersistenceEngine - реализации конкретного хранилища 
данных - JSON, SQLite или другого - методы которого можно вызывать для доступа к данным конкретного класса, в случае 
кода выше - класса Personnel. В случае если таких операций доступа планируется много, можно оптимизировать код выше:
```
# Создать объект класса пользователя
person = Personnel(id=uuid.uuid4(), 
                   name="Мовлатшо Довлатшоевич", score=0, birth_date=datetime.now(), supervisor_id=None)
personnel = Persistence.engine(Personnel) 
personnel.set(person)       # Сохранить его
print(personnel.get())      # Напечатать весь список объектов в хранилище
```
Для работы с данными предусмотренные следующие 4 метода интерфейса (подклассов класса) PersistenceEngine. Обратите 
внимание, то не все исключения перехватываются данными методами, поэтому если для приложения важно продолжать выполнение 
в случае возникновения исключений, ему необходимо перехватывать их самостоятельно.

| Метод                            | Описание                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
|----------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| get(element_id: UUID = None)     | Возвращает массив всех сохраненных объектов с полем id=element_id, если они существуют, или пустой массив, если таких объектов нет; возвращает массив всех сохраненных объектов данного класса или пустой массив, если нет сохраненных объектов; возвращает пустой массив в случае возникновения ошибки; реализация JSON перехватывает исключения FileNotFoundError и json.JSONDecodeError и записывает объект ошибки и сообщение об ошибке в свойства подкласса PersistenceEngine error_object и error_message, соответственно |
| set(element) -> bool             | Заменяет все сохраненные объекты с полем id=element.id, если таковые существуют, на element, или сохраняет новый объект element, если сохраненных объектов с id=element.id нет; возвращает True в случае успеха и False в случае какой-либо перехваченной ошибки, например, реализация JSON перехватывает FileNotFoundError и json.JSONDecodeError                                                                                                                                                                              |
| delete(element_id: UUID) -> bool | Удаляет все сохраненные объекты с полем id=element_id, если таковые существуют; возвращает True в случае успеха и False в случае какой-либо перехваченной ошибки, например, реализация JSON перехватывает FileNotFoundError и json.JSONDecodeError                                                                                                                                                                                                                                                                              |
| append(element) -> bool          | Сохраняет новый объект element, независимо от того, есть ли уже такой сохраненный объект; возвращает True в случае успеха и False в случае какой-либо перехваченной ошибки, например, реализация JSON перехватывает FileNotFoundError и json.JSONDecodeError                                                                                                                                                                                                                                                                    |

Для работы всех вышеперечисленных методов библиотеки, в случае если производится проверка совпадения значений поля id,
требуется, чтобы каждый объект сохраняемого класса имел свойство id класса uuid.UUID; в случае если совпадение id не
проверяется, допускается отсутствие такого свойства у объекта. Таким образом, например, можно реализовать логгер с 
использованием библиотеки Persistence для записи логов, используя методы get() 
(без параметров) - для получения записей лога для отображения - и .append() - для записи логов.

### Реализация дополнительного хранилища данных

Библиотека позволяет реализовывать дополнительные хранилища данных без необходимости изменять существующий код 
библиотеки. Например, можно реализовать хранение данных на удаленном сервере, передавая ему и загружая с него данные 
по HTTP, при этом в коде приложения пользователя понадобится только зарегистрировать новое хранилище и указать на него
при регистрации классов пользователя, как это описано выше - код приложения для работы с данными не поменяется.

Демонстрацией возможности расширения библиотеки является модуль persistence_sqlite.py - реализация хранилища SQLite.

Для реализации дополнительного хранилища данных необходимо в файле Python данной реализации (для хранилища SQLite - 
persistence_sqlite.py):
1. Объявить строковую константу типа хранилища, например, для SQLite:
```
ENGINE_SQLITE = 'sqlite'                # Константа типа хранилища для регистрации и ссылки на него из приложения
```
2. Создать класс - реализацию интерфейса PersistenceEngine, реализовав его инициализатор и все необходимые абстрактные 
методы работы с данными, а также статический метод register() для регистрации хранилища в библиотеке Persistence, 
например, для SQLite:
```
from framework.persistence import Persistence, PersistenceEngine

class PersistenceSQLite(PersistenceEngine):
  ...                                   # Реализация методов интерфейса PersistenceEngine
  
    @staticmethod                       # Метод для регистрации хранилища в библиотеке Persistence
    def register() -> bool:
        return Persistence.register_engine(ENGINE_SQLITE, PersistenceSQLite)
```
Хотя метод register() и не указан как обязательный в интерфейсе PersistenceEngine, его рекомендуется реализовать,
чтобы минимизировать ошибки пользователя при регистрации хранилища и упростить код приложения.

Регистрация хранилища в библиотеке Persistence осуществляется с помощью статического метода класса Persistence 
register_engine() со следующими обязательными параметрами:

| Параметр           | Тип                                                   | Описание                                                                    |
|--------------------|-------------------------------------------------------|-----------------------------------------------------------------------------|
| engine_type        | str                                                   | строковая константа, идентифицирующая тип хранилища                         |
| engine_initializer | (идентификатор класса - наследника PersistenceEngine) | идентификатор класса - наследника PersistenceEngine, реализующего хранилище |

Метод register_engine() возвращает True в случае успеха и False в ином случае.
3. В приложении пользователя, использующем дополнительное хранилище, зарегистрировать его в библиотеке Persistence перед 
использованием, например, для SQLite:
```
from framework.persistence_sqlite import PersistenceSQLite

PersistenceSQLite.register()  # Зарегистрировать хранилище данных SQLite в библиотеке Persistence
```
После этого с дополнительным хранилищем можно в приложениии пользователя работать, как с любым другим хранилищем 
библиотеки Persistence.

Интерфейс PersistenceEngine объявлен следующим образом:
```
class PersistenceEngine(ABC):

    @abstractmethod                     # Инициализатор
    def __init__(self, data_source: str, class_initializer):
        pass

    @abstractmethod                     # Получение объекта(-ов)
    def get(self, element_id: UUID = None):
        pass

    @abstractmethod                     # Изменение или добавление объекта(-ов)
    def set(self, element) -> bool:
        pass

    @abstractmethod                     # Удаление объекта(-ов)
    def delete(self, element_id: UUID) -> bool:
        pass

    @abstractmethod                     # Добавление объекта(-ов) без проверки дубликатов
    def append(self, element) -> bool:
        pass
```
Подробное описание алгоритмов методов работы с данными, которые должно реализовывать дополнительное хранилище, дано 
выше в разделе, посвященном использованию хранилищ библиотеки Persistence.

Подробности реализации интерфейса к хранилищу остаются на усмотрение пользователя. Например, реализация SQLite не 
использует класс Singleton, но задает атрибут класса _connections - словарь (dict), содержащий все открытые 
соединения к файлам SQLite во избежание дублирования таких соединений; подобная практика может использоваться и при
работе с базами данных, где дублирование соединений также нежелательно.

## Библиотека Logger - простой иерархический логгер (logger.py)

Библиотека предназначена для упрощения организации логирования событий в приложении пользователя. Она предоставляет 
пользователю неограниченное количество именованных логгеров, каждый из которых может записывать логи в один или 
несколько именованных обработчиков (handler) одновременно, при этом количество обработчиков также не ограничено. 
Каждый обработчик, в зависимости от своего типа, может 
в свою очередь записывать логи в файл или выводить их на консоль; возможно добавление пользователем других типов 
обработчиков, например, для передачи логов по сети или во внешние системы через интерфейсы интеграции. 
Обработчиков одного типа - консольных, файловых и прочих - может быть неограниченное количество, например, 
для вывода логов в разные файлы. 
Также для каждого логгера и обработчика можно задавать уровень логирования (log level) по умолчанию, когда он не указан
для данного сообщения пользователя. Можно, кроме того, задать для каждого логгера и обработчика минимальный уровень 
логирования сообщения, которое попадет в лог, а сообщения ниже уровнем будут сброшены: например, если минимальный 
уровень - WARNING, то сообщения уровня INFORMATION в логи не попадут. Можно также индивидуально отключать любой логгер
и обработчик, а также отключить логирование в целом. Есть возможность указать логгер по умолчанию, когда для данного
сообщения он не указан.

Пример работы с библиотекой приведен в файле app_test_logger.py . Фрагменты его кода используются ниже для объяснения
принципов работы библиотеки.

### Инициализация библиотеки

Для начала работы с библиотекой ее необходимо инициализировать. Для этого программой пользователя создается 
Singleton-объект LoggerFabric - фабрика логгеров. Во фрагменте кода ниже при инициализации логгер включается (turn_on), 
задаются конфигурации обработчиков (handlers) и логгеров (loggers) (эти структуры описываются ниже в данном разделе), 
в качестве логгера по умолчанию (default_logger) указывается логгер "debug", а уровень логирования по умолчанию
(default_level) устанавливается как DEBUG:
```
from framework.logger import LoggerFabric, LoggerLevel

# До инициализации логгера необходимо задать конфигурацию обработчиков (handlers) и логгеров (loggers)
...

logger = LoggerFabric(turn_on=True, 
                      handlers=handlers, loggers=loggers, 
                      default_logger="debug", default_level=LoggerLevel.DEBUG)
```
В таблице ниже описаны параметры инициализатора класса LoggerFabric:

| Параметр       | Тип          | Значение<br/>по умолчанию | Описание                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
|----------------|--------------|---------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| turn_on        | bool         | None                      | Включает или выключает логирование в библиотеке. Если значение не передано при первой инициализации объекта, логирование включается, как если бы turn_on=True. Если turn_on=False, логирование не производится вообще.                                                                                                                                                                                                                                                                                                                                                                                  |
| handlers       | dict of dict | None                      | Инициализирует обработчики. Параметр принимает словарь словарей, где ключом каждой записи словаря является имя обработчика, а значением - словарь, указывающий тип обработчика и его параметры. Имена логгеров и обработчиков могут совпадать - их пространства имен независимы. При каждом повторном вызове инициализатора с указанием данного параметра обновляется конфигурация существующих одноименных обработчиков и добавляются новые. Детальное описание инициализации обработчика и формата данных словаря дается ниже в этом разделе.                                                         |
| loggers        | dict of dict | None                      | Инициализирует логгеры. Параметр принимает словарь словарей, где ключом каждой записи словаря является имя логгера, а значением - словарь, указывающий тип логгера и его параметры. Имена логгеров и обработчиков могут совпадать - их пространства имен независимы. При каждом повторном вызове инициализатора с указанием данного параметра обновляется конфигурация существующих одноименных логгеров и добавляются новые. Детальное описание инициализации логгера и формата данных словаря дается ниже в этом разделе.                                                                             |
| default_logger | str          | None                      | Устанавливает логгер по умолчанию для библиотеки, в который будет записано сообщение лога, когда логгер не указан приложением пользователя при вызове функции логирования. Если такого логгера не существует, то в случае попытки его использования сообщение лога не будет записано. Если параметр не задан при первой инициализации, в качестве логгера по умолчанию будет выбран первый в текущем списке зарегистрированных в библиотеке логгеров, а если список пуст - логгер по умолчанию не будет задан, и попытка записи в логгер по умолчанию приведет к тому, что сообщение не будет записано. |
| default_level  | LoggerLevel  | None                      | Устанавливает уровень логирования (log level) по умолчанию для библиотеки, который будет присвоен сообщению, уровень логирования для которого не указан, если он также не указан ни для обработчика, ни для логгера. Если параметр не задан при первой инициализации, уровень логирования по умолчанию устанавливается равным DEBUG.                                                                                                                                                                                                                                                                    |

Каждый раз когда инициализатор класса LoggerFabric вызывается программой пользователя, значения переданных параметров
используются для изменения конфигурации библиотеки, то есть, можно в ходе выполнения программы включать и выключать 
логирование, менять его уровень и т.п.

Для указания уровня логирования (log level) в библиотеке используется класс-перечисление LoggerLevel. 
Список уровней логирования приведен в следующей таблице вместе с их численными значениями. Когда в данном описании
говорится, что уровень логирования ниже, имеется в виду, что его численное значение меньше, например, уровень
WARNING ниже уровня ERROR:

| Уровень логирования  | Численное значение | Описание                       |
|----------------------|--------------------|--------------------------------|
| LoggerLevel.DEBUG    | 0                  | Отладочное сообщение           |
| LoggerLevel.INFO     | 1                  | Информационное сообщение       |
| LoggerLevel.WARNING  | 2                  | Предупреждение                 |
| LoggerLevel.ERROR    | 3                  | Сообщение об ошибке            |
| LoggerLevel.CRITICAL | 4                  | Сообщение о критической ошибке |

#### Функции обработчиков и формат файла лога

Обработчики служат для записи в них сообщений логгерами и представляют собой интерфейс логгера к конкретной среде 
хранения или отображения текстовой информации. Каждый обработчик, в зависимости от своего типа, может 
записывать логи в файл или выводить их на консоль; возможно добавление пользователем других типов 
обработчиков, например, для передачи логов по сети или во внешние системы через интерфейсы интеграции. 
Обработчиков одного типа - консольных, файловых и прочих - может быть неограниченное количество, например, 
для вывода логов в разные файлы. Каждый обработчик имеет уникальное имя, при этом имена логгеров и обработчиков 
могут совпадать - их пространства имен независимы. 
 
Как и для библиотеки в целом, для каждого обработчика можно задавать уровень логирования (log level) по умолчанию, 
когда он не указан для данного сообщения пользователя. Можно, кроме того, задать для каждого обработчика 
минимальный уровень логирования сообщения, которое попадет в лог, а сообщения ниже уровнем будут сброшены: 
например, если минимальный уровень - WARNING, то сообщения уровня INFORMATION в логи не попадут. 
Можно также индивидуально отключать любой обработчик.

Все обработчики являются наследниками абстрактного класса LoggerHandler. Реализация этого класса инициализирует общие
для всех обработчиков свойства в своем инициализаторе __init__(), а также в методе log() проверяет, 
нужно ли записывать сообщение в лог данного обработчика, сравнивая уровень логирования сообщения 
с минимальным уровнем логирования обработчика, и конструирует строку сообщения лога вида:
```
2022-04-29T19:56:02.917618, DEBUG, debug, app, Test debug message
```
Видно, что лог имеет формат CSV. Поля лога в порядке их следования описаны в таблице ниже:

| Номер п/п | Поле                                | Описание                                                                        |
|-----------|-------------------------------------|---------------------------------------------------------------------------------|
| 1         | Метка даты/времени                  | Дата/время формирования сообщения обработчиком в формате ISO                    |
| 2         | Уровень логирования<br/>(log level) | Общепринятое мнемоническое обозначение уровня логирования                       |
| 3         | Имя логгера                         | Имя логгера, вызвавшего данный обработчик для записи сообщения                  |
| 4         | Источник сообщения                  | Источник сообщения, как он указан логгером, вызвавшим обработчик для его записи |
| 5         | Сообщение                           | Сообщение, как оно было передано логгером                                       |

Следует иметь в виду, что если имя логгера, источник сообщения или само сообщение содержит запятые, 
информация в столбцах сместится. В связи с этим, не рекомендуется использовать запятые в имени логгера и
названии источника сообщения, а само сообщение при автоматизированном импорте необходимо конкатенировать из столбцов 
с пятого по последний.

#### Иницализация обработчиков

Для инициализации или изменения параметров обработчиков, описывающую их структуру необходимо передать в качестве 
параметра handlers в инициализатор класса LoggerFabric (см. описание выше). Эта структура представляет собой словарь 
словарей, где ключом каждой записи словаря является имя обработчика, а значением - словарь, 
содержащий конфигурацию обработчика - его тип, имя и параметры (см. пример ниже).      

**!ВАЖНО!** Обратите внимание, что если вызов инициализатора класса LoggerFabric осуществляется повторно и в него 
передается параметр handlers, содержащий конфигурацию уже существующего обработчика, такой обработчик будет создан 
заново, то есть, в новой конфигурации нельзя указать только измененные параметры, 
а необходимо указать его полную конфигурацию, принимая во внимание значения по умолчанию для опущенных необязательных 
параметров, описание которых дается в таблице ниже.

Например, приведенный ниже фрагмент кода задает конфигурацию 3 (трех) обработчиков - 
'console' типа LoggerConsoleHandler и 'debug_file' и 'app_file' типа LoggerFileHandler:
```
# ********** Данный раздел в реальном проекте рекомендуется вынести в отдельный файл logs.py *********
handlers = {
    'console': {
        'handler': 'LoggerConsoleHandler',
        'name': 'console',
        'level': 0,
        'default_level': 0,
        'is_on': True,
    },
    'debug_file': {
        'handler': 'LoggerFileHandler',
        'name': 'debug_file',
        'file_name': 'debug.log',
        'level': 0,
        'default_level': 0,
        'is_on': True,
    },
    'app_file': {
        'handler': 'LoggerFileHandler',
        'name': 'app_file',
        'file_name': 'app.log',
        'level': 0,
        'default_level': 0,
        'is_on': True,
    },
}
```
Обратите внимание, что описание конфигурации логгера в реальном проекте рекомендуется вынести 
в отдельный файл - logs.py .

Ниже описываются параметры, которые могут быть указаны в словаре, являющемся значением для ключа - 
имени каждого обработчика в параметре handlers. Для каждого параметра указано, к какому типу обработчика он применим:

| Тип обработчика   | Параметр      | Тип         | Значение<br/>по умолчанию | Описание                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
|-------------------|---------------|-------------|---------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| (все)             | handler       | str         | (обязательный)            | Тип (имя класса) обработчика. Доступные типы обработчиков описаны в таблице ниже                                                                                                                                                                                                                                                                                                                                                                          |
| (все)             | name          | str         | (обязательный)            | Уникальное имя данного обработчика; должно совпадать с ключом словаря данного обработчика; может совпадать с именем логгера - пространства имен обработчиков и логгеров независимы                                                                                                                                                                                                                                                                        |
| LoggerFileHandler | file_name     | str         | (обязательный)            | Имя файла лога - для логгеров, пишущих в файл                                                                                                                                                                                                                                                                                                                                                                                                             |
| (все)             | level         | LoggerLevel | None                      | Минимальный уровень логирования (log level) для данного обработчика; если уровень логирования для любого данного сообщения ниже указанного в данном параметре, такое сообщение будет сброшено и не попадет в лог, создаваемый данным обработчиком; если этот параметр не указан, обработчику присваивается минимальный уровень логирования, равный уровню логирования по умолчанию, который установлен для библиотеки на момент инициализации обработчика |
| (все)             | default_level | LoggerLevel | None                      | Уровень логирования (log level) по умолчанию, который будет присвоен данным обработчиком сообщению, для которого не указан уровень логирования; если этот параметр не указан, обработчику присваивается уровень логирования по умолчанию, который установлен для библиотеки на момент инициализации обработчика                                                                                                                                           |
| (все)             | is_on         | bool        | True                      | Флаг, указывающий, включен ли обработчик. Если обработчик выключен, логирование сообщений в него не производится. Если данный параметр не указан, по умолчанию обработчик включен                                                                                                                                                                                                                                                                         |

В настоящее время реализованы следующие типы обработчиков (параметр handler конфигурации обработчика):

| Тип обработчика      | Описание                                                                                                                                                                                                                                                                                                                                                                                                           |
|----------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| LoggerConsoleHandler | Обработчик, используемый для вывода сообщений лога на консоль. Имеет смысл создавать несколько таких обработчиков, если, например, требуется отдельно отключать вывод какого-то типа сообщений приложения пользователя на консоль, но хочется выводить их в файл - тогда можно отдельно отключить такой обработчик или, например, повысить его уровень логирования (log level), чтобы отсекать неважные сообщения. |
| LoggerFileHandler    | Обработчик, используемый для вывода сообщений в файл. Имеет смысл создавать несколько таких обработчиков, если, например, требуется отделить сообщения отладки от сообщений, связанных с безопасностью или аудитом финансовых транзакций.                                                                                                                                                                          |

#### Функции логгеров

Логгеры - это абстрактные сущности, не связанная с конкретной средой, в которую записываются текстовые 
данные сообщений. Логгеры целесообразно создавать для различных прикладных целей, например, можно создать отдельные 
логгеры для сохранения отладочной информации и информации аудита финансовых транзакций, для логирования
действий отдельных подсистем и библиотек. Для записи сообщений в логи логгеры используют обработчики. Каждый логгер
может использовать один или одновременно несколько обработчиков, тогда как несколько логгеров могут писать 
в один и тот же обработчик, при этом количество логгеров и обработчиков не ограничено. 
Каждый логгер имеет уникальное имя, при этом имена логгеров и обработчиков 
могут совпадать - их пространства имен независимы. 
 
Как и для библиотеки в целом, для каждого логгера можно задавать уровень логирования (log level) по умолчанию, 
когда он не указан для данного сообщения пользователя. Можно, кроме того, задать для каждого логгера 
минимальный уровень логирования сообщения, которое попадет в лог, а сообщения ниже уровнем будут сброшены: 
например, если минимальный уровень - WARNING, то сообщения уровня INFORMATION в логи не попадут. 
Можно также индивидуально отключать любой логгер.

В настоящее время реализован только один класс логгера - Logger, но в дальнейшем при возникновении такой потребности 
могут быть созданы и другие классы, реализующие специфическую функциональность. Реализация этого класса 
задает конфигурацию логгера в инициализаторе __init__(), а в методе log() проверяет, 
нужно ли записывать сообщение в лог данного логгера, сравнивая уровень логирования сообщения 
с минимальным уровнем логирования логгера, и передает строку сообщения последовательно во все обработчики,
назначенные для данного логгера, например, для одновременного вывода на консоль и записи в файл.

#### Иницализация логгеров

Инициализация логгера очень похожа на аналогичную процедуру для обработчика.

Для инициализации или изменения параметров логгеров, описывающую их структуру необходимо передать в качестве 
параметра loggers в инициализатор класса LoggerFabric (см. описание выше). Эта структура представляет собой словарь 
словарей, где ключом каждой записи словаря является имя логгера, а значением - словарь, 
содержащий конфигурацию логгера - его тип, имя и параметры (см. пример ниже).      

**!ВАЖНО!** Обратите внимание, что если вызов инициализатора класса LoggerFabric осуществляется повторно и в него 
передается параметр loggers, содержащий конфигурацию уже существующего логгера, такой логгер будет создан 
заново, то есть, в новой конфигурации нельзя указать только измененные параметры, 
а необходимо указать его полную конфигурацию, принимая во внимание значения по умолчанию для опущенных необязательных 
параметров, описание которых дается в таблице ниже.

Например, приведенный ниже фрагмент кода задает конфигурацию 2 (двух) логгеров - 
'debug' и 'runtime' типа Logger; к логгеру 'debug' привязаны обработчики 'console' и 'debug_file', 
а к логгеру 'runtime' - 'console' и 'app_file', описанные выше в разделе, посвященном обработчикам:
```
# ********** Данный раздел в реальном проекте рекомендуется вынести в отдельный файл logs.py *********
loggers = {
    'debug': {
        'logger': 'Logger',
        'name': 'debug',
        'handlers': ['console', 'debug_file'],
        'level': 0,
        'default_level': 0,
        'success_on_any_log': False,
        'is_on': True,
    },
    'runtime': {
        'logger': 'Logger',
        'name': 'runtime',
        'handlers': ['console', 'app_file'],
        'level': 0,
        'default_level': 0,
        'success_on_any_log': False,
        'is_on': True,
    },
}
```
Обратите внимание, что описание конфигурации логгера в реальном проекте рекомендуется вынести 
в отдельный файл - logs.py .

Ниже описываются параметры, которые могут быть указаны в словаре, являющемся значением для ключа - 
имени каждого логгера в параметре loggers:

| Параметр           | Тип         | Значение<br/>по умолчанию | Описание                                                                                                                                                                                                                                                                                                                                                                                                                                  |
|--------------------|-------------|---------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| logger             | str         | (обязательный)            | Тип (имя класса) логгера. В настоящее время доступен только один тип - 'Logger'                                                                                                                                                                                                                                                                                                                                                           |
| name               | str         | (обязательный)            | Уникальное имя данного логгера; должно совпадать с ключом словаря данного логгера; может совпадать с именем обработчика - пространства имен обработчиков и логгеров независимы                                                                                                                                                                                                                                                            |
| handlers           | list of str | (обязательный)            | Список имен обработчиков, в которые данный логгер будет записывать сообщения; в список обработчиков логгера попадут только обработчики, которые уже существуют на момент инициализации логгера                                                                                                                                                                                                                                            |
| level              | LoggerLevel | None                      | Минимальный уровень логирования (log level) для данного логгера; если уровень логирования для любого данного сообщения ниже указанного в данном параметре, такое сообщение будет сброшено и не попадет в лог, создаваемый данным логгером; если этот параметр не указан, логгеру присваивается минимальный уровень логирования, равный уровню логирования по умолчанию, который установлен для библиотеки на момент инициализации логгера |
| default_level      | LoggerLevel | None                      | Уровень логирования (log level) по умолчанию, который будет присвоен данным логгером сообщению, для которого не указан уровень логирования; если этот параметр не указан, логгеру присваивается уровень логирования по умолчанию, который установлен для библиотеки на момент инициализации логгера                                                                                                                                       |
| success_on_any_log | bool        | False                     | Флаг, определяющий условие, когда считать логирование успешным: когда успешной была запись сообщения во все обработчики (False) или хотя бы в один обработчик (True); по умолчанию, сообщение должно успешно записаться во все обработчики, чтобы функция логирования в данный логгер завершилась успешно (вернула True)                                                                                                                  |
| is_on              | bool        | True                      | Флаг, указывающий, включен ли логгер. Если логгер выключен, логирование сообщений в него не производится. Если данный параметр не указан, по умолчанию логгер включен                                                                                                                                                                                                                                                                     |

### Логирование

После того как библиотека инициализирована, приложение пользователя может использовать интерфейс библиотеки 
для записи логов. Для этого предусмотрено два способа:
1. с использованием класса Log; этот способ удобен для логирования произвольных данных в произвольных точках 
программного кода;
2. с использованием декоратора @debug; этот способ удобен для отладки и тайминга выполнения функций и инициализаторов 
классов.

Ниже описываются оба эти способа.

#### Логирование с использованием класса Log

Чтобы начать логирование с использованием класса Log, необходимо создать объект этого класса и задать его
параметры: используемый логгер, уровень логирования по умолчанию и источник лога - произвольную строку, которую затем 
можно использовать в стороннем ПО для поиска сообщений из того или иного источника. После создания, объект класса Log
достаточно вызвать с параметром - строкой сообщения, и строка будет записана во все обработчики данного лога.

В данной таблице описаны параметры инициализатора класса Log:

| Параметр       | Тип         | Значение<br/>по умолчанию | Описание                                                                                                                                                                                                                                                                             |
|----------------|-------------|---------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| logger         | str         | None                      | Имя логгера для записи логов. Если параметр не указан, будет использован логгер по умолчанию для библиотеки. Если логгер по умолчанию для библиотеки не задан, сообщение, записываемое в данный лог, будет сброшено                                                                  |
| default_level  | LoggerLevel | None                      | Уровень логирования (log level) по умолчанию для сообщений. Если параметр не указан, будет использован уровень по умолчанию для логгера, а если логгер не задан - уровень по умолчанию для библиотеки. Для каждого индивидуального сообщения можно задавать свой уровень логирования |
| default_source | str         | None                      | Источник сообщения - произвольная строка - по умолчанию для сообщений. Если параметр не указан, будет использована пустая строка. Для каждого индивидуального сообщения можно задавать своюстроку - источник сообщения                                                               |

В примере ниже создаются два объекта Log - с использованием логгера 'debug' с минимальным уровнем логирования DEBUG и
'runtime' с минимальным уровнем INFO, источником обоих указано 'app' - так, например, можно обозначать сообщения из
основного приложения.
```
log_d = Log('debug', LoggerLevel.DEBUG, 'app')
log_r = Log('runtime', LoggerLevel.INFO, 'app')
log_d("Test debug message")
log_r("Test runtime warning message", level=LoggerLevel.WARNING, source='broken_app')
```
В соответствии с конфигурацией логгеров, указанных в примерах кода выше, сообщение лога log_d будет выведено на консоль
и записано в файл 'debug.log', а сообщение лога log_r будет выведено на консоль и записано в файл 'app.log'. Вывод
на консоль данного фрагмента кода будет выглядеть следующим образом:
```
2022-05-01T11:16:03.633540, DEBUG, debug, app, Test debug message
2022-05-01T11:16:03.634527, WARNING, runtime, broken_app, Test runtime warning message
```
Первое сообщение будет также записано в файл 'debug.log', а второе - в файл 'app.log'.

Видно, что для второго сообщения были явно указаны уровень логирования и источник. Параметры вызова объекта Log
приведены в таблице ниже:

| Параметр | Тип         | Значение<br/>по умолчанию | Описание                                                                                                                                  |
|----------|-------------|---------------------------|-------------------------------------------------------------------------------------------------------------------------------------------|
| message  | str         | (обязательный)            | Сообщение, записываемое в лог                                                                                                             |
| level    | LoggerLevel | None                      | Уровень логирования для данного сообщения. Если параметр не указан, используется значение, заданное при инициализации данного объекта Log |
| source   | str         | None                      | Источник сообщения - произвольная строка. Если параметр не указан, используется значение, заданное при инициализации данного объекта Log  |

Вызов объекта Log возвращает True, если сообщение было записано, и False в противном случае. Сообщение считается 
записанным в логгер в соответствии с установкой success_on_any_log для данного логгера (см. описание инициализации
логгера выше).

#### Логирование с использованием декоратора @debug

Декоратор debug предназначен для вывода отладочных сообщений перед и по окончании выполнения именованных фрагментов 
кода, к которым применимы декораторы. Данный декоратор можно использовать для декорирования как классов, так и функций. 
В случае декорирования класса, отладочное сообщение выводится перед началом и по окончании работы инициализатора 
объекта данного класса. В случае декорирования методов класса или отдельных функций, 
отладочное сообщение выводится перед и по окончании работы функции или метода.

Например, следующий код:
```
@debug
class Test:
    def __init__(self):
        print("Инициализация класса")

    def __call__(self, *args, **kwargs):
        print("Вызов объекта класса")

    @debug(logger='runtime', level=LoggerLevel.INFO, message_before="Вызов метода", message_after="Метод отработал")
    def do_something(self):
        print("Вызов метода объекта")

test = Test()               # Создаем объект класса Test
test()                      # Неявно вызываем метод __call__()
test.do_something()         # Явно вызываем метод 
```
выведет следующие сообщения:
```
2022-05-01T13:02:06.900947, DEBUG, debug, Test, Execution started
Инициализация класса
2022-05-01T13:02:06.901238, DEBUG, debug, Test, Execution finished
Вызов объекта класса
2022-05-01T13:02:06.901341, INFO, runtime, do_something, Вызов метода
Вызов метода объекта
2022-05-01T13:02:06.901439, INFO, runtime, do_something, Метод отработал
```
Видно, что декоратор класса выводит сообщение до и после инициализатора __init__() и не выводит сообщения при вызове
каких-либо методов класса, в том числе при неявном вызове метода __call__(). Декоратор метода выводит сообщение
до и по окончании выполнения метода. Также видно, что декоратор класса выводит в качестве источника имя класса,
а декоратор метода - имя метода.

##### Инициализация декоратора

Декоратор debug может быть использован как без параметров, так и с параметрами. В случае использования без параметров 
скобки не обязательны, и в этом случае будут использованы значения параметров по умолчанию.
Список параметров декоратора приведен в таблице ниже:

| Параметр       | Тип         | Значение по умолчанию       | Описание                                                                                                                                                                                                            |
|----------------|-------------|-----------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| logger         | str         | None                        | Имя логгера для записи логов. Если параметр не указан, будет использован логгер по умолчанию для библиотеки. Если логгер по умолчанию для библиотеки не задан, сообщение, записываемое в данный лог, будет сброшено |
| level          | LoggerLevel | LoggerLevel.DEBUG           | Уровень логирования (log level) для сообщений.                                                                                                                                                                      |
| message_before | str         | None ('Execution started')  | Сообщение по умолчанию, записываемое в лог перед началом выполнения декорированного фрагмента кода. Если параметр не указан, выводится сообщение 'Execution started'                                                |
| message_after  | str         | None ('Execution finished') | Сообщение по умолчанию, записываемое в лог по окончании выполнения декорированного фрагмента кода. Если параметр не указан, выводится сообщение 'Execution finished'                                                |

