##############################################################################
#
# Copyright (c) 2001, 2002 Zope Foundation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
"""Radio Widget Tests
"""
import doctest
import unittest

from zope.interface import Interface
from zope.interface import implementer
from zope.interface.verify import verifyClass
from zope.publisher.browser import TestRequest
from zope.schema import Choice

from zope.formlib.interfaces import IInputWidget
from zope.formlib.tests.test_browserwidget import SimpleInputWidgetTest
from zope.formlib.widgets import RadioWidget


class RadioWidgetTest(SimpleInputWidgetTest):
    """Documents and tests the radio widget.

        >>> verifyClass(IInputWidget, RadioWidget)
        True
    """

    _FieldFactory = Choice
    _WidgetFactory = RadioWidget

    def setUpContent(self, desc='', title='Foo Title'):
        class ITestContent(Interface):
            foo = self._FieldFactory(
                title=title,
                description=desc,
                values=('foo', 'bar')
            )

        @implementer(ITestContent)
        class TestObject:
            pass

        self.content = TestObject()
        field = ITestContent['foo']
        field = field.bind(self.content)
        request = TestRequest(HTTP_ACCEPT_LANGUAGE='pl')
        request.form['field.foo'] = 'Foo Value'
        self._widget = self._WidgetFactory(field, field.vocabulary, request)

    def testProperties(self):
        self.assertEqual(self._widget.cssClass, "")
        self.assertEqual(self._widget.extra, '')
        self.assertEqual(self._widget.orientation, 'vertical')

    def testRenderItem(self):
        check_list = ('type="radio"', 'id="field.bar.0"',
                      'name="field.bar"', 'value="foo"', 'Foo')
        self.verifyResult(
            self._widget.renderItem(0, 'Foo', 'foo', 'field.bar', None),
            check_list)
        check_list += ('checked="checked"',)
        self.verifyResult(
            self._widget.renderSelectedItem(
                0, 'Foo', 'foo', 'field.bar', None),
            check_list)

    def testRenderItems(self):
        check_list = ('type="radio"', 'id="field.foo.0"', 'name="field.foo"',
                      'value="bar"', 'bar', 'value="foo"', 'foo',
                      'checked="checked"')
        self.verifyResult('\n'.join(self._widget.renderItems('bar')),
                          check_list)

    def testRender(self):
        value = 'bar'
        self._widget.setRenderedValue(value)
        check_list = ('type="radio"', 'id="field.foo.0"',
                      'name="field.foo"', 'value="bar"', 'bar',
                      'value="foo"', 'foo', 'checked="checked"')
        self.verifyResult(self._widget(), check_list)

        check_list = ('type="hidden"', 'id="field.foo"',
                      'name="field.foo"', 'value="bar"')
        self.verifyResult(self._widget.hidden(), check_list)
        check_list = ('style="color: red"',) + check_list
        self._widget.extra = 'style="color: red"'
        self.verifyResult(self._widget.hidden(), check_list)

    def testHasInput(self):
        self._widget.request.form.clear()
        self.assertTrue(not self._widget.hasInput())
        self._widget.request.form['field.foo-empty-marker'] = '1'
        self.assertTrue(self._widget.hasInput())
        self._widget.request.form['field.foo'] = 'Foo Value'
        self.assertTrue(self._widget.hasInput())
        del self._widget.request.form['field.foo-empty-marker']
        self.assertTrue(self._widget.hasInput())

    def testRenderEmptyMarker(self):
        self.verifyResult(self._widget(), ('field.foo-empty-marker',))
        self._widget.setRenderedValue('bar')
        self.verifyResult(self._widget(), ('field.foo-empty-marker',))
        self._widget.setRenderedValue('not a legal value')
        self.verifyResult(self._widget(), ('field.foo-empty-marker',))


def test_suite():
    return unittest.TestSuite((
        unittest.defaultTestLoader.loadTestsFromTestCase(RadioWidgetTest),
        doctest.DocTestSuite(),
    ))
