"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('S3Origin', () => {
    test('as bucket, renders all required properties, including S3Origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = new lib_1.S3Origin({ bucket });
        origin.bind(stack, { originIndex: 0 });
        expect(origin.renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketRegionalDomainName,
            s3OriginConfig: {
                originAccessIdentity: 'origin-access-identity/cloudfront/${Token[TOKEN.69]}',
            },
        });
    });
    test('as bucket, creates an OriginAccessIdentity and grants read permissions on the bucket', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = new lib_1.S3Origin({ bucket });
        new lib_1.Distribution(stack, 'Dist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
            CloudFrontOriginAccessIdentityConfig: {
                Comment: 'Allows CloudFront to reach the bucket',
            },
        });
        expect(stack).toHaveResourceLike('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Statement: [{
                        Principal: {
                            CanonicalUser: { 'Fn::GetAtt': ['DistS3Origin1C4519663', 'S3CanonicalUserId'] },
                        },
                    }],
            },
        });
    });
});
describe('HttpOrigin', () => {
    test('renders a minimal example with required props', () => {
        const origin = new lib_1.HttpOrigin('www.example.com');
        origin.bind(stack, { originIndex: 0 });
        expect(origin.renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: 'www.example.com',
            customOriginConfig: {
                originProtocolPolicy: 'https-only',
            },
        });
    });
    test('renders an example with all available props', () => {
        const origin = new lib_1.HttpOrigin('www.example.com', {
            originPath: '/app',
            connectionTimeout: core_1.Duration.seconds(5),
            connectionAttempts: 2,
            customHeaders: { AUTH: 'NONE' },
            protocolPolicy: lib_1.OriginProtocolPolicy.MATCH_VIEWER,
            httpPort: 8080,
            httpsPort: 8443,
            readTimeout: core_1.Duration.seconds(45),
            keepaliveTimeout: core_1.Duration.seconds(3),
        });
        origin.bind(stack, { originIndex: 0 });
        expect(origin.renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: 'www.example.com',
            originPath: '/app',
            connectionTimeout: 5,
            connectionAttempts: 2,
            originCustomHeaders: [{
                    headerName: 'AUTH',
                    headerValue: 'NONE',
                }],
            customOriginConfig: {
                originProtocolPolicy: 'match-viewer',
                httpPort: 8080,
                httpsPort: 8443,
                originReadTimeout: 45,
                originKeepaliveTimeout: 3,
            },
        });
    });
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(60.5),
        core_1.Duration.seconds(61),
        core_1.Duration.minutes(5),
    ])('validates readTimeout is an integer between 1 and 60 seconds', (readTimeout) => {
        expect(() => {
            new lib_1.HttpOrigin('www.example.com', {
                readTimeout,
            });
        }).toThrow(`readTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${readTimeout.toSeconds()}.`);
    });
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(60.5),
        core_1.Duration.seconds(61),
        core_1.Duration.minutes(5),
    ])('validates keepaliveTimeout is an integer between 1 and 60 seconds', (keepaliveTimeout) => {
        expect(() => {
            new lib_1.HttpOrigin('www.example.com', {
                keepaliveTimeout,
            });
        }).toThrow(`keepaliveTimeout: Must be an int between 1 and 60 seconds (inclusive); received ${keepaliveTimeout.toSeconds()}.`);
    });
});
;
describe('Origin', () => {
    test.each([
        core_1.Duration.seconds(0),
        core_1.Duration.seconds(0.5),
        core_1.Duration.seconds(10.5),
        core_1.Duration.seconds(11),
        core_1.Duration.minutes(5),
    ])('validates connectionTimeout is an int between 1 and 10 seconds', (connectionTimeout) => {
        expect(() => {
            new TestOrigin('www.example.com', {
                connectionTimeout,
            });
        }).toThrow(`connectionTimeout: Must be an int between 1 and 10 seconds (inclusive); received ${connectionTimeout.toSeconds()}.`);
    });
    test.each([-0.5, 0.5, 1.5, 4])('validates connectionAttempts is an int between 1 and 3', (connectionAttempts) => {
        expect(() => {
            new TestOrigin('www.example.com', {
                connectionAttempts,
            });
        }).toThrow(`connectionAttempts: Must be an int between 1 and 3 (inclusive); received ${connectionAttempts}.`);
    });
    test.each(['api', '/api', '/api/', 'api/'])('enforces that originPath starts but does not end, with a /', (originPath) => {
        const origin = new TestOrigin('www.example.com', {
            originPath,
        });
        origin.bind(stack, { originIndex: 0 });
        expect(origin.renderOrigin().originPath).toEqual('/api');
    });
});
/** Used for testing common Origin functionality */
class TestOrigin extends lib_1.Origin {
    constructor(domainName, props = {}) { super(domainName, props); }
    renderS3OriginConfig() {
        return { originAccessIdentity: 'origin-access-identity/cloudfront/MyOAIName' };
    }
}
//# sourceMappingURL=data:application/json;base64,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