"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'Distribution with custom origin failover'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'ADistribution', {
            originConfigs: [
                {
                    originHeaders: {
                        'X-Custom-Header': 'somevalue',
                    },
                    customOriginSource: {
                        domainName: 'myorigin.com',
                    },
                    failoverCustomOriginSource: {
                        domainName: 'myoriginfallback.com',
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                OriginGroups: {
                    Items: [
                        {
                            FailoverCriteria: {
                                StatusCodes: {
                                    Items: [
                                        500,
                                        502,
                                        503,
                                        504,
                                    ],
                                    Quantity: 4,
                                },
                            },
                            Id: 'OriginGroup1',
                            Members: {
                                Items: [
                                    {
                                        OriginId: 'origin1',
                                    },
                                    {
                                        OriginId: 'originSecondary1',
                                    },
                                ],
                                Quantity: 2,
                            },
                        },
                    ],
                    Quantity: 1,
                },
                Origins: [
                    {
                        CustomOriginConfig: {
                            HTTPPort: 80,
                            HTTPSPort: 443,
                            OriginKeepaliveTimeout: 5,
                            OriginProtocolPolicy: 'https-only',
                            OriginReadTimeout: 30,
                            OriginSSLProtocols: [
                                'TLSv1.2',
                            ],
                        },
                        DomainName: 'myoriginfallback.com',
                        Id: 'originSecondary1',
                        OriginCustomHeaders: [
                            {
                                HeaderName: 'X-Custom-Header',
                                HeaderValue: 'somevalue',
                            },
                        ],
                    },
                    {
                        CustomOriginConfig: {
                            HTTPPort: 80,
                            HTTPSPort: 443,
                            OriginKeepaliveTimeout: 5,
                            OriginProtocolPolicy: 'https-only',
                            OriginReadTimeout: 30,
                            OriginSSLProtocols: [
                                'TLSv1.2',
                            ],
                        },
                        DomainName: 'myorigin.com',
                        Id: 'origin1',
                        OriginCustomHeaders: [
                            {
                                HeaderName: 'X-Custom-Header',
                                HeaderValue: 'somevalue',
                            },
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    'Distribution with s3 origin failover'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'ADistribution', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: s3.Bucket.fromBucketName(stack, 'aBucket', 'myoriginbucket'),
                        originPath: '/',
                        originHeaders: {
                            myHeader: '42',
                        },
                    },
                    failoverS3OriginSource: {
                        s3BucketSource: s3.Bucket.fromBucketName(stack, 'aBucketFallback', 'myoriginbucketfallback'),
                        originPath: '/somwhere',
                        originHeaders: {
                            myHeader2: '21',
                        },
                    },
                    failoverCriteriaStatusCodes: [lib_1.FailoverStatusCode.INTERNAL_SERVER_ERROR],
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                OriginGroups: {
                    Items: [
                        {
                            FailoverCriteria: {
                                StatusCodes: {
                                    Items: [
                                        500,
                                    ],
                                    Quantity: 1,
                                },
                            },
                            Id: 'OriginGroup1',
                            Members: {
                                Items: [
                                    {
                                        OriginId: 'origin1',
                                    },
                                    {
                                        OriginId: 'originSecondary1',
                                    },
                                ],
                                Quantity: 2,
                            },
                        },
                    ],
                    Quantity: 1,
                },
                Origins: [
                    {
                        DomainName: {
                            'Fn::Join': [
                                '',
                                [
                                    'myoriginbucketfallback.s3.',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                ],
                            ],
                        },
                        Id: 'originSecondary1',
                        OriginCustomHeaders: [
                            {
                                HeaderName: 'myHeader2',
                                HeaderValue: '21',
                            },
                        ],
                        OriginPath: '/somwhere',
                        S3OriginConfig: {},
                    },
                    {
                        DomainName: {
                            'Fn::Join': [
                                '',
                                [
                                    'myoriginbucket.s3.',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                ],
                            ],
                        },
                        Id: 'origin1',
                        OriginCustomHeaders: [
                            {
                                HeaderName: 'myHeader',
                                HeaderValue: '42',
                            },
                        ],
                        OriginPath: '/',
                        S3OriginConfig: {},
                    },
                ],
            },
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JpZ2luLWdyb3Vwcy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3JpZ2luLWdyb3Vwcy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNENBQTJEO0FBQzNELHNDQUFzQztBQUN0QyxxQ0FBcUM7QUFDckMsaURBQW1EO0FBQ25ELGdDQUF1RTtBQUV2RSw0QkFBWSxDQUFDO0lBQ1gsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixJQUFJLCtCQUF5QixDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDcEQsYUFBYSxFQUFFO2dCQUNiO29CQUNFLGFBQWEsRUFBRTt3QkFDYixpQkFBaUIsRUFBRSxXQUFXO3FCQUMvQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLGNBQWM7cUJBQzNCO29CQUNELDBCQUEwQixFQUFFO3dCQUMxQixVQUFVLEVBQUUsc0JBQXNCO3FCQUNuQztvQkFDRCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsaUJBQWlCLEVBQUUsSUFBSTt5QkFDeEI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQ2QseUJBQWdCLENBQUMsK0JBQStCLEVBQUU7WUFDaEQsa0JBQWtCLEVBQUU7Z0JBQ2xCLFlBQVksRUFBRTtvQkFDWixLQUFLLEVBQUU7d0JBQ0w7NEJBQ0UsZ0JBQWdCLEVBQUU7Z0NBQ2hCLFdBQVcsRUFBRTtvQ0FDWCxLQUFLLEVBQUU7d0NBQ0wsR0FBRzt3Q0FDSCxHQUFHO3dDQUNILEdBQUc7d0NBQ0gsR0FBRztxQ0FDSjtvQ0FDRCxRQUFRLEVBQUUsQ0FBQztpQ0FDWjs2QkFDRjs0QkFDRCxFQUFFLEVBQUUsY0FBYzs0QkFDbEIsT0FBTyxFQUFFO2dDQUNQLEtBQUssRUFBRTtvQ0FDTDt3Q0FDRSxRQUFRLEVBQUUsU0FBUztxQ0FDcEI7b0NBQ0Q7d0NBQ0UsUUFBUSxFQUFFLGtCQUFrQjtxQ0FDN0I7aUNBQ0Y7Z0NBQ0QsUUFBUSxFQUFFLENBQUM7NkJBQ1o7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsUUFBUSxFQUFFLENBQUM7aUJBQ1o7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQO3dCQUNFLGtCQUFrQixFQUFFOzRCQUNsQixRQUFRLEVBQUUsRUFBRTs0QkFDWixTQUFTLEVBQUUsR0FBRzs0QkFDZCxzQkFBc0IsRUFBRSxDQUFDOzRCQUN6QixvQkFBb0IsRUFBRSxZQUFZOzRCQUNsQyxpQkFBaUIsRUFBRSxFQUFFOzRCQUNyQixrQkFBa0IsRUFBRTtnQ0FDbEIsU0FBUzs2QkFDVjt5QkFDRjt3QkFDRCxVQUFVLEVBQUUsc0JBQXNCO3dCQUNsQyxFQUFFLEVBQUUsa0JBQWtCO3dCQUN0QixtQkFBbUIsRUFBRTs0QkFDbkI7Z0NBQ0UsVUFBVSxFQUFFLGlCQUFpQjtnQ0FDN0IsV0FBVyxFQUFFLFdBQVc7NkJBQ3pCO3lCQUNGO3FCQUNGO29CQUNEO3dCQUNFLGtCQUFrQixFQUFFOzRCQUNsQixRQUFRLEVBQUUsRUFBRTs0QkFDWixTQUFTLEVBQUUsR0FBRzs0QkFDZCxzQkFBc0IsRUFBRSxDQUFDOzRCQUN6QixvQkFBb0IsRUFBRSxZQUFZOzRCQUNsQyxpQkFBaUIsRUFBRSxFQUFFOzRCQUNyQixrQkFBa0IsRUFBRTtnQ0FDbEIsU0FBUzs2QkFDVjt5QkFDRjt3QkFDRCxVQUFVLEVBQUUsY0FBYzt3QkFDMUIsRUFBRSxFQUFFLFNBQVM7d0JBQ2IsbUJBQW1CLEVBQUU7NEJBQ25CO2dDQUNFLFVBQVUsRUFBRSxpQkFBaUI7Z0NBQzdCLFdBQVcsRUFBRSxXQUFXOzZCQUN6Qjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0NBQXNDLENBQUMsSUFBVTtRQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixJQUFJLCtCQUF5QixDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDcEQsYUFBYSxFQUFFO2dCQUNiO29CQUNFLGNBQWMsRUFBRTt3QkFDZCxjQUFjLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQzt3QkFDNUUsVUFBVSxFQUFFLEdBQUc7d0JBQ2YsYUFBYSxFQUFFOzRCQUNiLFFBQVEsRUFBRSxJQUFJO3lCQUNmO3FCQUNGO29CQUNELHNCQUFzQixFQUFFO3dCQUN0QixjQUFjLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFLHdCQUF3QixDQUFDO3dCQUM1RixVQUFVLEVBQUUsV0FBVzt3QkFDdkIsYUFBYSxFQUFFOzRCQUNiLFNBQVMsRUFBRSxJQUFJO3lCQUNoQjtxQkFDRjtvQkFDRCwyQkFBMkIsRUFBRSxDQUFDLHdCQUFrQixDQUFDLHFCQUFxQixDQUFDO29CQUN2RSxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsaUJBQWlCLEVBQUUsSUFBSTt5QkFDeEI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQ2QseUJBQWdCLENBQUMsK0JBQStCLEVBQUU7WUFDaEQsa0JBQWtCLEVBQUU7Z0JBQ2xCLFlBQVksRUFBRTtvQkFDWixLQUFLLEVBQUU7d0JBQ0w7NEJBQ0UsZ0JBQWdCLEVBQUU7Z0NBQ2hCLFdBQVcsRUFBRTtvQ0FDWCxLQUFLLEVBQUU7d0NBQ0wsR0FBRztxQ0FDSjtvQ0FDRCxRQUFRLEVBQUUsQ0FBQztpQ0FDWjs2QkFDRjs0QkFDRCxFQUFFLEVBQUUsY0FBYzs0QkFDbEIsT0FBTyxFQUFFO2dDQUNQLEtBQUssRUFBRTtvQ0FDTDt3Q0FDRSxRQUFRLEVBQUUsU0FBUztxQ0FDcEI7b0NBQ0Q7d0NBQ0UsUUFBUSxFQUFFLGtCQUFrQjtxQ0FDN0I7aUNBQ0Y7Z0NBQ0QsUUFBUSxFQUFFLENBQUM7NkJBQ1o7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsUUFBUSxFQUFFLENBQUM7aUJBQ1o7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixVQUFVLEVBQUU7Z0NBQ1YsRUFBRTtnQ0FDRjtvQ0FDRSw0QkFBNEI7b0NBQzVCO3dDQUNFLEdBQUcsRUFBRSxhQUFhO3FDQUNuQjtvQ0FDRCxHQUFHO29DQUNIO3dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUNBQ3RCO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNELEVBQUUsRUFBRSxrQkFBa0I7d0JBQ3RCLG1CQUFtQixFQUFFOzRCQUNuQjtnQ0FDRSxVQUFVLEVBQUUsV0FBVztnQ0FDdkIsV0FBVyxFQUFFLElBQUk7NkJBQ2xCO3lCQUNGO3dCQUNELFVBQVUsRUFBRSxXQUFXO3dCQUN2QixjQUFjLEVBQUUsRUFBRTtxQkFDbkI7b0JBQ0Q7d0JBQ0UsVUFBVSxFQUFFOzRCQUNWLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGO29DQUNFLG9CQUFvQjtvQ0FDcEI7d0NBQ0UsR0FBRyxFQUFFLGFBQWE7cUNBQ25CO29DQUNELEdBQUc7b0NBQ0g7d0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQ0FDdEI7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsRUFBRSxFQUFFLFNBQVM7d0JBQ2IsbUJBQW1CLEVBQUU7NEJBQ25CO2dDQUNFLFVBQVUsRUFBRSxVQUFVO2dDQUN0QixXQUFXLEVBQUUsSUFBSTs2QkFDbEI7eUJBQ0Y7d0JBQ0QsVUFBVSxFQUFFLEdBQUc7d0JBQ2YsY0FBYyxFQUFFLEVBQUU7cUJBQ25CO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZUxpa2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IG5vZGV1bml0U2hpbSwgVGVzdCB9IGZyb20gJ25vZGV1bml0LXNoaW0nO1xuaW1wb3J0IHsgQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiwgRmFpbG92ZXJTdGF0dXNDb2RlIH0gZnJvbSAnLi4vbGliJztcblxubm9kZXVuaXRTaGltKHtcbiAgJ0Rpc3RyaWJ1dGlvbiB3aXRoIGN1c3RvbSBvcmlnaW4gZmFpbG92ZXInKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHN0YWNrLCAnQURpc3RyaWJ1dGlvbicsIHtcbiAgICAgIG9yaWdpbkNvbmZpZ3M6IFtcbiAgICAgICAge1xuICAgICAgICAgIG9yaWdpbkhlYWRlcnM6IHtcbiAgICAgICAgICAgICdYLUN1c3RvbS1IZWFkZXInOiAnc29tZXZhbHVlJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGN1c3RvbU9yaWdpblNvdXJjZToge1xuICAgICAgICAgICAgZG9tYWluTmFtZTogJ215b3JpZ2luLmNvbScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBmYWlsb3ZlckN1c3RvbU9yaWdpblNvdXJjZToge1xuICAgICAgICAgICAgZG9tYWluTmFtZTogJ215b3JpZ2luZmFsbGJhY2suY29tJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGJlaGF2aW9yczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpc0RlZmF1bHRCZWhhdmlvcjogdHJ1ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKFxuICAgICAgaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICAgIE9yaWdpbkdyb3Vwczoge1xuICAgICAgICAgICAgSXRlbXM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIEZhaWxvdmVyQ3JpdGVyaWE6IHtcbiAgICAgICAgICAgICAgICAgIFN0YXR1c0NvZGVzOiB7XG4gICAgICAgICAgICAgICAgICAgIEl0ZW1zOiBbXG4gICAgICAgICAgICAgICAgICAgICAgNTAwLFxuICAgICAgICAgICAgICAgICAgICAgIDUwMixcbiAgICAgICAgICAgICAgICAgICAgICA1MDMsXG4gICAgICAgICAgICAgICAgICAgICAgNTA0LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICBRdWFudGl0eTogNCxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBJZDogJ09yaWdpbkdyb3VwMScsXG4gICAgICAgICAgICAgICAgTWVtYmVyczoge1xuICAgICAgICAgICAgICAgICAgSXRlbXM6IFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIE9yaWdpbklkOiAnb3JpZ2luMScsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBPcmlnaW5JZDogJ29yaWdpblNlY29uZGFyeTEnLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIFF1YW50aXR5OiAyLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUXVhbnRpdHk6IDEsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBPcmlnaW5zOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgICAgIEhUVFBQb3J0OiA4MCxcbiAgICAgICAgICAgICAgICBIVFRQU1BvcnQ6IDQ0MyxcbiAgICAgICAgICAgICAgICBPcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiA1LFxuICAgICAgICAgICAgICAgIE9yaWdpblByb3RvY29sUG9saWN5OiAnaHR0cHMtb25seScsXG4gICAgICAgICAgICAgICAgT3JpZ2luUmVhZFRpbWVvdXQ6IDMwLFxuICAgICAgICAgICAgICAgIE9yaWdpblNTTFByb3RvY29sczogW1xuICAgICAgICAgICAgICAgICAgJ1RMU3YxLjInLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIERvbWFpbk5hbWU6ICdteW9yaWdpbmZhbGxiYWNrLmNvbScsXG4gICAgICAgICAgICAgIElkOiAnb3JpZ2luU2Vjb25kYXJ5MScsXG4gICAgICAgICAgICAgIE9yaWdpbkN1c3RvbUhlYWRlcnM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBIZWFkZXJOYW1lOiAnWC1DdXN0b20tSGVhZGVyJyxcbiAgICAgICAgICAgICAgICAgIEhlYWRlclZhbHVlOiAnc29tZXZhbHVlJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgSFRUUFBvcnQ6IDgwLFxuICAgICAgICAgICAgICAgIEhUVFBTUG9ydDogNDQzLFxuICAgICAgICAgICAgICAgIE9yaWdpbktlZXBhbGl2ZVRpbWVvdXQ6IDUsXG4gICAgICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgICAgICAgICBPcmlnaW5SZWFkVGltZW91dDogMzAsXG4gICAgICAgICAgICAgICAgT3JpZ2luU1NMUHJvdG9jb2xzOiBbXG4gICAgICAgICAgICAgICAgICAnVExTdjEuMicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgRG9tYWluTmFtZTogJ215b3JpZ2luLmNvbScsXG4gICAgICAgICAgICAgIElkOiAnb3JpZ2luMScsXG4gICAgICAgICAgICAgIE9yaWdpbkN1c3RvbUhlYWRlcnM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBIZWFkZXJOYW1lOiAnWC1DdXN0b20tSGVhZGVyJyxcbiAgICAgICAgICAgICAgICAgIEhlYWRlclZhbHVlOiAnc29tZXZhbHVlJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdEaXN0cmlidXRpb24gd2l0aCBzMyBvcmlnaW4gZmFpbG92ZXInKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHN0YWNrLCAnQURpc3RyaWJ1dGlvbicsIHtcbiAgICAgIG9yaWdpbkNvbmZpZ3M6IFtcbiAgICAgICAge1xuICAgICAgICAgIHMzT3JpZ2luU291cmNlOiB7XG4gICAgICAgICAgICBzM0J1Y2tldFNvdXJjZTogczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHN0YWNrLCAnYUJ1Y2tldCcsICdteW9yaWdpbmJ1Y2tldCcpLFxuICAgICAgICAgICAgb3JpZ2luUGF0aDogJy8nLFxuICAgICAgICAgICAgb3JpZ2luSGVhZGVyczoge1xuICAgICAgICAgICAgICBteUhlYWRlcjogJzQyJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBmYWlsb3ZlclMzT3JpZ2luU291cmNlOiB7XG4gICAgICAgICAgICBzM0J1Y2tldFNvdXJjZTogczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHN0YWNrLCAnYUJ1Y2tldEZhbGxiYWNrJywgJ215b3JpZ2luYnVja2V0ZmFsbGJhY2snKSxcbiAgICAgICAgICAgIG9yaWdpblBhdGg6ICcvc29td2hlcmUnLFxuICAgICAgICAgICAgb3JpZ2luSGVhZGVyczoge1xuICAgICAgICAgICAgICBteUhlYWRlcjI6ICcyMScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgZmFpbG92ZXJDcml0ZXJpYVN0YXR1c0NvZGVzOiBbRmFpbG92ZXJTdGF0dXNDb2RlLklOVEVSTkFMX1NFUlZFUl9FUlJPUl0sXG4gICAgICAgICAgYmVoYXZpb3JzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlzRGVmYXVsdEJlaGF2aW9yOiB0cnVlLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG8oXG4gICAgICBoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICAgICAgT3JpZ2luR3JvdXBzOiB7XG4gICAgICAgICAgICBJdGVtczogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgRmFpbG92ZXJDcml0ZXJpYToge1xuICAgICAgICAgICAgICAgICAgU3RhdHVzQ29kZXM6IHtcbiAgICAgICAgICAgICAgICAgICAgSXRlbXM6IFtcbiAgICAgICAgICAgICAgICAgICAgICA1MDAsXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICAgIFF1YW50aXR5OiAxLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIElkOiAnT3JpZ2luR3JvdXAxJyxcbiAgICAgICAgICAgICAgICBNZW1iZXJzOiB7XG4gICAgICAgICAgICAgICAgICBJdGVtczogW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgT3JpZ2luSWQ6ICdvcmlnaW4xJyxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIE9yaWdpbklkOiAnb3JpZ2luU2Vjb25kYXJ5MScsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgUXVhbnRpdHk6IDIsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBRdWFudGl0eTogMSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgRG9tYWluTmFtZToge1xuICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICAnbXlvcmlnaW5idWNrZXRmYWxsYmFjay5zMy4nLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpSZWdpb24nLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnLicsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlVSTFN1ZmZpeCcsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIElkOiAnb3JpZ2luU2Vjb25kYXJ5MScsXG4gICAgICAgICAgICAgIE9yaWdpbkN1c3RvbUhlYWRlcnM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBIZWFkZXJOYW1lOiAnbXlIZWFkZXIyJyxcbiAgICAgICAgICAgICAgICAgIEhlYWRlclZhbHVlOiAnMjEnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIE9yaWdpblBhdGg6ICcvc29td2hlcmUnLFxuICAgICAgICAgICAgICBTM09yaWdpbkNvbmZpZzoge30sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBEb21haW5OYW1lOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICdteW9yaWdpbmJ1Y2tldC5zMy4nLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpSZWdpb24nLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnLicsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlVSTFN1ZmZpeCcsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIElkOiAnb3JpZ2luMScsXG4gICAgICAgICAgICAgIE9yaWdpbkN1c3RvbUhlYWRlcnM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBIZWFkZXJOYW1lOiAnbXlIZWFkZXInLFxuICAgICAgICAgICAgICAgICAgSGVhZGVyVmFsdWU6ICc0MicsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgT3JpZ2luUGF0aDogJy8nLFxuICAgICAgICAgICAgICBTM09yaWdpbkNvbmZpZzoge30sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59KTtcbiJdfQ==