"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = defaultS3Origin();
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            Origins: [{
                    DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                    Id: 'StackMyDistOrigin1D6D5E535',
                    S3OriginConfig: {
                        OriginAccessIdentity: { 'Fn::Join': ['',
                                ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                            ] },
                    },
                }],
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = defaultS3Origin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = defaultS3Origin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = defaultS3Origin();
        const bucket2 = new s3.Bucket(stack, 'Bucket2');
        const origin2 = new lib_1.S3Origin({ bucket: bucket2 });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    },
                    {
                        DomainName: { 'Fn::GetAtt': ['Bucket25524B414', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin20B96F3AD',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin2E88F08BB' }],
                                ] },
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = defaultS3Origin();
        const bucket2 = new s3.Bucket(stack, 'Bucket2');
        const origin2 = new lib_1.S3Origin({ bucket: bucket2 });
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    },
                    {
                        DomainName: { 'Fn::GetAtt': ['Bucket25524B414', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin20B96F3AD',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin2E88F08BB' }],
                                ] },
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = defaultS3Origin();
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region and the certificate you provided is in eu-west-1./);
    });
    test('adding a certificate renders the correct ViewerCertificate property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: defaultS3Origin() },
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                    SslSupportMethod: 'sni-only',
                    MinimumProtocolVersion: 'TLSv1.2_2018',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = defaultS3Origin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = defaultS3Origin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = defaultS3Origin();
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
describe('with Lambda@Edge functions', () => {
    let lambdaFunction;
    let origin;
    beforeEach(() => {
        lambdaFunction = new lambda.Function(stack, 'Function', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whatever'),
            handler: 'index.handler',
        });
        origin = defaultS3Origin();
    });
    test('can add an edge lambdas to the default behavior', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'FunctionCurrentVersion4E2B22619c0305f954e58f25575548280c0a3629',
                            },
                        },
                    ],
                },
            },
        });
    });
    test('can add an edge lambdas to additional behaviors', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'images/*': {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.currentVersion,
                            eventType: lib_1.LambdaEdgeEventType.VIEWER_REQUEST,
                        },
                    ],
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CacheBehaviors: [
                    {
                        PathPattern: 'images/*',
                        LambdaFunctionAssociations: [
                            {
                                EventType: 'viewer-request',
                                LambdaFunctionARN: {
                                    Ref: 'FunctionCurrentVersion4E2B22619c0305f954e58f25575548280c0a3629',
                                },
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('fails creation when attempting to add the $LATEST function version as an edge Lambda to the default behavior', () => {
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.latestVersion,
                            eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        },
                    ],
                },
            });
        }).toThrow(/\$LATEST function version cannot be used for Lambda@Edge/);
    });
});
test('price class is included if provided', () => {
    const origin = defaultS3Origin();
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
function defaultS3Origin() {
    return new lib_1.S3Origin({ bucket: new s3.Bucket(stack, 'Bucket') });
}
//# sourceMappingURL=data:application/json;base64,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