"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("@aws-cdk/core");
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     */
    renderOrigin() {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            originPath: this.originPath,
            connectionAttempts: this.connectionAttempts,
            connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            originCustomHeaders: this.renderCustomHeaders(),
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props.bucket.bucketRegionalDomainName, props);
        this.bucket = props.bucket;
    }
    bind(scope, options) {
        super.bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(domainName, props = {}) {
        var _a, _b;
        super(domainName, props);
        this.props = props;
        validateIntInRangeOrUndefined('readTimeout', 1, 60, (_a = props.readTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('keepaliveTimeout', 1, 60, (_b = props.keepaliveTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
    }
    renderCustomOriginConfig() {
        var _a, _b, _c;
        return {
            originProtocolPolicy: (_a = this.props.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: this.props.httpPort,
            httpsPort: this.props.httpsPort,
            originReadTimeout: (_b = this.props.readTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            originKeepaliveTimeout: (_c = this.props.keepaliveTimeout) === null || _c === void 0 ? void 0 : _c.toSeconds(),
        };
    }
}
exports.HttpOrigin = HttpOrigin;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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