"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.origins = new Set();
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        this.defaultBehavior = new cache_behavior_1.CacheBehavior({ pathPattern: '*', ...props.defaultBehavior });
        this.addOrigin(this.defaultBehavior.origin);
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior({ pathPattern, origin, ...behaviorOptions }));
        this.addOrigin(origin);
    }
    addOrigin(origin) {
        if (!this.origins.has(origin)) {
            this.origins.add(origin);
            origin.bind(this, { originIndex: this.origins.size });
        }
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.origins.forEach(origin => renderedOrigins.push(origin.renderOrigin()));
        return renderedOrigins;
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2018,
        };
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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