"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeType = exports.ElasticacheAutoMonitor = void 0;
const path = require("path");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sns_sub = require("@aws-cdk/aws-sns-subscriptions");
const actions = require("@aws-cdk/aws-cloudwatch-actions");
const cpu_monitor_1 = require("./cpu-monitor");
const swap_usage_monitor_1 = require("./swap-usage-monitor");
const evictions_monitor_1 = require("./evictions-monitor");
const current_connections_monitor_1 = require("./current-connections-monitor");
const freeable_memory_monitor_1 = require("./freeable-memory-monitor");
/**
 * ElasticacheAutoMonitor allows you to send email, sms, slack, or trigger aws sns topic when an alarm occurs.
 * You will get the following monitoring:
 *  1. Cpu Monitor: Should be less than 90%. (See below reference)
 *  2. SwapUsage Monitor: Should be less than 50M.
 *  3. Evictions Monitor: Should not have evictions value.
 *  4. CurrConnections Monitor: According to your business needs, default every 1 vcup is equal to 200 connections.
 *  5. FreeableMemory Monitor: Not less than 10%
 *
 * Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html
 */
class ElasticacheAutoMonitor extends cdk.Construct {
    static setUpWithLambda(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-lambda');
        topic.addSubscription(new sns_sub.LambdaSubscription(props.lambda));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSlack(scope, cacheClusterId, props) {
        let channel = props.channel || 'cloudwatch-alarm';
        let username = props.username || 'Webhookbot';
        let icon_emoji = props.iconEmoji || ':scream:';
        const fn = new lambda.Function(scope, 'alarm-' + cacheClusterId + '-to-slack', {
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(path.join(__dirname, '../asset/lambda')),
            environment: {
                SLACK_WEBHOOK_URL: props.webHookUrl,
                CHANNEL: channel,
                USERNAME: username,
                ICON_EMOJI: icon_emoji,
            },
        });
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-slack');
        topic.addSubscription(new sns_sub.LambdaSubscription(fn));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSms(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-sms-topic');
        for (let phone in props.phones) {
            topic.addSubscription(new sns_sub.SmsSubscription(phone));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setup(scope, cacheClusterId, topic, props) {
        // bind default and props
        const nodeType = props.nodeType || NodeType.DEFAULT;
        const _props = {
            nodeType,
            currConnectionsPeriod: cdk.Duration.minutes(1),
            currConnectionsThreshold: nodeType.vcupCount * 200,
            ...props,
        };
        cpu_monitor_1.CpuMonitor.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
        swap_usage_monitor_1.SwapUsageMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        evictions_monitor_1.EvictionsMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        current_connections_monitor_1.CurrConnectionsMomiton.setup(scope, cacheClusterId, _props.currConnectionsPeriod, _props.currConnectionsThreshold).addAlarmAction(new actions.SnsAction(topic));
        freeable_memory_monitor_1.FreeableMemoryMomiton.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
    }
}
exports.ElasticacheAutoMonitor = ElasticacheAutoMonitor;
class NodeType {
    constructor(name, vcupCount, memory) {
        this.name = name;
        this.vcupCount = vcupCount;
        this.memory = memory;
    }
}
exports.NodeType = NodeType;
NodeType.T2_MIRCO = new NodeType('cache.t2.mirco', 1, 0.555 * 1024 * 1024 * 1024);
NodeType.T2_SMALL = new NodeType('cache.t2.small', 1, 1.55 * 1024 * 1024 * 1024);
NodeType.T2_MEDIUM = new NodeType('cache.t2.medium', 2, 3.22 * 1024 * 1024 * 1024);
NodeType.T3_MICRO = new NodeType('cache.t3.micro', 2, 0.5 * 1024 * 1024 * 1024);
NodeType.T3_SMALL = new NodeType('cache.t3.small', 2, 1.37 * 1024 * 1024 * 1024);
NodeType.T3_MEDIUM = new NodeType('cache.t3.medium', 2, 3.09 * 1024 * 1024 * 1024);
NodeType.M4_LARGE = new NodeType('cache.m4.large', 2, 6.42 * 1024 * 1024 * 1024);
NodeType.M4_XLARGE = new NodeType('cache.m4.xlarge', 4, 14.28 * 1024 * 1024 * 1024);
NodeType.M4_2XLARGE = new NodeType('cache.m4.2xlarge', 8, 29.70 * 1024 * 1024 * 1024);
NodeType.M4_4XLARGE = new NodeType('cache.m4.4xlarge', 16, 60.78 * 1024 * 1024 * 1024);
NodeType.M4_10XLARGE = new NodeType('cache.m4.10xlarge', 40, 154.64 * 1024 * 1024 * 1024);
NodeType.M5_LARGE = new NodeType('cache.m5.large', 2, 6.38 * 1024 * 1024 * 1024);
NodeType.M5_XLARGE = new NodeType('cache.m5.xlarge', 4, 12.93 * 1024 * 1024 * 1024);
NodeType.M5_2XLARGE = new NodeType('cache.m5.2xlarge', 8, 26.04 * 1024 * 1024 * 1024);
NodeType.M5_4XLARGE = new NodeType('cache.m5.4xlarge', 16, 52.26 * 1024 * 1024 * 1024);
NodeType.M5_12XLARGE = new NodeType('cache.m5.12xlarge', 48, 157.12 * 1024 * 1024 * 1024);
NodeType.M5_24XLARGE = new NodeType('cache.m5.12xlarge', 96, 314.32 * 1024 * 1024 * 1024);
NodeType.R4_LARGE = new NodeType('cache.r4.large', 2, 12.3 * 1024 * 1024 * 1024);
NodeType.R4_XLARGE = new NodeType('cache.r4.xlarge', 4, 25.05 * 1024 * 1024 * 1024);
NodeType.R4_2XLARGE = new NodeType('cache.r4.2xlarge', 8, 50.47 * 1024 * 1024 * 1024);
NodeType.R4_4XLARGE = new NodeType('cache.r4.4xlarge', 16, 101.38 * 1024 * 1024 * 1024);
NodeType.R4_8XLARGE = new NodeType('cache.r4.8xlarge', 32, 203.26 * 1024 * 1024 * 1024);
NodeType.R4_16XLARGE = new NodeType('cache.r4.16xlarge', 64, 407 * 1024 * 1024 * 1024);
NodeType.R5_LARGE = new NodeType('cache.r5.large', 2, 13.07 * 1024 * 1024 * 1024);
NodeType.R5_XLARGE = new NodeType('cache.r5.xlarge', 4, 26.32 * 1024 * 1024 * 1024);
NodeType.R5_2XLARGE = new NodeType('cache.r5.2xlarge', 8, 52.82 * 1024 * 1024 * 1024);
NodeType.R5_4XLARGE = new NodeType('cache.r5.4xlarge', 16, 105.81 * 1024 * 1024 * 1024);
NodeType.R5_12XLARGE = new NodeType('cache.r5.12xlarge', 48, 317.77 * 1024 * 1024 * 1024);
NodeType.R5_24LARGE = new NodeType('cache.r5.12xlarge', 96, 635.61 * 1024 * 1024 * 1024);
NodeType.DEFAULT = NodeType.M5_LARGE;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHFDQUFxQztBQUNyQyw4Q0FBOEM7QUFDOUMsd0NBQXdDO0FBQ3hDLDBEQUEwRDtBQUMxRCwyREFBMkQ7QUFDM0QsK0NBQTJDO0FBQzNDLDZEQUF3RDtBQUN4RCwyREFBdUQ7QUFDdkQsK0VBQXVFO0FBQ3ZFLHVFQUFrRTtBQTJFbEU7Ozs7Ozs7Ozs7R0FVRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFFaEQsTUFBTSxDQUFDLGVBQWUsQ0FBQyxLQUFvQixFQUFFLGNBQXNCLEVBQUUsS0FBNEI7UUFDdEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLEdBQUcsY0FBYyxHQUFHLFNBQVMsQ0FBQyxDQUFDO1FBQzFFLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxPQUFPLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDcEUsc0JBQXNCLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFTSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQW9CLEVBQUUsY0FBc0IsRUFBRSxLQUEyQjtRQUVwRyxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxJQUFJLGtCQUFrQixDQUFDO1FBQ2xELElBQUksUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksWUFBWSxDQUFDO1FBQzlDLElBQUksVUFBVSxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksVUFBVSxDQUFDO1FBRS9DLE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxHQUFHLGNBQWMsR0FBRyxXQUFXLEVBQUU7WUFDN0UsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztZQUNwRSxXQUFXLEVBQUU7Z0JBQ1gsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLFVBQVU7Z0JBQ25DLE9BQU8sRUFBRSxPQUFPO2dCQUNoQixRQUFRLEVBQUUsUUFBUTtnQkFDbEIsVUFBVSxFQUFFLFVBQVU7YUFDdkI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsR0FBRyxjQUFjLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFDekUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRTFELHNCQUFzQixDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRU0sTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFvQixFQUFFLGNBQXNCLEVBQUUsS0FBeUI7UUFDaEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLEdBQUcsY0FBYyxHQUFHLGlCQUFpQixDQUFDLENBQUM7UUFDbEYsS0FBSyxJQUFJLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQzlCLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxPQUFPLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7U0FDM0Q7UUFDRCxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVPLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBb0IsRUFBRSxjQUFzQixFQUFFLEtBQWdCLEVBQUUsS0FBa0I7UUFFckcseUJBQXlCO1FBQ3pCLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLE9BQU8sQ0FBQztRQUNwRCxNQUFNLE1BQU0sR0FBRztZQUNiLFFBQVE7WUFDUixxQkFBcUIsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDOUMsd0JBQXdCLEVBQUUsUUFBUSxDQUFDLFNBQVMsR0FBRyxHQUFHO1lBQ2xELEdBQUcsS0FBSztTQUNULENBQUE7UUFFRCx3QkFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDdEcscUNBQWdCLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDM0Ysb0NBQWdCLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDM0Ysb0RBQXNCLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsTUFBTSxDQUFDLHFCQUFxQixFQUFFLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNoSywrQ0FBcUIsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsY0FBYyxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ25ILENBQUM7Q0FFRjtBQTFERCx3REEwREM7QUFFRCxNQUFhLFFBQVE7SUFNbkIsWUFBb0IsSUFBWSxFQUFFLFNBQWlCLEVBQUUsTUFBYztRQUNqRSxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztRQUMzQixJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztJQUN2QixDQUFDOztBQVZILDRCQTJDQztBQS9Cd0IsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDekUsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDeEUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDMUUsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsR0FBRyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDdkUsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDeEUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDMUUsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDeEUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDM0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDN0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDOUUsb0JBQVcsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDakYsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDeEUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDM0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDN0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDOUUsb0JBQVcsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDakYsb0JBQVcsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDakYsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDeEUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDM0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDN0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDL0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDL0Usb0JBQVcsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsR0FBRyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDOUUsaUJBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDekUsa0JBQVMsR0FBRyxJQUFJLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDM0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsS0FBSyxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDN0UsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDL0Usb0JBQVcsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFDakYsbUJBQVUsR0FBRyxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLEVBQUUsTUFBTSxHQUFHLElBQUksR0FBRyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUM7QUFFaEYsZ0JBQU8sR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdAYXdzLWNkay9hd3Mtc25zJztcbmltcG9ydCAqIGFzIHNuc19zdWIgZnJvbSAnQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zJztcbmltcG9ydCAqIGFzIGFjdGlvbnMgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gtYWN0aW9ucyc7XG5pbXBvcnQgeyBDcHVNb25pdG9yIH0gZnJvbSAnLi9jcHUtbW9uaXRvcic7XG5pbXBvcnQgeyBTd2FwVXNhZ2VNb21pdG9uIH0gZnJvbSAnLi9zd2FwLXVzYWdlLW1vbml0b3InO1xuaW1wb3J0IHsgRXZpY3Rpb25zTW9taXRvbiB9IGZyb20gJy4vZXZpY3Rpb25zLW1vbml0b3InO1xuaW1wb3J0IHsgQ3VyckNvbm5lY3Rpb25zTW9taXRvbiB9IGZyb20gJy4vY3VycmVudC1jb25uZWN0aW9ucy1tb25pdG9yJztcbmltcG9ydCB7IEZyZWVhYmxlTWVtb3J5TW9taXRvbiB9IGZyb20gJy4vZnJlZWFibGUtbWVtb3J5LW1vbml0b3InO1xuXG4vKipcbiAqIEVsYXN0aWNhY2hlIGF1dG8gbW9uaXRvciBzZXQgdXAgYmFzZSBwcm9wcy5cbiAqL1xuaW50ZXJmYWNlIElTZXRVcFByb3BzIHtcbiAgLyoqXG4gICAqIERlZmF1bHQgZWxhc3RpY2FjaGUgbm9kZSB0eXBlLlxuICAgKiBJdCBpcyBzdHJvbmdseSByZWNvbW1lbmRlZCB0byBzZXQgYWNjb3JkaW5nIHRvIHRoZSBhY3R1YWwgdmFsdWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vZGVUeXBlLk01X0xBUkdFXG4gICAqL1xuICByZWFkb25seSBub2RlVHlwZT86IE5vZGVUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgcGVyaW9kIG92ZXIgd2hpY2ggdGhlIHNwZWNpZmllZCBzdGF0aXN0aWMgaXMgYXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcygxKVxuICAgKi9cbiAgcmVhZG9ubHkgY3VyckNvbm5lY3Rpb25zUGVyaW9kPzogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBDdXJyQ29ubmVjdGlvbnMgdGhyZXNob2xkXG4gICAqIFxuICAgKiBFdmVyeSAxIHZjdXAgaXMgZXF1YWwgdG8gNTAgY29ubmVjdGlvbnNcbiAgICogXG4gICAqIEBkZWZhdWx0IDEwMFxuICAgKi9cbiAgcmVhZG9ubHkgY3VyckNvbm5lY3Rpb25zVGhyZXNob2xkPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIEVsYXN0aWNhY2hlIGF1dG8gbW9uaXRvciBzZXQgdXAgd2l0aCBsYWJtZGEgcHJvcHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVNldFVwV2l0aExhbWJkYVByb3BzIGV4dGVuZHMgSVNldFVwUHJvcHMge1xuICByZWFkb25seSBsYW1iZGE6IGxhbWJkYS5GdW5jdGlvbixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJU2V0VXBXaXRoU2xhY2tQcm9wcyBleHRlbmRzIElTZXRVcFByb3BzIHtcblxuICAvKipcbiAgICogR28gdG8gdGhpcyhodHRwczovL3NsYWNrLmNvbS9hcHBzL0EwRjdYRFVBWi1pbmNvbWluZy13ZWJob29rcykgbGluayB0byBhcHBseSBmb3Igd2ViaG9vayBcbiAgICovXG4gIHJlYWRvbmx5IHdlYkhvb2tVcmw6IHN0cmluZztcblxuICAvKipcbiAgICogU2V0dGluZyBjaGFubmVsLlxuICAgKiBXaGF0IGlzIGEgY2hhbm5lbDogKGh0dHBzOi8vc2xhY2suY29tL2ludGwvZW4tY24vaGVscC9hcnRpY2xlcy8zNjAwMTc5Mzg5OTMtV2hhdC1pcy1hLWNoYW5uZWwpXG4gICAqXG4gICAqIEBkZWZhdWx0IGNsb3Vkd2F0Y2gtYWxhcm1cbiAgICovXG4gIHJlYWRvbmx5IGNoYW5uZWw/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNldHRpbmcgU2xhY2sgYm90IG5hbWUuXG4gICAqIFxuICAgKiBAZGVmYXVsdCBXZWJob29rYm90XG4gICAqL1xuICByZWFkb25seSB1c2VybmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogRW1vamkgZm9yIGJvdC5cbiAgICogXG4gICAqIEBkZWZhdWx0IDpzY3JlYW06XG4gICAqL1xuICByZWFkb25seSBpY29uRW1vamk/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVNldFVwV2l0aFNtc1Byb3BzIGV4dGVuZHMgSVNldFVwUHJvcHMge1xuICAvKipcbiAgICogSW5jbHVkZSBjb3VudHJ5IGNvZGUgYW5kIHBob25lIG51bWJlciwgZm9yIGV4YW1wbGU6ICsxNTU1MTIzMTIzNFxuICAgKi9cbiAgcmVhZG9ubHkgcGhvbmVzOiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBFbGFzdGljYWNoZUF1dG9Nb25pdG9yIGFsbG93cyB5b3UgdG8gc2VuZCBlbWFpbCwgc21zLCBzbGFjaywgb3IgdHJpZ2dlciBhd3Mgc25zIHRvcGljIHdoZW4gYW4gYWxhcm0gb2NjdXJzLlxuICogWW91IHdpbGwgZ2V0IHRoZSBmb2xsb3dpbmcgbW9uaXRvcmluZzpcbiAqICAxLiBDcHUgTW9uaXRvcjogU2hvdWxkIGJlIGxlc3MgdGhhbiA5MCUuIChTZWUgYmVsb3cgcmVmZXJlbmNlKVxuICogIDIuIFN3YXBVc2FnZSBNb25pdG9yOiBTaG91bGQgYmUgbGVzcyB0aGFuIDUwTS5cbiAqICAzLiBFdmljdGlvbnMgTW9uaXRvcjogU2hvdWxkIG5vdCBoYXZlIGV2aWN0aW9ucyB2YWx1ZS5cbiAqICA0LiBDdXJyQ29ubmVjdGlvbnMgTW9uaXRvcjogQWNjb3JkaW5nIHRvIHlvdXIgYnVzaW5lc3MgbmVlZHMsIGRlZmF1bHQgZXZlcnkgMSB2Y3VwIGlzIGVxdWFsIHRvIDIwMCBjb25uZWN0aW9ucy5cbiAqICA1LiBGcmVlYWJsZU1lbW9yeSBNb25pdG9yOiBOb3QgbGVzcyB0aGFuIDEwJVxuICogXG4gKiBSZWZlcmVuY2U6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FbGFzdGlDYWNoZS9sYXRlc3QvcmVkLXVnL0NhY2hlTWV0cmljcy5XaGljaFNob3VsZElNb25pdG9yLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIEVsYXN0aWNhY2hlQXV0b01vbml0b3IgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcblxuICBwdWJsaWMgc3RhdGljIHNldFVwV2l0aExhbWJkYShzY29wZTogY2RrLkNvbnN0cnVjdCwgY2FjaGVDbHVzdGVySWQ6IHN0cmluZywgcHJvcHM6IElTZXRVcFdpdGhMYW1iZGFQcm9wcykge1xuICAgIGNvbnN0IHRvcGljID0gbmV3IHNucy5Ub3BpYyhzY29wZSwgJ2FsYXJtLScgKyBjYWNoZUNsdXN0ZXJJZCArICctbGFtYmRhJyk7XG4gICAgdG9waWMuYWRkU3Vic2NyaXB0aW9uKG5ldyBzbnNfc3ViLkxhbWJkYVN1YnNjcmlwdGlvbihwcm9wcy5sYW1iZGEpKTtcbiAgICBFbGFzdGljYWNoZUF1dG9Nb25pdG9yLnNldHVwKHNjb3BlLCBjYWNoZUNsdXN0ZXJJZCwgdG9waWMsIHByb3BzKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgc2V0VXBXaXRoU2xhY2soc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGNhY2hlQ2x1c3RlcklkOiBzdHJpbmcsIHByb3BzOiBJU2V0VXBXaXRoU2xhY2tQcm9wcykge1xuXG4gICAgbGV0IGNoYW5uZWwgPSBwcm9wcy5jaGFubmVsIHx8ICdjbG91ZHdhdGNoLWFsYXJtJztcbiAgICBsZXQgdXNlcm5hbWUgPSBwcm9wcy51c2VybmFtZSB8fCAnV2ViaG9va2JvdCc7XG4gICAgbGV0IGljb25fZW1vamkgPSBwcm9wcy5pY29uRW1vamkgfHwgJzpzY3JlYW06JztcblxuICAgIGNvbnN0IGZuID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzY29wZSwgJ2FsYXJtLScgKyBjYWNoZUNsdXN0ZXJJZCArICctdG8tc2xhY2snLCB7XG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldC9sYW1iZGEnKSksXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBTTEFDS19XRUJIT09LX1VSTDogcHJvcHMud2ViSG9va1VybCxcbiAgICAgICAgQ0hBTk5FTDogY2hhbm5lbCxcbiAgICAgICAgVVNFUk5BTUU6IHVzZXJuYW1lLFxuICAgICAgICBJQ09OX0VNT0pJOiBpY29uX2Vtb2ppLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHRvcGljID0gbmV3IHNucy5Ub3BpYyhzY29wZSwgJ2FsYXJtLScgKyBjYWNoZUNsdXN0ZXJJZCArICctc2xhY2snKTtcbiAgICB0b3BpYy5hZGRTdWJzY3JpcHRpb24obmV3IHNuc19zdWIuTGFtYmRhU3Vic2NyaXB0aW9uKGZuKSk7XG5cbiAgICBFbGFzdGljYWNoZUF1dG9Nb25pdG9yLnNldHVwKHNjb3BlLCBjYWNoZUNsdXN0ZXJJZCwgdG9waWMsIHByb3BzKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgc2V0VXBXaXRoU21zKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBjYWNoZUNsdXN0ZXJJZDogc3RyaW5nLCBwcm9wczogSVNldFVwV2l0aFNtc1Byb3BzKSB7XG4gICAgY29uc3QgdG9waWMgPSBuZXcgc25zLlRvcGljKHNjb3BlLCAnYWxhcm0tJyArIGNhY2hlQ2x1c3RlcklkICsgJy13aXRoLXNtcy10b3BpYycpO1xuICAgIGZvciAobGV0IHBob25lIGluIHByb3BzLnBob25lcykge1xuICAgICAgdG9waWMuYWRkU3Vic2NyaXB0aW9uKG5ldyBzbnNfc3ViLlNtc1N1YnNjcmlwdGlvbihwaG9uZSkpO1xuICAgIH1cbiAgICBFbGFzdGljYWNoZUF1dG9Nb25pdG9yLnNldHVwKHNjb3BlLCBjYWNoZUNsdXN0ZXJJZCwgdG9waWMsIHByb3BzKTtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIHNldHVwKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBjYWNoZUNsdXN0ZXJJZDogc3RyaW5nLCB0b3BpYzogc25zLlRvcGljLCBwcm9wczogSVNldFVwUHJvcHMpIHtcblxuICAgIC8vIGJpbmQgZGVmYXVsdCBhbmQgcHJvcHNcbiAgICBjb25zdCBub2RlVHlwZSA9IHByb3BzLm5vZGVUeXBlIHx8IE5vZGVUeXBlLkRFRkFVTFQ7XG4gICAgY29uc3QgX3Byb3BzID0ge1xuICAgICAgbm9kZVR5cGUsXG4gICAgICBjdXJyQ29ubmVjdGlvbnNQZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgY3VyckNvbm5lY3Rpb25zVGhyZXNob2xkOiBub2RlVHlwZS52Y3VwQ291bnQgKiAyMDAsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9XG5cbiAgICBDcHVNb25pdG9yLnNldHVwKHNjb3BlLCBjYWNoZUNsdXN0ZXJJZCwgX3Byb3BzLm5vZGVUeXBlKS5hZGRBbGFybUFjdGlvbihuZXcgYWN0aW9ucy5TbnNBY3Rpb24odG9waWMpKTtcbiAgICBTd2FwVXNhZ2VNb21pdG9uLnNldHVwKHNjb3BlLCBjYWNoZUNsdXN0ZXJJZCkuYWRkQWxhcm1BY3Rpb24obmV3IGFjdGlvbnMuU25zQWN0aW9uKHRvcGljKSk7XG4gICAgRXZpY3Rpb25zTW9taXRvbi5zZXR1cChzY29wZSwgY2FjaGVDbHVzdGVySWQpLmFkZEFsYXJtQWN0aW9uKG5ldyBhY3Rpb25zLlNuc0FjdGlvbih0b3BpYykpO1xuICAgIEN1cnJDb25uZWN0aW9uc01vbWl0b24uc2V0dXAoc2NvcGUsIGNhY2hlQ2x1c3RlcklkLCBfcHJvcHMuY3VyckNvbm5lY3Rpb25zUGVyaW9kLCBfcHJvcHMuY3VyckNvbm5lY3Rpb25zVGhyZXNob2xkKS5hZGRBbGFybUFjdGlvbihuZXcgYWN0aW9ucy5TbnNBY3Rpb24odG9waWMpKTtcbiAgICBGcmVlYWJsZU1lbW9yeU1vbWl0b24uc2V0dXAoc2NvcGUsIGNhY2hlQ2x1c3RlcklkLCBfcHJvcHMubm9kZVR5cGUpLmFkZEFsYXJtQWN0aW9uKG5ldyBhY3Rpb25zLlNuc0FjdGlvbih0b3BpYykpO1xuICB9XG5cbn1cblxuZXhwb3J0IGNsYXNzIE5vZGVUeXBlIHtcblxuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHZjdXBDb3VudDogbnVtYmVyO1xuICByZWFkb25seSBtZW1vcnk6IG51bWJlcjtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZywgdmN1cENvdW50OiBudW1iZXIsIG1lbW9yeTogbnVtYmVyKSB7XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgICB0aGlzLnZjdXBDb3VudCA9IHZjdXBDb3VudDtcbiAgICB0aGlzLm1lbW9yeSA9IG1lbW9yeTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVDJfTUlSQ08gPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnQyLm1pcmNvJywgMSwgMC41NTUgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFQyX1NNQUxMID0gbmV3IE5vZGVUeXBlKCdjYWNoZS50Mi5zbWFsbCcsIDEsIDEuNTUgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFQyX01FRElVTSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUudDIubWVkaXVtJywgMiwgMy4yMiAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVDNfTUlDUk8gPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnQzLm1pY3JvJywgMiwgMC41ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBUM19TTUFMTCA9IG5ldyBOb2RlVHlwZSgnY2FjaGUudDMuc21hbGwnLCAyLCAxLjM3ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBUM19NRURJVU0gPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnQzLm1lZGl1bScsIDIsIDMuMDkgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE00X0xBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5tNC5sYXJnZScsIDIsIDYuNDIgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE00X1hMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUubTQueGxhcmdlJywgNCwgMTQuMjggKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE00XzJYTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLm00LjJ4bGFyZ2UnLCA4LCAyOS43MCAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTTRfNFhMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUubTQuNHhsYXJnZScsIDE2LCA2MC43OCAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTTRfMTBYTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLm00LjEweGxhcmdlJywgNDAsIDE1NC42NCAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTTVfTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLm01LmxhcmdlJywgMiwgNi4zOCAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTTVfWExBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5tNS54bGFyZ2UnLCA0LCAxMi45MyAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTTVfMlhMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUubTUuMnhsYXJnZScsIDgsIDI2LjA0ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNNV80WExBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5tNS40eGxhcmdlJywgMTYsIDUyLjI2ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNNV8xMlhMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUubTUuMTJ4bGFyZ2UnLCA0OCwgMTU3LjEyICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNNV8yNFhMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUubTUuMTJ4bGFyZ2UnLCA5NiwgMzE0LjMyICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSNF9MQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUucjQubGFyZ2UnLCAyLCAxMi4zICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSNF9YTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI0LnhsYXJnZScsIDQsIDI1LjA1ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSNF8yWExBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5yNC4yeGxhcmdlJywgOCwgNTAuNDcgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFI0XzRYTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI0LjR4bGFyZ2UnLCAxNiwgMTAxLjM4ICogMTAyNCAqIDEwMjQgKiAxMDI0KTtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSNF84WExBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5yNC44eGxhcmdlJywgMzIsIDIwMy4yNiAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUjRfMTZYTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI0LjE2eGxhcmdlJywgNjQsIDQwNyAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUjVfTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI1LmxhcmdlJywgMiwgMTMuMDcgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFI1X1hMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUucjUueGxhcmdlJywgNCwgMjYuMzIgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFI1XzJYTEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI1LjJ4bGFyZ2UnLCA4LCA1Mi44MiAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUjVfNFhMQVJHRSA9IG5ldyBOb2RlVHlwZSgnY2FjaGUucjUuNHhsYXJnZScsIDE2LCAxMDUuODEgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFI1XzEyWExBUkdFID0gbmV3IE5vZGVUeXBlKCdjYWNoZS5yNS4xMnhsYXJnZScsIDQ4LCAzMTcuNzcgKiAxMDI0ICogMTAyNCAqIDEwMjQpO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFI1XzI0TEFSR0UgPSBuZXcgTm9kZVR5cGUoJ2NhY2hlLnI1LjEyeGxhcmdlJywgOTYsIDYzNS42MSAqIDEwMjQgKiAxMDI0ICogMTAyNCk7XG5cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZBVUxUID0gTm9kZVR5cGUuTTVfTEFSR0U7XG59XG4iXX0=