#
#  This file is part of Sequana software
#
#  Copyright (c) 2020 - Sequana Development Team
#
#  Distributed under the terms of the 3-clause BSD license.
#  The full license is in the LICENSE file, distributed with this software.
#
#  website: https://github.com/sequana/sequana
#  documentation: http://sequana.readthedocs.io
#
##############################################################################
"""Module to write KEGG enrichment report"""
import ast
import os
import sys

from sequana.lazy import pandas as pd
from sequana.lazy import pylab

from sequana.modules_report.base_module import SequanaBaseModule
from sequana.utils.datatables_js import DataTable

from easydev import Progress

import colorlog

logger = colorlog.getLogger(__name__)

from sequana.utils import config


class ModuleKEGGEnrichment(SequanaBaseModule):
    """Write HTML report of variant calling. This class takes a csv file
    generated by sequana_variant_filter.
    """

    def __init__(
        self,
        gene_lists,
        kegg_name,
        dataframe,
        enrichment_params={
            "padj": 0.05,
            "log2_fc": 3,
            "nmax": 15,
            "max_entries": 3000,
            "kegg_background": None,
            "mapper": None,
            "preload_directory": None,
            "plot_logx": True,
        },
        command="",
    ):
        """.. rubric:: constructor"""
        super().__init__()
        self.title = "Enrichment"

        self.command = command
        self.gene_lists = gene_lists
        self.enrichment_params = enrichment_params
        self.data = dataframe
        self.organism = kegg_name

        if self.enrichment_params["preload_directory"]:
            pathname = self.enrichment_params["preload_directory"]
            if not os.path.exists(pathname):
                logger.error(f"{pathname} does not exist")
                sys.exit(1)
        self.nmax = enrichment_params.get("nmax", 15)

        self.create_report_content()
        self.create_html("enrichment.html")

    def create_report_content(self):
        self.sections = list()
        self.summary()
        self.add_kegg()
        self.sections.append(
            {"name": "3 - Info", "anchor": "command", "content": self.command}
        )

    def summary(self):
        """Add information of filter."""
        total_up = len(self.gene_lists["up"])
        total_down = len(self.gene_lists["down"])
        total = total_up + total_down
        log2fc = self.enrichment_params["log2_fc"]

        self.sections.append(
            {
                "name": "1 - Summary",
                "anchor": "filters_option",
                "content": f"""

<p>In the following sections, you will find the KEGG Pathway enrichment.
The input data for those analyis is the output of the RNADiff
analysis where adjusted p-values above 0.05 are excluded. Moreover, we removed 
candidates with log2 fold change below {log2fc}. Using these filters, the list of
differentially expressed genes is made of {total_up} up and {total_down} down genes (total {total})</p>
<p> In the following plots you can find the first KEGG Pathways that are enriched, keeping a 
maximum of {self.nmax} pathways. </p>

<p>The KEGG name used is {self.organism}.<br>




""",
            }
        )

    def add_kegg(self):
        logger.info("Enrichment module: kegg term")
        style = "width:45%"
        from sequana.enrichment.kegg import KEGGPathwayEnrichment

        ke = KEGGPathwayEnrichment(
            self.gene_lists,
            self.organism,
            mapper=self.enrichment_params["mapper"],
            background=self.enrichment_params["kegg_background"],
            preload_directory=self.enrichment_params["preload_directory"],
        )

        logger.info(f"Saving all pathways in kegg_pathways/{self.organism}")
        ke.export_pathways_to_json()

        # Image kegg pathways down
        def plot_barplot_down(filename):
            ke.barplot("down", nmax=self.nmax)
            pylab.savefig(filename)

        img_barplot_down = self.create_embedded_png(
            plot_barplot_down, "filename", style=style
        )

        def plot_scatter_down(filename):
            ke.scatterplot("down", nmax=self.nmax)
            pylab.savefig(filename)

        img_scatter_down = self.create_embedded_png(
            plot_scatter_down, "filename", style=style
        )

        # Image kegg pathways up
        def plot_barplot_up(filename):
            ke.barplot("up", nmax=self.nmax)
            pylab.savefig(filename)

        img_barplot_up = self.create_embedded_png(
            plot_barplot_up, "filename", style=style
        )

        def plot_scatter_up(filename):
            ke.scatterplot("up", nmax=self.nmax)
            pylab.savefig(filename)

        img_scatter_up = self.create_embedded_png(
            plot_scatter_up, "filename", style=style
        )

        # Results down (pathway info)
        html_before_table = """<p>Enrichment pathways summary</p>"""
        df_down = ke.barplot("down", nmax=self.nmax)

        if len(df_down):
            links = [
                "https://www.genome.jp/dbget-bin/www_bget?path:{}".format(x)
                for x in df_down["pathway_id"]
            ]
            df_down["links"] = links
            df_down = df_down[
                [
                    "pathway_id",
                    "name",
                    "size",
                    "Overlap",
                    "P-value",
                    "Adjusted P-value",
                    "Genes",
                    "links",
                ]
            ]

            # save pathways and add fotorama
            logger.setLevel("WARNING")
            pb = Progress(len(df_down))
            files = []
            for i, ID in enumerate(df_down["pathway_id"]):
                df = ke.save_pathway(
                    ID, self.data, filename=f"{config.output_dir}/{ID}.png"
                )
                files.append(f"{ID}.png")
                pb.animate(i + 1)
            fotorama_down = self.add_fotorama(files, width=800)

            datatable = DataTable(df_down, "kegg_down")
            datatable.datatable.set_links_to_column("links", "pathway_id")
            datatable.datatable.datatable_options = {
                "scrollX": "true",
                "pageLength": 20,
                "scrollCollapse": "true",
                "dom": "Bfrtip",
                "buttons": ["copy", "csv"],
            }
            js_table_down = datatable.create_javascript_function()
            html_table_down = datatable.create_datatable(float_format="%E")

        # Results up (pathway info)
        df_up = ke.barplot("up", nmax=self.nmax)
        if len(df_up):
            links = [
                "https://www.genome.jp/dbget-bin/www_bget?path:{}".format(x)
                for x in df_up["pathway_id"]
            ]
            df_up["links"] = links
            df_up = df_up[
                [
                    "pathway_id",
                    "name",
                    "size",
                    "Overlap",
                    "P-value",
                    "Adjusted P-value",
                    "Genes",
                    "links",
                ]
            ]
            datatable = DataTable(df_up, "kegg_up")
            datatable.datatable.set_links_to_column("links", "pathway_id")
            datatable.datatable.datatable_options = {
                "scrollX": "true",
                "pageLength": 20,
                "scrollCollapse": "true",
                "dom": "Bfrtip",
                "buttons": ["copy", "csv"],
            }
            js_table_up = datatable.create_javascript_function()
            html_table_up = datatable.create_datatable(float_format="%E")
            pb = Progress(len(df_up))
            files = []
            for i, ID in enumerate(df_up["pathway_id"]):
                df = ke.save_pathway(
                    ID, self.data, filename=f"{config.output_dir}/{ID}.png"
                )
                files.append(f"{ID}.png")
                pb.animate(i + 1)
            fotorama_up = self.add_fotorama(files, width=800)
            # logger.setLevel(level)

        Ndown = len(df_down)
        Nup = len(df_up)

        if Ndown == 0:
            img_barplot_down = ""
            img_scatter_down = ""
            fotorama_down = ""
            js_table_down = ""
            html_table_down = ""
        if Nup == 0:
            img_barplot_up = ""
            img_scatter_up = ""
            fotorama_up = ""
            js_table_up = ""
            html_table_up = ""

        html = f"""
<h3>2.1 - KEGG pathways down regulated</h3>
<p>{Ndown} KEGG pathways are found to be down regulated</p>
<br>
{img_barplot_down}
{img_scatter_down}
<hr>
{js_table_down} {html_table_down}
<hr>
<p>Here below are the pathways with gene colored according to their fold change.
Blue colors are for down-regulated genes and Orange are for up-regulated genes. 
(Note that absolute log2 fold change above 4 are clipped to 4; So a gene with a
log2 fold change of 4 of 40 will have the same darkest color.). </p>
{fotorama_down}


<h3>2.1 - KEGG pathways up regulated</h3>
<p>{Nup} KEGG pathways are found to be up regulated</p>
<br>
{img_barplot_up}
{img_scatter_up}
<hr>
{js_table_up} {html_table_up}
<hr>
<p>Here below are the pathways with gene colored according to their fold change.
Blue colors are for down-regulated genes and Orange are for up-regulated genes. 
(Note that absolute log2 fold change above 4 are clipped to 4; So a gene with a
log2 fold change of 4 of 40 will have the same darkest color.). </p>
{fotorama_up}
"""
        self.sections.append({"name": "2 - KEGG", "anchor": "kegg", "content": html})
