# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['reltools']
setup_kwargs = {
    'name': 'reltools',
    'version': '1.0.0',
    'description': 'Relation tools for Python.',
    'long_description': "reltools\n========\n.. image:: https://circleci.com/gh/ymoch/reltools.svg?style=svg\n    :target: https://circleci.com/gh/ymoch/reltools\n.. image:: https://codecov.io/gh/ymoch/reltools/branch/master/graph/badge.svg\n    :target: https://codecov.io/gh/ymoch/reltools\n.. image:: https://badge.fury.io/py/reltools.svg\n    :target: https://badge.fury.io/py/reltools\n.. image:: https://img.shields.io/badge/python-3.6+-blue.svg\n    :target: https://www.python.org/\n.. image:: https://img.shields.io/lgtm/grade/python/g/ymoch/reltools.svg\n    :target: https://lgtm.com/projects/g/ymoch/reltools/context:python\n\nRelation tools for Python.\nThis relates two data (sorted by certain keys) like SQL joining.\n\nInspired by `itertools.groupby <https://docs.python.org/3.6/library/itertools.html#itertools.groupby>`_,\nas long as input data are sorted, almost all processes are evaluated lazily,\nwhich results in the reduction of memory usage.\nThis feature is for the big data joining without any SQL engines.\n\nInstallation\n------------\nInstall with `pip <https://pypi.org/project/pip/>`_.\n\n.. code-block:: sh\n\n   pip install reltools\n\nFeatures\n--------\n\nOne-To-Many\n***********\n*One-to-many* relationing is provided by ``relate_one_to_many``.\n\nHere, input left-hand-side (``lhs``) and right-hand-side (``rhs``)\nare sorted in 1st (and also 2nd) keys.\n\n>>> lhs = [\n...     (1, 'a', 's'),\n...     (2, 'a', 't'),\n...     (3, 'b', 'u'),\n... ]\n>>> rhs = [\n...     (1, 'a', 'v'),\n...     (1, 'b', 'w'),\n...     (3, 'b', 'x'),\n... ]\n\n``relate_one_to_many`` relates ``rhs`` items\nto each ``lhs`` item using the first items as keys.\n\n>>> from reltools import relate_one_to_many\n>>> one_to_many_related = relate_one_to_many(lhs, rhs)\n>>> for left, right in one_to_many_related:\n...     left, list(right)\n((1, 'a', 's'), [(1, 'a', 'v'), (1, 'b', 'w')])\n((2, 'a', 't'), [])\n((3, 'b', 'u'), [(3, 'b', 'x')])\n\nYou can use custom key functions\nfor not only ``relate_one_to_many`` but also API functions.\n\n>>> import operator\n>>> custom_key = operator.itemgetter(0, 1)\n>>> one_to_many_related = relate_one_to_many(\n...     lhs, rhs, lhs_key=custom_key, rhs_key=custom_key)\n>>> for left, right in one_to_many_related:\n...     left, list(right)\n((1, 'a', 's'), [(1, 'a', 'v')])\n((2, 'a', 't'), [])\n((3, 'b', 'u'), [(3, 'b', 'x')])\n\n``OneToManyChainer`` helps to relate many ``rhs`` iterables to ``lhs``.\n\n>>> another_rhs = [\n...     ('s', 'f'),\n...     ('t', 'g'),\n...     ('t', 'h'),\n... ]\n>>> from reltools import OneToManyChainer\n>>> chainer = OneToManyChainer(lhs)\n>>> chainer.append(rhs)\n>>> chainer.append(\n...     another_rhs,\n...     lhs_key=operator.itemgetter(2),\n...     rhs_key=operator.itemgetter(0),\n... )\n>>> for left, right, another_right in chainer.chain():\n...     left, list(right), list(another_right)\n((1, 'a', 's'), [(1, 'a', 'v'), (1, 'b', 'w')], [('s', 'f')])\n((2, 'a', 't'), [], [('t', 'g'), ('t', 'h')])\n((3, 'b', 'u'), [(3, 'b', 'x')], [])\n\nLeft Outer Join\n***************\nLeft outer joining is provided by ``left_join``.\nWhile SQL left outer joining returns all the combinations,\nthis returns the pair of items.\nNote that the ``right`` can empty, like SQL left joining.\n\n>>> from reltools import left_join\n>>> lhs = [(1, 'a'), (1, 'b'), (2, 'c'), (4, 'd')]\n>>> rhs = [(1, 's'), (1, 't'), (3, 'u'), (4, 'v')]\n>>> relations = left_join(lhs, rhs)\n>>> for left, right in relations:\n...     list(left), list(right)\n([(1, 'a'), (1, 'b')], [(1, 's'), (1, 't')])\n([(2, 'c')], [])\n([(4, 'd')], [(4, 'v')])\n\nRight Outer Join\n****************\nRight outer joining is not supported\nbecause it is left-and-right-opposite of left joining.\nUse ``left_join(rhs, lhs, rhs_key, lhs_key)``.\n\nFull Outer Join\n***************\nFull outer joining, which is an original feature of *reltools*,\nis provided by ``outer_join``.\nIn contrast to ``left_join``, full outer joining preserve keys\nthat are only in ``rhs``.\n\n>>> from reltools import outer_join\n>>> lhs = [(1, 'a'), (1, 'b'), (2, 'c'), (4, 'd')]\n>>> rhs = [(1, 's'), (1, 't'), (3, 'u'), (4, 'v')]\n>>> relations = outer_join(lhs, rhs)\n>>> for left, right in relations:\n...     list(left), list(right)\n([(1, 'a'), (1, 'b')], [(1, 's'), (1, 't')])\n([(2, 'c')], [])\n([], [(3, 'u')])\n([(4, 'd')], [(4, 'v')])\n\nInner Join\n**********\nInner joining is provided by ``inner_join``.\nIn contrast to ``left_join``, ``right`` cannot be empty,\nlike SQL inner joining.\n\n>>> from reltools import inner_join\n>>> relations = inner_join(lhs, rhs)\n>>> for left, right in relations:\n...     list(left), list(right)\n([(1, 'a'), (1, 'b')], [(1, 's'), (1, 't')])\n([(4, 'd')], [(4, 'v')])\n\nMany-To-Many\n************\nSQL-like *many-to-many* relationing using an internal table is not supported.\nThis is because *reltools* supports only sorted data\nand does not prefer random accessing.\nTo achieve *many-to-many* relationing, unnormalize data on preproceing and\nuse outer joining or inner joining.\n\nMemory Efficiency\n*****************\nAlmost all processes are evaluated lazily,\nwhich results in the reduction of memory usage.\n(You can try the efficiency by commands like\n``RELTOOLS_TRY_COUNT=10000000 python3 -m doctest README.rst``)\n\n>>> import os\n>>> n = int(os.environ.get('RELTOOLS_TRY_COUNT', 1000))\n>>> lhs = ((i, 'left') for i in range(n))\n>>> rhs = ((i, 'right') for i in range(n))\n>>> for left, right in relate_one_to_many(lhs, rhs):\n...     assert len(list(right)) == 1\n\nDevelopment\n-----------\nThis project's structure is based on `Poetry <https://poetry.eustace.io/>`_.\nAll tests are written with `doctest <https://docs.python.jp/3/library/doctest.html>`_\nand run with `pytest <https://docs.pytest.org/en/latest/>`_.\n\n.. code-block:: sh\n\n    poetry install\n    poetry run pytest\n\nFor stability, following checks are also run when testing.\n\n- `pyflakes <https://github.com/PyCQA/pyflakes>`_\n- `pycodestyle <https://pycodestyle.readthedocs.io/en/latest/>`_\n- `mypy <https://mypy.readthedocs.io/en/stable/index.html>`_\n\nLicense\n-------\n.. image:: https://img.shields.io/badge/License-MIT-brightgreen.svg\n    :target: https://opensource.org/licenses/MIT\n\nCopyright (c) 2018 Yu MOCHIZUKI\n",
    'author': 'ymoch',
    'author_email': 'ymoch.dev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ymoch/reltools',
    'py_modules': modules,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
