from typing import Callable, Dict

from autumn8.common.config.cloud_info import CloudInfoService, InstanceDescription
from autumn8.common.config.settings import CloudServiceProvider
from autumn8.common.types import DEFAULT_CLOUD_ZONES_CONFIG, ZoneConfig
from autumn8.lib import logging

logger = logging.getLogger(__name__)


def get_default_cloud_info_services_per_provider():
    return {
        provider: CloudInfoService(provider, zone)
        for provider, zone in DEFAULT_CLOUD_ZONES_CONFIG.items()
    }


def find_instance_config(
    predictor_target, fetch_data_from_cloud_info=True, num_threads=None
) -> InstanceDescription:
    instances = get_supported_cloud_instances(
        fetch_data_from_cloud_info=fetch_data_from_cloud_info
    )

    if predictor_target not in instances:
        raise ValueError(
            f'Instance config not found for target="{predictor_target}"'
        )

    instances_by_num_threads = instances[predictor_target]

    if len(instances_by_num_threads.items()) == 1:
        the_only_key = [*instances_by_num_threads.keys()][0]
        num_threads = the_only_key

    if not num_threads:
        # TODO: after replacing predictor_target with label, remove this check
        raise ValueError(
            f"You have to provide num_threads to be able to query Oracle instances"
        )

    return instances_by_num_threads[num_threads]


# TODO: use some smarter caching mechanism here
ALL_CLOUD_INFO_SERVICES = {}

PLACEHOLDER_ZONE_FOR_DISCONNECTED_CLOUD_INFO_SERVICES = "disconnected"


def get_cloud_info_service(
    provider: CloudServiceProvider,
    zone_config: ZoneConfig,
    fetch_data_from_cloud_info=True,
):
    if provider not in ALL_CLOUD_INFO_SERVICES:
        ALL_CLOUD_INFO_SERVICES[provider] = {}

    provider_services = ALL_CLOUD_INFO_SERVICES[provider]
    zone = (
        zone_config[provider]
        if fetch_data_from_cloud_info
        else PLACEHOLDER_ZONE_FOR_DISCONNECTED_CLOUD_INFO_SERVICES
    )

    if zone not in provider_services:
        provider_services[zone] = CloudInfoService(
            provider,
            zone,
            fetch_data_from_cloud_info=fetch_data_from_cloud_info,
        )

    return provider_services[zone]


INSTANCE_DESCRIPTIONS_PER_ZONE = {}


def get_instances_for_cloud_provider(
    provider: CloudServiceProvider,
    zone_config: ZoneConfig,
    instances_getter: Callable,
    fetch_data_from_cloud_info=True,
):
    cloud_info_service = get_cloud_info_service(
        provider,
        zone_config,
        fetch_data_from_cloud_info=fetch_data_from_cloud_info,
    )
    zone = (
        zone_config[provider]
        if fetch_data_from_cloud_info
        else PLACEHOLDER_ZONE_FOR_DISCONNECTED_CLOUD_INFO_SERVICES
    )

    if provider not in INSTANCE_DESCRIPTIONS_PER_ZONE:
        INSTANCE_DESCRIPTIONS_PER_ZONE[provider] = {}

    provider_instances = INSTANCE_DESCRIPTIONS_PER_ZONE[provider]

    if zone not in provider_instances:
        provider_instances[zone] = instances_getter(cloud_info_service)

    return provider_instances[zone]


INSTANCE_RECOMMENDATIONS_LABELS = [
    "t3.medium",
    "c5.large",
    "g5.2xlarge",
    "e2-standard-2",
    "c2-standard-4",
    "VM.Standard.A1.Flex 2vCPU",
    "Azure.StandardB2s",
]


def get_recommendations(fetch_data_from_cloud_info=True):
    all_supported_instances = get_supported_cloud_instances(
        fetch_data_from_cloud_info=fetch_data_from_cloud_info
    )
    instance_descriptions = [
        description
        for instances_by_thread_count in all_supported_instances.values()
        for description in instances_by_thread_count.values()
    ]
    return [
        instance
        for instance in instance_descriptions
        if instance.label in INSTANCE_RECOMMENDATIONS_LABELS
    ]


# TODO gracefully handle cases where a service is not available in a specific zone
# TODO users should be able to choose cheaper regions
# ASSUMPTION: pair (predictor_target, predictor_num_threads) is unique across
#   all instance_types, across all cloud service_providers
# TODO: patch Oracle metrics to use label instead of the duplicate
#   predictor_target, drop num_threads from unique constraint
def get_supported_cloud_instances(
    zone_config=DEFAULT_CLOUD_ZONES_CONFIG,
    fetch_data_from_cloud_info=True,
) -> Dict[str, Dict[float, InstanceDescription]]:

    return {
        # TODO: replace predictor_target with 'label' instead, eliminate
        #   second dict layer indexing by num_threads - run migration
        #   for all instances, rename benchmarks
        **get_instances_for_cloud_provider(
            CloudServiceProvider.AMAZON,
            zone_config,
            get_supported_amazon_cloud_instances,
            fetch_data_from_cloud_info=fetch_data_from_cloud_info,
        ),
        **get_instances_for_cloud_provider(
            CloudServiceProvider.ORACLE,
            zone_config,
            get_supported_oracle_cloud_instances,
            fetch_data_from_cloud_info=fetch_data_from_cloud_info,
        ),
        **get_instances_for_cloud_provider(
            CloudServiceProvider.GOOGLE,
            zone_config,
            get_supported_google_cloud_instances,
            fetch_data_from_cloud_info=fetch_data_from_cloud_info,
        ),
        **get_instances_for_cloud_provider(
            CloudServiceProvider.AZURE,
            zone_config,
            get_supported_azure_cloud_instances,
            fetch_data_from_cloud_info=fetch_data_from_cloud_info,
        ),
    }


def get_supported_amazon_cloud_instances(
    amazon_cloud_info_service: CloudInfoService,
):
    return {
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="lambda_10240MB",
            family="lambda",
            instance_description_link="https://aws.amazon.com/lambda/",
            hw="",
            cores=5,
            hyperthreading=True,
            hardcoded_usd_per_hr=0.0000001667
            * 1000
            * 3600,  # TODO cloudinfo doesn't support lambdas pricing, handle that across zones somehow
            predictor_target="lambda_10240MB",
            predictor_num_threads=5,
            ram=10,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.large",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=2,
            hyperthreading=True,
            predictor_target="c5.large",
            predictor_num_threads=1,
            ram=4,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=4,
            hyperthreading=True,
            predictor_target="c5.xlarge",
            predictor_num_threads=2,
            ram=8,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.2xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=8,
            hyperthreading=True,
            predictor_target="c5.2xlarge",
            predictor_num_threads=4,
            ram=16,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.4xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=16,
            hyperthreading=True,
            predictor_target="c5.4xlarge",
            predictor_num_threads=8,
            ram=32,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.9xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=36,
            hyperthreading=True,
            predictor_target="c5.9xlarge",
            predictor_num_threads=18,
            ram=72,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.12xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=48,
            hyperthreading=True,
            predictor_target="c5.12xlarge",
            predictor_num_threads=24,
            ram=96,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.18xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=72,
            hyperthreading=True,
            predictor_target="c5.18xlarge",
            predictor_num_threads=36,
            ram=144,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5.24xlarge",
            family="c5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=96,
            hyperthreading=True,
            predictor_target="c5.24xlarge",
            predictor_num_threads=48,
            ram=192,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.large",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=2,
            hyperthreading=True,
            predictor_target="c5a.large",
            predictor_num_threads=1,
            ram=4,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=4,
            hyperthreading=True,
            predictor_target="c5a.xlarge",
            predictor_num_threads=2,
            ram=8,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.2xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=8,
            hyperthreading=True,
            predictor_target="c5a.2xlarge",
            predictor_num_threads=4,
            ram=16,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.4xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=16,
            hyperthreading=True,
            predictor_target="c5a.4xlarge",
            predictor_num_threads=8,
            ram=32,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.8xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=32,
            hyperthreading=True,
            predictor_target="c5a.8xlarge",
            predictor_num_threads=16,
            ram=64,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.12xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=48,
            hyperthreading=True,
            predictor_target="c5a.12xlarge",
            predictor_num_threads=24,
            ram=96,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.16xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=64,
            hyperthreading=True,
            predictor_target="c5a.16xlarge",
            predictor_num_threads=32,
            ram=128,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5a.24xlarge",
            family="c5a",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="AMD EPYC",
            cores=96,
            hyperthreading=True,
            predictor_target="c5a.24xlarge",
            predictor_num_threads=48,
            ram=192,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.large",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=2,
            hyperthreading=True,
            predictor_target="c5n.large",
            predictor_num_threads=2,
            ram=5.25,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.xlarge",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=4,
            hyperthreading=True,
            predictor_target="c5n.xlarge",
            predictor_num_threads=4,
            ram=10.5,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.2xlarge",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=8,
            hyperthreading=True,
            predictor_target="c5n.2xlarge",
            predictor_num_threads=8,
            ram=21,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.4xlarge",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=16,
            hyperthreading=True,
            predictor_target="c5n.4xlarge",
            predictor_num_threads=16,
            ram=42,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.9xlarge",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=36,
            hyperthreading=True,
            predictor_target="c5n.9xlarge",
            predictor_num_threads=36,
            ram=96,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c5n.18xlarge",
            family="c5n",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c5/",
            hw="Intel Xeon",
            cores=72,
            hyperthreading=True,
            predictor_target="c5n.18xlarge",
            predictor_num_threads=72,
            ram=192,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t2.small",
            family="t2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t2/",
            hw="AMD EPYC",
            cores=1,
            hyperthreading=True,
            predictor_target="t2.small",
            predictor_num_threads=1,
            ram=2,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t2.medium",
            family="t2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t2/",
            hw="AMD EPYC",
            cores=2,
            hyperthreading=True,
            predictor_target="t2.medium",
            predictor_num_threads=2,
            ram=4,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t2.large",
            family="t2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t2/",
            hw="AMD EPYC",
            cores=2,
            hyperthreading=True,
            predictor_target="t2.large",
            predictor_num_threads=2,
            ram=8,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t2.xlarge",
            family="t2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t2/",
            hw="AMD EPYC",
            cores=4,
            hyperthreading=True,
            predictor_target="t2.xlarge",
            predictor_num_threads=4,
            ram=16,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t2.2xlarge",
            family="t2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t2/",
            hw="AMD EPYC",
            cores=8,
            hyperthreading=True,
            predictor_target="t2.2xlarge",
            predictor_num_threads=8,
            ram=32,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t3.small",
            family="t3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t3/",
            hw="Intel Xeon Platinum",
            cores=2,
            hyperthreading=True,
            predictor_target="t3.small",
            predictor_num_threads=2,
            ram=2,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t3.medium",
            family="t3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t3/",
            hw="Intel Xeon Platinum",
            cores=2,
            hyperthreading=True,
            predictor_target="t3.medium",
            predictor_num_threads=2,
            ram=4,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t3.large",
            family="t3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t3/",
            hw="Intel Xeon Platinum",
            cores=2,
            hyperthreading=True,
            predictor_target="t3.large",
            predictor_num_threads=2,
            ram=8,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t3.xlarge",
            family="t3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t3/",
            hw="Intel Xeon Platinum",
            cores=4,
            hyperthreading=True,
            predictor_target="t3.xlarge",
            predictor_num_threads=4,
            ram=16,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t3.2xlarge",
            family="t3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t3/",
            hw="Intel Xeon Platinum",
            cores=8,
            hyperthreading=True,
            predictor_target="t3.2xlarge",
            predictor_num_threads=8,
            ram=32,
            aws_govcloud=True,
        ),
        # t4g micro and nano missing
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t4g.small",
            family="t4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t4/",
            hw="Amazon Graviton2",
            cores=2,
            hyperthreading=True,
            predictor_target="t4g.small",
            predictor_num_threads=4,
            ram=2,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t4g.medium",
            family="t4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t4/",
            hw="Amazon Graviton2",
            cores=2,
            hyperthreading=True,
            predictor_target="t4g.medium",
            predictor_num_threads=4,
            ram=4,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t4g.large",
            family="t4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t4/",
            hw="Amazon Graviton2",
            cores=2,
            hyperthreading=True,
            predictor_target="t4g.large",
            predictor_num_threads=4,
            ram=8,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t4g.xlarge",
            family="t4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t4/",
            hw="Amazon Graviton2",
            cores=4,
            hyperthreading=True,
            predictor_target="t4g.xlarge",
            predictor_num_threads=8,
            ram=16,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="t4g.2xlarge",
            family="t4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/t4/",
            hw="Amazon Graviton2",
            cores=8,
            hyperthreading=True,
            predictor_target="t4g.2xlarge",
            predictor_num_threads=16,
            ram=32,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.medium",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=1,
            hyperthreading=False,
            predictor_target="c6g.medium",
            predictor_num_threads=1,
            ram=2,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.large",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=2,
            hyperthreading=False,
            predictor_target="c6g.large",
            predictor_num_threads=2,
            ram=4,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=4,
            hyperthreading=False,
            predictor_target="c6g.xlarge",
            predictor_num_threads=4,
            ram=8,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.2xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=8,
            hyperthreading=False,
            predictor_target="c6g.2xlarge",
            predictor_num_threads=8,
            ram=16,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.4xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=16,
            hyperthreading=False,
            predictor_target="c6g.4xlarge",
            predictor_num_threads=16,
            ram=32,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.8xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=32,
            hyperthreading=False,
            predictor_target="c6g.8xlarge",
            predictor_num_threads=32,
            ram=64,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.12xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=48,
            hyperthreading=False,
            predictor_target="c6g.12xlarge",
            predictor_num_threads=48,
            ram=96,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6g.16xlarge",
            family="c6g",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=64,
            hyperthreading=False,
            predictor_target="c6g.16xlarge",
            predictor_num_threads=64,
            ram=128,
            cpu_arch="arm64",
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.medium",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=1,
            hyperthreading=False,
            predictor_target="c6gn.medium",
            predictor_num_threads=1,
            ram=2,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.large",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=2,
            hyperthreading=False,
            predictor_target="c6gn.large",
            predictor_num_threads=2,
            ram=4,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=4,
            hyperthreading=False,
            predictor_target="c6gn.xlarge",
            predictor_num_threads=4,
            ram=8,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.2xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=8,
            hyperthreading=False,
            predictor_target="c6gn.2xlarge",
            predictor_num_threads=8,
            ram=16,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.4xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=16,
            hyperthreading=False,
            predictor_target="c6gn.4xlarge",
            predictor_num_threads=16,
            ram=32,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.8xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=32,
            hyperthreading=False,
            predictor_target="c6gn.8xlarge",
            predictor_num_threads=32,
            ram=64,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.12xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=48,
            hyperthreading=False,
            predictor_target="c6gn.12xlarge",
            predictor_num_threads=48,
            ram=96,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c6gn.16xlarge",
            family="c6gn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/c6g/",
            hw="Amazon Graviton2",
            cores=64,
            hyperthreading=False,
            predictor_target="c6gn.16xlarge",
            predictor_num_threads=64,
            ram=128,
            cpu_arch="arm64",
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c4.large",
            family="c4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/",
            hw="Intel Xeon",
            cores=2,
            hyperthreading=True,
            predictor_target="c4.large",
            predictor_num_threads=2,
            ram=3.75,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c4.xlarge",
            family="c4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/",
            hw="Intel Xeon",
            cores=4,
            hyperthreading=True,
            predictor_target="c4.xlarge",
            predictor_num_threads=4,
            ram=7.5,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c4.2xlarge",
            family="c4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/",
            hw="Intel Xeon",
            cores=8,
            hyperthreading=True,
            predictor_target="c4.2xlarge",
            predictor_num_threads=8,
            ram=15,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c4.4xlarge",
            family="c4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/",
            hw="Intel Xeon",
            cores=16,
            hyperthreading=True,
            predictor_target="c4.4xlarge",
            predictor_num_threads=16,
            ram=30,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="c4.8xlarge",
            family="c4",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/",
            hw="Intel Xeon",
            cores=36,
            hyperthreading=True,
            predictor_target="c4.8xlarge",
            predictor_num_threads=36,
            ram=60,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g4ad.2xlarge",
            family="g4ad",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g4/",
            hw="NVIDIA T4 GPUs",
            cores=8,
            hyperthreading=False,
            hardcoded_usd_per_hr=0.752,
            predictor_target="g4ad.2xlarge",
            predictor_num_threads=8,
            ram=32,
            gpuram=16,
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="p3.2xlarge",
            family="p3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/p3/",
            hw="NVIDIA V100 GPUs",
            cores=8,
            hyperthreading=False,
            hardcoded_usd_per_hr=3.06,
            predictor_target="p3.2xlarge",
            predictor_num_threads=8,
            ram=61,
            gpuram=16,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="p2.xlarge",
            family="p2",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/p2/",
            hw="NVIDIA K80 GPUs",
            cores=4,
            hyperthreading=False,
            predictor_target="p2.xlarge",
            predictor_num_threads=4,
            ram=61,
            gpuram=24,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="p3.8xlarge",
            family="p3",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/p3/",
            hw="NVIDIA V100 GPUs",
            cores=32,
            hyperthreading=False,
            predictor_target="p3.8xlarge",
            predictor_num_threads=16,
            ram=244,
            gpuram=64,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g3s.xlarge",
            family="g3s",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g3/",
            hw="NVIDIA Tesla M60 GPU",
            cores=2,
            hyperthreading=False,
            predictor_target="g3s.xlarge",
            predictor_num_threads=4,
            ram=30.5,
            gpuram=16,
            aws_govcloud="US West only",
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g4dn.xlarge",
            family="g4dn",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g4/",
            hw="NVIDIA T4 GPU",
            cores=4,
            hyperthreading=False,
            predictor_target="g4dn.xlarge",
            predictor_num_threads=4,
            ram=16,
            gpuram=16,
            aws_govcloud=True,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=4,
            hyperthreading=False,
            predictor_target="g5.xlarge",
            predictor_num_threads=4,
            ram=16,
            gpuram=24,
            aws_govcloud=False,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.2xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=8,
            hyperthreading=False,
            predictor_target="g5.2xlarge",
            predictor_num_threads=4,
            ram=32,
            gpuram=24,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.4xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=16,
            hyperthreading=False,
            predictor_target="g5.4xlarge",
            predictor_num_threads=8,
            ram=64,
            gpuram=24,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.8xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=32,
            hyperthreading=False,
            predictor_target="g5.8xlarge",
            predictor_num_threads=16,
            ram=128,
            gpuram=24,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.12xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=48,
            hyperthreading=False,
            predictor_target="g5.12xlarge",
            predictor_num_threads=24,
            ram=192,
            gpuram=24 * 4,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.16xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=64,
            hyperthreading=False,
            predictor_target="g5.16xlarge",
            predictor_num_threads=32,
            ram=256,
            gpuram=24,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.24xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=96,
            hyperthreading=False,
            predictor_target="g5.24xlarge",
            predictor_num_threads=48,
            ram=384,
            gpuram=24 * 4,
        ),
        **amazon_cloud_info_service.build_keyed_instance_description(
            label="g5.48xlarge",
            family="g5",
            instance_description_link="https://aws.amazon.com/ec2/instance-types/g5/",
            hw="NVIDIA A10G",
            cores=192,
            hyperthreading=False,
            predictor_target="g5.48xlarge",
            predictor_num_threads=96,
            ram=768,
            gpuram=24 * 8,
        ),
    }


def get_supported_oracle_cloud_instances(
    oracle_cloud_info_service: CloudInfoService,
):
    return {
        "VM.Standard.A1.Flex": {
            2: InstanceDescription(
                label="VM.Standard.A1.Flex 2vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.026,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=2,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            4: InstanceDescription(
                label="VM.Standard.A1.Flex 4vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=4,
                hyperthreading=False,
                usd_per_hr=0.052,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            8: InstanceDescription(
                label="VM.Standard.A1.Flex 8vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=8,
                hyperthreading=False,
                usd_per_hr=0.104,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            16: InstanceDescription(
                label="VM.Standard.A1.Flex 16vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=16,
                hyperthreading=False,
                usd_per_hr=0.208,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=16,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            32: InstanceDescription(
                label="VM.Standard.A1.Flex 32vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=32,
                hyperthreading=False,
                usd_per_hr=0.416,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=32,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            48: InstanceDescription(
                label="VM.Standard.A1.Flex 48vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=48,
                hyperthreading=False,
                usd_per_hr=0.624,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=48,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            64: InstanceDescription(
                label="VM.Standard.A1.Flex 64vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=64,
                hyperthreading=False,
                usd_per_hr=0.832,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=64,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
            80: InstanceDescription(
                label="VM.Standard.A1.Flex 80vCPU",
                family="VM.Standard.A1.Flex",
                instance_description_link="https://www.oracle.com/cloud/compute/arm/",
                hw="Ampere Altra",
                cores=80,
                hyperthreading=False,
                usd_per_hr=1.040,
                predictor_target="VM.Standard.A1.Flex",
                predictor_num_threads=80,
                service_provider=CloudServiceProvider.ORACLE,
                ram=None,
                gpuram=None,
                aws_govcloud=False,
            ),
        },
    }


def get_supported_azure_cloud_instances(
    azure_cloud_info_service: CloudInfoService,
):
    return {
        # TODO: why these instances can have 8 threads support with just 2 phys cores?
        "Azure.StandardD2sv3": {
            8: InstanceDescription(
                label="Azure.StandardD2sv3",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv3-dsv3-series",
                hw="Intel Xeon",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardD2sv3",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardDS2v2": {
            8: InstanceDescription(
                label="Azure.StandardD2sv2",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv2-dsv2-series",
                hw="Intel Xeon",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardDS2v2",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardB2s": {
            8: InstanceDescription(
                label="Azure.StandardB2s",
                family="B2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/sizes-b-series-burstable",
                hw="Intel Xeon",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardB2s",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardB2ms": {
            8: InstanceDescription(
                label="Azure.StandardB2ms",
                family="B2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/sizes-b-series-burstable",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardB2ms",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2sv4": {
            8: InstanceDescription(
                label="Azure.StandardD2sv4",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv4-dsv4-series",
                hw="Intel Xeon",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardD2sv4",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2sv5": {
            8: InstanceDescription(
                label="Azure.StandardD2sv5",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv5-dsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.09,
                predictor_target="Azure.StandardD2sv5",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardDS1v2": {
            2: InstanceDescription(
                label="Azure.StandardD1sv2",
                family="D1",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv2-dsv2-series",
                hw="Intel Xeon",
                cores=1,
                hyperthreading=False,
                usd_per_hr=0.073,
                predictor_target="Azure.StandardDS1v2",  # TODO: typo?
                predictor_num_threads=2,
                service_provider=CloudServiceProvider.AZURE,
                ram=3.5,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardB4ms": {
            8: InstanceDescription(
                label="Azure.StandardB4ms",
                family="B4",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/sizes-b-series-burstable",
                hw="Intel Xeon Platinum",
                cores=4,
                hyperthreading=False,
                usd_per_hr=0.166,
                predictor_target="Azure.StandardB4ms",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2sv3": {
            8: InstanceDescription(
                label="Azure.StandardE2sv3",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/ev3-esv3-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.126,
                predictor_target="Azure.StandardE2sv3",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardF2sv2": {
            8: InstanceDescription(
                label="Azure.StandardF2sv2",
                family="F2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/fsv2-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.0846,
                predictor_target="Azure.StandardF2sv2",
                predictor_num_threads=8,
                service_provider=CloudServiceProvider.AZURE,
                ram=4,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardDS11-1v2": {
            4: InstanceDescription(
                label="Azure.StandardDS11-1v2",
                family="DS11",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/constrained-vcpu",
                hw="Intel Xeon",
                cores=1,
                hyperthreading=False,
                usd_per_hr=0.185,
                predictor_target="Azure.StandardDS11-1v2",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=14,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardDS11v2": {
            4: InstanceDescription(
                label="Azure.StandardDS11v2",
                family="DS11",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dv2-dsv2-series-memory",
                hw="Intel Xeon",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.185,
                predictor_target="Azure.StandardDS11v2",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=14,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2s_v4": {
            4: InstanceDescription(
                label="Azure.StandardE2sv4",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/ev4-esv4-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.126,
                predictor_target="Azure.StandardE2s_v4",  # TODO inconsistent
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2dsv4": {
            4: InstanceDescription(
                label="Azure.StandardE2dsv4",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/edv4-edsv4-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.144,
                predictor_target="Azure.StandardE2dsv4",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2asv4": {
            4: InstanceDescription(
                label="Azure.StandardE2asv4",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/eav4-easv4-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.126,
                predictor_target="Azure.StandardE2asv4",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2sv5": {
            4: InstanceDescription(
                label="Azure.StandardE2sv5",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/ev5-esv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.126,
                predictor_target="Azure.StandardE2sv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2dsv5": {
            4: InstanceDescription(
                label="Azure.StandardE2dsv5",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/edv5-edsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.144,
                predictor_target="Azure.StandardE2dsv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2asv5": {
            4: InstanceDescription(
                label="Azure.StandardE2asv5",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/easv5-eadsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.1129,
                predictor_target="Azure.StandardE2asv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardE2adsv5": {
            4: InstanceDescription(
                label="Azure.StandardE2adsv5",
                family="E2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/easv5-eadsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.1309,
                predictor_target="Azure.StandardE2adsv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=16,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2dsv4": {
            4: InstanceDescription(
                label="Azure.StandardD2dsv4",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/ddv4-ddsv4-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.1129,
                predictor_target="Azure.StandardD2dsv4",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2asv4": {
            4: InstanceDescription(
                label="Azure.StandardD2asv4",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dav4-dasv4-series",
                hw="Intel Xeon Platinum",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.096,
                predictor_target="Azure.StandardD2asv4",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2dsv5": {
            4: InstanceDescription(
                label="Azure.StandardD2dsv5",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/ddv5-ddsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.1129,
                predictor_target="Azure.StandardD2dsv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2asv5": {
            4: InstanceDescription(
                label="Azure.StandardD2asv5",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dasv5-dadsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.086,
                predictor_target="Azure.StandardD2asv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
        "Azure.StandardD2adsv5": {
            4: InstanceDescription(
                label="Azure.StandardD2adsv5",
                family="D2",
                instance_description_link="https://docs.microsoft.com/en-us/azure/virtual-machines/dasv5-dadsv5-series",
                hw="Intel Xeon Platinum Ice Lake",
                cores=2,
                hyperthreading=False,
                usd_per_hr=0.103,
                predictor_target="Azure.StandardD2adsv5",
                predictor_num_threads=4,
                service_provider=CloudServiceProvider.AZURE,
                ram=8,
                gpuram=0,
                aws_govcloud=False,
            )
        },
    }


def get_supported_google_cloud_instances(
    google_cloud_info_service: CloudInfoService,
):
    return {
        # TODO: rename all predictor targets here to label values? then change 'label' used in Rafal's/GCP code back to 'predictor_target'
        **google_cloud_info_service.build_keyed_instance_description(
            label="c2-standard-4",
            family="C2",
            instance_description_link="https://cloud.google.com/compute/docs/compute-optimized-machines#c2_machine_types",
            hw="Intel Cascade Lake CPU",
            cores=4,
            hyperthreading=False,
            predictor_target="gcp.C2",
            predictor_num_threads=8,
            ram=16,
            gpuram=0,
        ),
        **google_cloud_info_service.build_keyed_instance_description(
            label="c2-standard-8",
            family="C2",
            instance_description_link="https://cloud.google.com/compute/docs/compute-optimized-machines#c2_machine_types",
            hw="Intel Cascade Lake CPU",
            cores=8,
            hyperthreading=False,
            predictor_target="gcp.C2_8cores",
            predictor_num_threads=16,
            ram=32,
            gpuram=0,
        ),
        **google_cloud_info_service.build_keyed_instance_description(
            label="e2-standard-2",
            family="E2",
            instance_description_link="https://cloud.google.com/compute/docs/general-purpose-machines#e2_machine_types",
            hw="Intel Skylake CPU",
            cores=2,
            hyperthreading=False,
            predictor_target="gcp.E2_2cores_8gb",
            predictor_num_threads=4,
            ram=8,
            gpuram=0,
        ),
        **google_cloud_info_service.build_keyed_instance_description(
            label="e2-standard-4",
            family="E2",
            instance_description_link="https://cloud.google.com/compute/docs/general-purpose-machines#e2_machine_types",
            hw="Intel Skylake CPU",
            cores=4,
            hyperthreading=False,
            predictor_target="gcp.e2_4cores",
            predictor_num_threads=8,
            ram=16,
            gpuram=0,
        ),
        **google_cloud_info_service.build_keyed_instance_description(
            label="e2-standard-8",
            family="E2",
            instance_description_link="https://cloud.google.com/compute/docs/general-purpose-machines#e2_machine_types",
            hw="Intel Skylake CPU",
            cores=8,
            hyperthreading=False,
            predictor_target="gcp.e2_8cores",
            predictor_num_threads=16,
            ram=32,
            gpuram=0,
        ),
        **google_cloud_info_service.build_keyed_instance_description(
            label="e2-standard-16",
            family="E2",
            instance_description_link="https://cloud.google.com/compute/docs/general-purpose-machines#e2_machine_types",
            hw="Intel Skylake CPU",
            cores=16,
            hyperthreading=False,
            predictor_target="gcp.e2_16cores",
            predictor_num_threads=32,
            ram=64,
            gpuram=0,
        ),
    }
