# Copyright 2019 Splunk Inc. All rights reserved.

"""
### Malware, viruses, malicious content, user security standards (static checks)
"""

# Python Standard Libraries
from builtins import str as text
import logging
import os.path
import re
import platform
import ipaddress

# Third-Party Libraries
# N/A
# Custom Libraries
import splunk_appinspect

logger = logging.getLogger(__name__)
report_display_order = 5


@splunk_appinspect.tags("splunk_appinspect", "offensive", "manual")
@splunk_appinspect.cert_version(min="1.0.0")
def check_for_offensive_material(app, reporter):
    """Check that the app does not include any offensive material."""
    if platform.system() == "Windows":
        reporter.manual_check(
            "Offensive language checks will be done manually during code review."
        )
    else:
        # False positives? False negatives?  You betcha. But it's a start. (And
        # that's why we warn either way.)
        for directory, filename, _ in app.iterate_files():
            file_path = os.path.join(directory, filename)
            for lineno, line, found, _ in splunk_appinspect.offense.scan_file(
                app.get_filename(directory, filename)
            ):
                formatted = line.replace(found, "<<<" + found.upper() + ">>>")
                if len(formatted) > 65:
                    formatted = formatted[:65] + "..."
                reporter.manual_check(
                    "{0} ({1}:{2}) [{3}]. File: {1}, Line: {2}.".format(
                        formatted, file_path, lineno, found
                    ),
                    file_path,
                    lineno,
                )


@splunk_appinspect.tags("splunk_appinspect", "manual")
@splunk_appinspect.cert_version(min="1.0.0")
def check_embedded_links(reporter):
    """Check that embedded links included in the app are not malicious."""
    reporter.manual_check("Links will be manually inspected during code review.")


@splunk_appinspect.tags("splunk_appinspect", "manual")
@splunk_appinspect.cert_version(min="1.0.0")
def check_authorization_credentials(reporter):
    """Check that no plain text authorization credentials are stored in the
    app.
    """
    reporter.manual_check("Code will be manually inspected during code review.")


@splunk_appinspect.tags("splunk_appinspect", "cloud")
@splunk_appinspect.cert_version(min="1.0.0")
def check_hostnames_and_ips(app, reporter):
    """Check that no sensitive hostnames/IPs are stored in the app."""

    PUBLIC_DNS_LIST = [
        "208.67.222.222",
        "208.67.220.220",  # OpenDNS
        "1.1.1.1",
        "1.0.0.1",  # Cloudflare
        "8.8.8.8",
        "8.8.4.4",  # Google Public DNS
        "199.85.126.10",
        "199.85.127.10",  # Norton ConnectSafe
        "8.26.56.26",
        "8.20.247.20",  # Comodo Secure DNS
        "9.9.9.9",
        "149.112.112.112",  # Quad9
    ]
    SAFE_PRIVATE_IP_LIST = ["0.0.0.0", "127.0.0.1", "255.255.255.255"]

    results = app.search_for_pattern(
        "(^|[^\\d\\w\\.])(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})(?!/[1-3]?\\d($|[^\\d\\w\\.]))($|[^\\d\\w\\.])"
    )

    distinct_ip = {}
    for result in results:
        ip = result[1].groups()[1]
        address = None
        try:
            address = ipaddress.ip_address(text(ip))
        except ValueError:
            pass

        if address:
            filename, lineno = result[0].rsplit(":", 1)
            if address.is_loopback:
                continue
            if ip in SAFE_PRIVATE_IP_LIST or ip in PUBLIC_DNS_LIST:
                continue
            if re.search(
                "((requests|botocore|boto)/utils|scopes|socks|ipv4|dns/(query|resolver)|ipaddr|backports/socket|ipaddress(/__init__)?|ip/__init__)\\.py",
                filename,
            ):
                continue

            if distinct_ip.get(filename) is None:
                distinct_ip[filename] = set()

            if ip in distinct_ip[filename]:
                continue
            if len(distinct_ip[filename]) == 100:
                continue

            if address.is_global:
                ip_type = "PUBLIC"
            elif address.is_private:
                ip_type = "PRIVATE"
            elif address.is_reserved:
                ip_type = "RESERVED"
            elif address.is_multicast:
                ip_type = "MULTICASE"
            elif address.is_unspecified:
                ip_type = "UNSPECIFIED"
            else:
                ip_type = "UNKNOWN"

            distinct_ip[filename].add(ip)
            report = "{} IP {} is found in {}:{}".format(ip_type, ip, filename, lineno)
            reporter.warn(report, filename, lineno)
