# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------------------

"""
This file contains commonly accessed data for pure Elements. If you would like to
view all of this data in a friendly format, you should look at element_data.csv.
It's the same exact data, but in a cleaner CSV table.

All data was ported from /pymatgen/core/periodic_table.json, where we host the
data directly in python rather than JSON for speed.

While Simmate typically prefers CSV or SQL, we use JSON/dict format
here because there is a performance boost with the lazy-loading style that the Element
base class uses. This file is therefore an exception to our coding philophosy due
the specific use-case.
"""


# --------------------------------------------------------------------------------------


"""
There are a total of 42 potential attributes for each element. These are stored in
a dictionary with each element's symbol as the key (e.g. "H" or "Fe"). Here is a list of
all the possible fields and any extra information for each (such as units):
"""

SUPPORTED_PROPERTIES = [
    "atomic_mass",  # (amu)
    "atomic_number",
    "atomic_orbitals",  # (each orbital with it's energy in eV)
    # https://www.nist.gov/pml/data/atomic-reference-data-electronic-structure-calculations
    # The LDA values for neutral atoms are used
    "atomic_radius",  # (Angstroms)
    "atomic_radius_calculated",  # (Angstroms)
    # http://en.wikipedia.org/wiki/Atomic_radii_of_the_elements_(data_page)
    "boiling_point",  # (Kelvin)
    "brinell_hardness",  # (MN m^-2)
    "bulk_modulus",  # (GPa)
    "coefficient_of_linear_thermal_expansion",  # (x10^-6 K^-1)
    "common_oxidation_states",
    "critical_temperature",  # (Kelvin)
    "density_of_solid",  # (kg m^-3)
    "electrical_resistivity",  # (10^-8 Ohm m)
    "electronegativity",
    "electronic_structure",
    "icsd_oxidation_states",
    "ionic_radii",
    "ionic_radii_hs",
    "ionic_radii_ls",
    "iupac_ordering",
    "liquid_range",  # (Kelvin)
    "max_oxidation_state",
    "melting_point",  # (Kelvin)
    "mendeleev_number",
    # Mendeleev number from definition given by Pettifor, D. G. (1984).
    # A chemical scale for crystal-structure maps. Solid State Communications,
    # 51 (1), 31-34
    "metallic_radius",
    "min_oxidation_state",
    "mineral_hardness",  # (unitless)
    "molar_volume",  # (cm^3)
    "name",  # OPTIMIZE --> this is built-in by Enum, but is slow. Consider a "name_fast" attr
    "nmr_quadrupole_moment",
    "oxidation_states",
    "poissons_ratio",  # (unitless)
    "reflectivity",  # (%)
    "refractive_index",  # (unitless)
    "rigidity_modulus",  # (GPa)
    "shannon_radii",
    "superconduction_temperature",  # (Kelvin)
    "thermal_conductivity",  # (W m^-1 K^-1)
    "van_der_waals_radius",  # (Angstroms)
    # http://en.wikipedia.org/wiki/Atomic_radii_of_the_elements_(data_page).
    "velocity_of_sound",  # (m s^-1)
    "vickers_hardness",  # (MN m^-2)
    "youngs_modulus",  # (GPa)
]

"""
These are the attributes that pymatgen has but we are missing at the moment. There are
others, but I think these are the ones worth adding:
    ionization_energies
    mendeleev_no
    valence
    ground_level
    row_number
    group_number
    block (s,p,d,f)
    term_symbols
    ground_state_term_symbol
"""

# --------------------------------------------------------------------------------------


ALL_DATA = {
    "Ac": {
        "atomic_mass": 227.0,
        "atomic_number": 89,
        "atomic_orbitals": {
            "1s": -3443.110367,
            "2p": -572.7627,
            "2s": -592.622878,
            "3d": -119.541743,
            "3p": -137.654394,
            "3s": -147.320716,
            "4d": -23.57061,
            "4f": -12.278225,
            "4p": -31.761846,
            "4s": -36.15826,
            "5d": -3.222752,
            "5p": -6.06511,
            "5s": -7.713078,
            "6d": -0.137786,
            "6p": -0.744524,
            "6s": -1.19698,
            "7s": -0.126551,
        },
        "atomic_radius": 1.95,
        "boiling_point": 3573,
        "common_oxidation_states": [3],
        "density_of_solid": 10070,
        "electronic_structure": "[Rn].6d1.7s2",
        "ionic_radii": {"3": 1.26},
        "liquid_range": 2250,
        "melting_point": 1323,
        "mendeleev_number": 48,
        "molar_volume": 22.55,
        "name": "Actinium",
        "oxidation_states": [3],
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.26, "ionic_radius": 1.12}}}
        },
        "thermal_conductivity": 12,
        "electronegativity": 1.1,
        "metallic_radius": 1.878,
        "iupac_ordering": 32,
    },
    "Ag": {
        "atomic_mass": 107.8682,
        "atomic_number": 47,
        "atomic_orbitals": {
            "1s": -900.324578,
            "2p": -120.913351,
            "2s": -129.859807,
            "3d": -13.367803,
            "3p": -20.06763,
            "3s": -23.678437,
            "4d": -0.298706,
            "4p": -2.086602,
            "4s": -3.22309,
            "5s": -0.157407,
        },
        "atomic_radius": 1.6,
        "atomic_radius_calculated": 1.65,
        "boiling_point": 2435,
        "brinell_hardness": 24.5,
        "bulk_modulus": 100,
        "coefficient_of_linear_thermal_expansion": 18.9,
        "common_oxidation_states": [1],
        "density_of_solid": 10490,
        "electrical_resistivity": 1.63,
        "electronic_structure": "[Kr].4d10.5s1",
        "icsd_oxidation_states": [1, 2, 3],
        "ionic_radii": {"1": 1.29, "2": 1.08, "3": 0.89},
        "liquid_range": 1200.07,
        "melting_point": 1234.93,
        "mendeleev_number": 71,
        "mineral_hardness": 2.5,
        "molar_volume": 10.27,
        "name": "Silver",
        "oxidation_states": [1, 2, 3],
        "poissons_ratio": 0.37,
        "reflectivity": 97,
        "rigidity_modulus": 30,
        "shannon_radii": {
            "1": {
                "II": {"": {"crystal_radius": 0.81, "ionic_radius": 0.67}},
                "IV": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}},
                "IVSQ": {"": {"crystal_radius": 1.16, "ionic_radius": 1.02}},
                "V": {"": {"crystal_radius": 1.23, "ionic_radius": 1.09}},
                "VI": {"": {"crystal_radius": 1.29, "ionic_radius": 1.15}},
                "VII": {"": {"crystal_radius": 1.36, "ionic_radius": 1.22}},
                "VIII": {"": {"crystal_radius": 1.42, "ionic_radius": 1.28}},
            },
            "2": {
                "IVSQ": {"": {"crystal_radius": 0.93, "ionic_radius": 0.79}},
                "VI": {"": {"crystal_radius": 1.08, "ionic_radius": 0.94}},
            },
            "3": {
                "IVSQ": {"": {"crystal_radius": 0.81, "ionic_radius": 0.67}},
                "VI": {"": {"crystal_radius": 0.89, "ionic_radius": 0.75}},
            },
        },
        "thermal_conductivity": 430,
        "van_der_waals_radius": 1.72,
        "velocity_of_sound": 2600,
        "vickers_hardness": 251,
        "electronegativity": 1.93,
        "youngs_modulus": 83,
        "metallic_radius": 1.445,
        "iupac_ordering": 72,
    },
    "Al": {
        "atomic_mass": 26.9815386,
        "atomic_number": 13,
        "atomic_orbitals": {
            "1s": -55.156044,
            "2p": -2.564018,
            "2s": -3.934827,
            "3p": -0.102545,
            "3s": -0.286883,
        },
        "atomic_radius": 1.25,
        "atomic_radius_calculated": 1.18,
        "boiling_point": 2792,
        "brinell_hardness": 245,
        "bulk_modulus": 76,
        "coefficient_of_linear_thermal_expansion": 23.1,
        "common_oxidation_states": [3],
        "density_of_solid": 2700,
        "electrical_resistivity": 2.7,
        "electronic_structure": "[Ne].3s2.3p1",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 0.675},
        "liquid_range": 1858.53,
        "melting_point": 933.47,
        "mendeleev_number": 80,
        "mineral_hardness": 2.75,
        "molar_volume": 10.00,
        "name": "Aluminum",
        "oxidation_states": [1, 3],
        "poissons_ratio": 0.35,
        "reflectivity": 71,
        "rigidity_modulus": 26,
        "shannon_radii": {
            "3": {
                "IV": {"": {"crystal_radius": 0.53, "ionic_radius": 0.39}},
                "V": {"": {"crystal_radius": 0.62, "ionic_radius": 0.48}},
                "VI": {"": {"crystal_radius": 0.675, "ionic_radius": 0.535}},
            }
        },
        "superconduction_temperature": 1.175,
        "thermal_conductivity": 235,
        "van_der_waals_radius": 1.84,
        "velocity_of_sound": 5100,
        "vickers_hardness": 167,
        "electronegativity": 1.61,
        "youngs_modulus": 70,
        "nmr_quadrupole_moment": {"Al-27": 146.6},
        "metallic_radius": 1.43,
        "iupac_ordering": 80,
    },
    "Am": {
        "atomic_mass": 243.0,
        "atomic_number": 95,
        "atomic_radius": 1.75,
        "boiling_point": 2880,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f7.7s2",
        "ionic_radii": {"2": 1.4, "3": 1.115, "4": 0.99},
        "liquid_range": 1431,
        "melting_point": 1449,
        "mendeleev_number": 42,
        "molar_volume": 17.63,
        "name": "Americium",
        "oxidation_states": [2, 3, 4, 5, 6],
        "shannon_radii": {
            "2": {
                "VII": {"": {"crystal_radius": 1.35, "ionic_radius": 1.21}},
                "VIII": {"": {"crystal_radius": 1.4, "ionic_radius": 1.26}},
                "IX": {"": {"crystal_radius": 1.45, "ionic_radius": 1.31}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.115, "ionic_radius": 0.975}},
                "VIII": {"": {"crystal_radius": 1.23, "ionic_radius": 1.09}},
            },
            "4": {
                "VI": {"": {"crystal_radius": 0.99, "ionic_radius": 0.85}},
                "VIII": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
            },
        },
        "superconduction_temperature": 0.6,
        "thermal_conductivity": 10,
        "electronegativity": 1.3,
        "metallic_radius": 1.73,
        "iupac_ordering": 26,
    },
    "Ar": {
        "atomic_mass": 39.948,
        "atomic_number": 18,
        "atomic_orbitals": {
            "1s": -113.800134,
            "2p": -8.443439,
            "2s": -10.794172,
            "3p": -0.38233,
            "3s": -0.883384,
        },
        "atomic_radius": 0.71,
        "atomic_radius_calculated": 0.71,
        "boiling_point": 87.3,
        "critical_temperature": 150.8,
        "electronic_structure": "[Ne].3s2.3p6",
        "liquid_range": 3.5,
        "max_oxidation_state": 0.0,
        "melting_point": 83.8,
        "mendeleev_number": 3,
        "min_oxidation_state": 0.0,
        "molar_volume": 22.56,
        "name": "Argon",
        "refractive_index": 1.000281,
        "thermal_conductivity": 0.01772,
        "van_der_waals_radius": 1.88,
        "velocity_of_sound": 319,
        "iupac_ordering": 3,
    },
    "As": {
        "atomic_mass": 74.9216,
        "atomic_number": 33,
        "atomic_orbitals": {
            "1s": -423.336658,
            "2p": -47.527869,
            "2s": -53.093086,
            "3d": -1.542767,
            "3p": -4.851725,
            "3s": -6.730755,
            "4p": -0.197497,
            "4s": -0.52367,
        },
        "atomic_radius": 1.15,
        "atomic_radius_calculated": 1.14,
        "boiling_point": 887,
        "brinell_hardness": 1440,
        "bulk_modulus": 22,
        "common_oxidation_states": [-3, 3, 5],
        "critical_temperature": 1700,
        "density_of_solid": 5727,
        "electrical_resistivity": 33,
        "electronic_structure": "[Ar].3d10.4s2.4p3",
        "icsd_oxidation_states": [2, 3, 5, -2, -3, -1],
        "ionic_radii": {"3": 0.72, "5": 0.6},
        "liquid_range": 203,
        "melting_point": 1090,
        "mendeleev_number": 89,
        "mineral_hardness": 3.5,
        "molar_volume": 12.95,
        "name": "Arsenic",
        "oxidation_states": [-3, 2, 3, 5],
        "refractive_index": 1.001552,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}}},
            "5": {
                "IV": {"": {"crystal_radius": 0.475, "ionic_radius": 0.335}},
                "VI": {"": {"crystal_radius": 0.6, "ionic_radius": 0.46}},
            },
        },
        "thermal_conductivity": 50,
        "van_der_waals_radius": 1.85,
        "electronegativity": 2.18,
        "youngs_modulus": 8,
        "iupac_ordering": 89,
    },
    "At": {
        "atomic_mass": 210.0,
        "atomic_number": 85,
        "atomic_orbitals": {
            "1s": -3127.390276,
            "2p": -513.044243,
            "2s": -531.81835,
            "3d": -103.060375,
            "3p": -119.995013,
            "3s": -129.035542,
            "4d": -18.295162,
            "4f": -8.063483,
            "4p": -25.778264,
            "4s": -29.809515,
            "5d": -1.643758,
            "5p": -4.027061,
            "5s": -5.453383,
            "6p": -0.255453,
            "6s": -0.560189,
        },
        "common_oxidation_states": [-1, 1],
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p5",
        "ionic_radii": {"7": 0.76},
        "melting_point": 575,
        "mendeleev_number": 96,
        "name": "Astatine",
        "oxidation_states": [-1, 1, 3, 5],
        "shannon_radii": {
            "7": {"VI": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}}}
        },
        "thermal_conductivity": 2,  # estimate
        "van_der_waals_radius": 2.02,
        "electronegativity": 2.2,
        "iupac_ordering": 98,
    },
    "Au": {
        "atomic_mass": 196.966569,
        "atomic_number": 79,
        "atomic_orbitals": {
            "1s": -2683.508245,
            "2p": -430.725701,
            "2s": -447.888973,
            "3d": -81.511751,
            "3p": -96.707,
            "3s": -104.824516,
            "4d": -12.131815,
            "4f": -3.486824,
            "4p": -18.578652,
            "4s": -22.078357,
            "5d": -0.304738,
            "5p": -2.002495,
            "5s": -3.113936,
            "6s": -0.162334,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.74,
        "boiling_point": 3129,
        "brinell_hardness": 2450,
        "bulk_modulus": 220,
        "coefficient_of_linear_thermal_expansion": 14.2,
        "common_oxidation_states": [3],
        "density_of_solid": 19300,
        "electrical_resistivity": 2.2,
        "electronic_structure": "[Xe].4f14.5d10.6s1",
        "ionic_radii": {"1": 1.51, "3": 0.99, "5": 0.71},
        "liquid_range": 1791.67,
        "melting_point": 1337.33,
        "mendeleev_number": 70,
        "mineral_hardness": 2.5,
        "molar_volume": 10.21,
        "name": "Gold",
        "oxidation_states": [-1, 1, 2, 3, 5],
        "poissons_ratio": 0.44,
        "reflectivity": 95,
        "rigidity_modulus": 27,
        "shannon_radii": {
            "1": {"VI": {"": {"crystal_radius": 1.51, "ionic_radius": 1.37}}},
            "3": {
                "IVSQ": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}},
                "VI": {"": {"crystal_radius": 0.99, "ionic_radius": 0.85}},
            },
            "5": {"VI": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}}},
        },
        "thermal_conductivity": 320,
        "van_der_waals_radius": 1.66,
        "velocity_of_sound": 1740,
        "vickers_hardness": 216,
        "electronegativity": 2.54,
        "youngs_modulus": 78,
        "metallic_radius": 1.442,
        "iupac_ordering": 71,
    },
    "B": {
        "atomic_mass": 10.811,
        "atomic_number": 5,
        "atomic_orbitals": {"1s": -6.564347, "2p": -0.136603, "2s": -0.344701},
        "atomic_radius": 0.85,
        "atomic_radius_calculated": 0.87,
        "boiling_point": 4200,
        "bulk_modulus": 320,
        "coefficient_of_linear_thermal_expansion": 6,
        "common_oxidation_states": [3],
        "density_of_solid": 2460,
        "electrical_resistivity": 1e12,  # greater than this
        "electronic_structure": "[He].2s2.2p1",
        "icsd_oxidation_states": [3, -3],
        "ionic_radii": {"3": 0.41},
        "liquid_range": 1851,
        "melting_point": 2349,
        "mendeleev_number": 86,
        "mineral_hardness": 9.3,
        "molar_volume": 4.39,
        "name": "Boron",
        "oxidation_states": [1, 2, 3],
        "shannon_radii": {
            "3": {
                "III": {"": {"crystal_radius": 0.15, "ionic_radius": 0.01}},
                "IV": {"": {"crystal_radius": 0.25, "ionic_radius": 0.11}},
                "VI": {"": {"crystal_radius": 0.41, "ionic_radius": 0.27}},
            }
        },
        "thermal_conductivity": 27,
        "van_der_waals_radius": 1.92,
        "velocity_of_sound": 16200,
        "vickers_hardness": 49000,
        "electronegativity": 2.04,
        "nmr_quadrupole_moment": {"B-10": 84.59, "B-11": 40.59},
        "iupac_ordering": 81,
    },
    "Ba": {
        "atomic_mass": 137.327,
        "atomic_number": 56,
        "atomic_orbitals": {
            "1s": -1305.743258,
            "2p": -189.598483,
            "2s": -200.844444,
            "3d": -28.528933,
            "3p": -37.536931,
            "3s": -42.359434,
            "4d": -3.432441,
            "4p": -6.497622,
            "4s": -8.257061,
            "5p": -0.698605,
            "5s": -1.157159,
            "6s": -0.118967,
        },
        "atomic_radius": 2.15,
        "atomic_radius_calculated": 2.53,
        "boiling_point": 2143,
        "bulk_modulus": 9.6,
        "coefficient_of_linear_thermal_expansion": 20.6,
        "common_oxidation_states": [2],
        "density_of_solid": 3510,
        "electrical_resistivity": 34,
        "electronic_structure": "[Xe].6s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 1.49},
        "liquid_range": 1143,
        "melting_point": 1000,
        "mendeleev_number": 14,
        "mineral_hardness": 1.25,
        "molar_volume": 38.16,
        "name": "Barium",
        "oxidation_states": [2],
        "rigidity_modulus": 4.9,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.49, "ionic_radius": 1.35}},
                "VII": {"": {"crystal_radius": 1.52, "ionic_radius": 1.38}},
                "VIII": {"": {"crystal_radius": 1.56, "ionic_radius": 1.42}},
                "IX": {"": {"crystal_radius": 1.61, "ionic_radius": 1.47}},
                "electronegativity": {
                    "": {"crystal_radius": 1.66, "ionic_radius": 1.52}
                },
                "XI": {"": {"crystal_radius": 1.71, "ionic_radius": 1.57}},
                "XII": {"": {"crystal_radius": 1.75, "ionic_radius": 1.61}},
            }
        },
        "thermal_conductivity": 18,
        "van_der_waals_radius": 2.68,
        "velocity_of_sound": 1620,
        "electronegativity": 0.89,
        "youngs_modulus": 13,
        "metallic_radius": 2.236,
        "iupac_ordering": 13,
    },
    "Be": {
        "atomic_mass": 9.012182,
        "atomic_number": 4,
        "atomic_orbitals": {"1s": -3.856411, "2s": -0.205744},
        "atomic_radius": 1.05,
        "atomic_radius_calculated": 1.12,
        "boiling_point": 2742,
        "brinell_hardness": 600,
        "bulk_modulus": 130,
        "coefficient_of_linear_thermal_expansion": 11.3,
        "common_oxidation_states": [2],
        "density_of_solid": 1848,
        "electrical_resistivity": 3.8,
        "electronic_structure": "[He].2s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 0.59},
        "liquid_range": 1182,
        "melting_point": 1560,
        "mendeleev_number": 77,
        "mineral_hardness": 5.5,
        "molar_volume": 4.85,
        "name": "Beryllium",
        "oxidation_states": [2],
        "poissons_ratio": 0.032,
        "rigidity_modulus": 132,
        "shannon_radii": {
            "2": {
                "III": {"": {"crystal_radius": 0.3, "ionic_radius": 0.16}},
                "IV": {"": {"crystal_radius": 0.41, "ionic_radius": 0.27}},
                "VI": {"": {"crystal_radius": 0.59, "ionic_radius": 0.45}},
            }
        },
        "superconduction_temperature": 0.026,
        "thermal_conductivity": 190,
        "van_der_waals_radius": 1.53,
        "velocity_of_sound": 13000,
        "vickers_hardness": 1670,
        "electronegativity": 1.57,
        "youngs_modulus": 287,
        "nmr_quadrupole_moment": {"Be-9": 52.88},
        "metallic_radius": 1.12,
        "iupac_ordering": 17,
    },
    "Bi": {
        "atomic_mass": 208.9804,
        "atomic_number": 83,
        "atomic_orbitals": {
            "1s": -2975.550959,
            "2p": -484.716359,
            "2s": -502.950758,
            "3d": -95.532476,
            "3p": -111.883393,
            "3s": -120.613998,
            "4d": -16.084817,
            "4f": -6.382744,
            "4p": -23.218641,
            "4s": -27.07034,
            "5d": -1.139408,
            "5p": -3.293637,
            "5s": -4.611934,
            "6p": -0.180198,
            "6s": -0.426129,
        },
        "atomic_radius": 1.6,
        "atomic_radius_calculated": 1.43,
        "boiling_point": 1837,
        "brinell_hardness": 94.2,
        "bulk_modulus": 31,
        "coefficient_of_linear_thermal_expansion": 13.4,
        "common_oxidation_states": [3],
        "density_of_solid": 9780,
        "electrical_resistivity": 130,
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p3",
        "icsd_oxidation_states": [1, 2, 3, 5],
        "ionic_radii": {"3": 1.17, "5": 0.9},
        "liquid_range": 1292.6,
        "melting_point": 544.4,
        "mendeleev_number": 87,
        "mineral_hardness": 2.25,
        "molar_volume": 21.31,
        "name": "Bismuth",
        "oxidation_states": [-3, 3, 5],
        "poissons_ratio": 0.33,
        "rigidity_modulus": 12,
        "shannon_radii": {
            "3": {
                "V": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
                "VI": {"": {"crystal_radius": 1.17, "ionic_radius": 1.03}},
                "VIII": {"": {"crystal_radius": 1.31, "ionic_radius": 1.17}},
            },
            "5": {"VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}}},
        },
        "thermal_conductivity": 8,
        "van_der_waals_radius": 2.07,
        "velocity_of_sound": 1790,
        "electronegativity": 2.02,
        "youngs_modulus": 32,
        "metallic_radius": 1.82,
        "iupac_ordering": 87,
    },
    "Bk": {
        "atomic_mass": 247.0,
        "atomic_number": 97,
        "common_oxidation_states": [3],
        "density_of_solid": 14780,
        "electronic_structure": "[Rn].5f9.7s2",
        "ionic_radii": {"3": 1.1, "4": 0.97},
        "melting_point": 1259,
        "mendeleev_number": 40,
        "molar_volume": 16.84,
        "name": "Berkelium",
        "oxidation_states": [3, 4],
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}}},
            "4": {
                "VI": {"": {"crystal_radius": 0.97, "ionic_radius": 0.83}},
                "VIII": {"": {"crystal_radius": 1.07, "ionic_radius": 0.93}},
            },
        },
        "thermal_conductivity": 10,
        "electronegativity": 1.3,
        "metallic_radius": 1.703,
        "iupac_ordering": 24,
    },
    "Br": {
        "atomic_mass": 79.904,
        "atomic_number": 35,
        "atomic_orbitals": {
            "1s": -480.182643,
            "2p": -55.67796,
            "2s": -61.710022,
            "3d": -2.52211,
            "3p": -6.298805,
            "3s": -8.409057,
            "4p": -0.295334,
            "4s": -0.720066,
        },
        "atomic_radius": 1.15,
        "atomic_radius_calculated": 0.94,
        "boiling_point": 332,
        "bulk_modulus": 1.9,
        "common_oxidation_states": [-1, 1, 3, 5, 7],
        "critical_temperature": 586,
        "electrical_resistivity": 1e18,  # greater than
        "electronic_structure": "[Ar].3d10.4s2.4p5",
        "icsd_oxidation_states": [5, -1],
        "ionic_radii": {"-1": 1.82, "3": 0.73, "5": 0.45, "7": 0.53},
        "liquid_range": 66.2,
        "melting_point": 265.8,
        "mendeleev_number": 98,
        "molar_volume": 19.78,
        "name": "Bromine",
        "oxidation_states": [-1, 1, 3, 4, 5, 7],
        "refractive_index": 1.001132,
        "shannon_radii": {
            "-1": {"VI": {"": {"crystal_radius": 1.82, "ionic_radius": 1.96}}},
            "3": {"IVSQ": {"": {"crystal_radius": 0.73, "ionic_radius": 0.59}}},
            "5": {"IIIPY": {"": {"crystal_radius": 0.45, "ionic_radius": 0.31}}},
            "7": {
                "IV": {"": {"crystal_radius": 0.39, "ionic_radius": 0.25}},
                "VI": {"": {"crystal_radius": 0.53, "ionic_radius": 0.39}},
            },
        },
        "thermal_conductivity": 0.12,
        "van_der_waals_radius": 1.85,
        "electronegativity": 2.96,
        "metallic_radius": 1.14,
        "iupac_ordering": 100,
    },
    "C": {
        "atomic_mass": 12.0107,
        "atomic_number": 6,
        "atomic_orbitals": {"1s": -9.947718, "2p": -0.199186, "2s": -0.500866},
        "atomic_radius": 0.7,
        "atomic_radius_calculated": 0.67,
        "boiling_point": 4300,
        "bulk_modulus": 33,
        "coefficient_of_linear_thermal_expansion": 7.1,
        "common_oxidation_states": [-4, 4],
        "density_of_solid": 2267,
        "electrical_resistivity": 1000,  # about this value. direction dependent
        "electronic_structure": "[He].2s2.2p2",
        "icsd_oxidation_states": [2, 3, 4, -4, -3, -2],
        "ionic_radii": {"4": 0.3},
        "liquid_range": 500,
        "melting_point": 3800,
        "mendeleev_number": 95,
        "mineral_hardness": 0.5,  # for graphite. diamond is 10.0
        "molar_volume": 5.29,
        "name": "Carbon",
        "oxidation_states": [-4, -3, -2, -1, 1, 2, 3, 4],
        "reflectivity": 27,
        "refractive_index": 2.417,  # for diamond
        "shannon_radii": {
            "4": {
                "III": {"": {"crystal_radius": 0.06, "ionic_radius": -0.08}},
                "IV": {"": {"crystal_radius": 0.29, "ionic_radius": 0.15}},
                "VI": {"": {"crystal_radius": 0.3, "ionic_radius": 0.16}},
            }
        },
        "thermal_conductivity": 140,
        "van_der_waals_radius": 1.7,
        "velocity_of_sound": 18350,
        "electronegativity": 2.55,
        "nmr_quadrupole_moment": {"C-11": 33.27},
        "iupac_ordering": 86,
    },
    "Ca": {
        "atomic_mass": 40.078,
        "atomic_number": 20,
        "atomic_orbitals": {
            "1s": -143.935181,
            "2p": -12.285376,
            "2s": -15.046905,
            "3p": -1.030572,
            "3s": -1.706331,
            "4s": -0.141411,
        },
        "atomic_radius": 1.8,
        "atomic_radius_calculated": 1.94,
        "boiling_point": 1757,
        "brinell_hardness": 167,
        "bulk_modulus": 17,
        "coefficient_of_linear_thermal_expansion": 22.3,
        "common_oxidation_states": [2],
        "density_of_solid": 1550,
        "electrical_resistivity": 3.4,
        "electronic_structure": "[Ar].4s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 1.14},
        "liquid_range": 642,
        "melting_point": 1115,
        "mendeleev_number": 16,
        "mineral_hardness": 1.75,
        "molar_volume": 26.20,
        "name": "Calcium",
        "oxidation_states": [2],
        "poissons_ratio": 0.31,
        "rigidity_modulus": 7.4,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}},
                "VII": {"": {"crystal_radius": 1.2, "ionic_radius": 1.06}},
                "VIII": {"": {"crystal_radius": 1.26, "ionic_radius": 1.12}},
                "IX": {"": {"crystal_radius": 1.32, "ionic_radius": 1.18}},
                "electronegativity": {
                    "": {"crystal_radius": 1.37, "ionic_radius": 1.23}
                },
                "XII": {"": {"crystal_radius": 1.48, "ionic_radius": 1.34}},
            }
        },
        "thermal_conductivity": 200,
        "van_der_waals_radius": 2.31,
        "velocity_of_sound": 3810,
        "electronegativity": 1.0,
        "youngs_modulus": 20,
        "nmr_quadrupole_moment": {"Ca-41": -66.5, "Ca-43": -40.8},
        "metallic_radius": 1.976,
        "iupac_ordering": 15,
    },
    "Cd": {
        "atomic_mass": 112.411,
        "atomic_number": 48,
        "atomic_orbitals": {
            "1s": -941.476646,
            "2p": -127.63512,
            "2s": -136.83249,
            "3d": -14.685252,
            "3p": -21.637522,
            "3s": -25.379908,
            "4d": -0.47053,
            "4p": -2.39526,
            "4s": -3.596069,
            "5s": -0.204228,
        },
        "atomic_radius": 1.55,
        "atomic_radius_calculated": 1.61,
        "boiling_point": 1040,
        "brinell_hardness": 203,
        "bulk_modulus": 42,
        "coefficient_of_linear_thermal_expansion": 30.8,
        "common_oxidation_states": [2],
        "density_of_solid": 8650,
        "electrical_resistivity": 7,
        "electronic_structure": "[Kr].4d10.5s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 1.09},
        "liquid_range": 445.78,
        "melting_point": 594.22,
        "mendeleev_number": 75,
        "mineral_hardness": 2.0,
        "molar_volume": 13.00,
        "name": "Cadmium",
        "oxidation_states": [1, 2],
        "poissons_ratio": 0.30,
        "reflectivity": 67,
        "rigidity_modulus": 19,
        "shannon_radii": {
            "2": {
                "IV": {"": {"crystal_radius": 0.92, "ionic_radius": 0.78}},
                "V": {"": {"crystal_radius": 1.01, "ionic_radius": 0.87}},
                "VI": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
                "VII": {"": {"crystal_radius": 1.17, "ionic_radius": 1.03}},
                "VIII": {"": {"crystal_radius": 1.24, "ionic_radius": 1.1}},
                "XII": {"": {"crystal_radius": 1.45, "ionic_radius": 1.31}},
            }
        },
        "superconduction_temperature": 0.517,
        "thermal_conductivity": 97,
        "van_der_waals_radius": 1.58,
        "velocity_of_sound": 2310,
        "electronegativity": 1.69,
        "youngs_modulus": 50,
        "metallic_radius": 1.51,
        "iupac_ordering": 75,
    },
    "Ce": {
        "atomic_mass": 140.116,
        "atomic_number": 58,
        "atomic_orbitals": {
            "1s": -1406.148284,
            "2p": -206.925148,
            "2s": -218.684842,
            "3d": -32.412569,
            "3p": -41.938282,
            "3s": -47.035283,
            "4d": -4.192548,
            "4f": -0.337442,
            "4p": -7.532106,
            "4s": -9.432744,
            "5d": -0.14055,
            "5p": -0.85011,
            "5s": -1.369728,
            "6s": -0.133974,
        },
        "atomic_radius": 1.85,
        "boiling_point": 3633,
        "brinell_hardness": 412,
        "bulk_modulus": 22,
        "coefficient_of_linear_thermal_expansion": 6.3,
        "common_oxidation_states": [3, 4],
        "density_of_solid": 6689,
        "electrical_resistivity": 74,
        "electronic_structure": "[Xe].4f1.5d1.6s2",
        "icsd_oxidation_states": [3, 4],
        "ionic_radii": {"3": 1.15, "4": 1.01},
        "liquid_range": 2565,
        "melting_point": 1068,
        "mendeleev_number": 32,
        "mineral_hardness": 2.5,
        "molar_volume": 20.69,
        "name": "Cerium",
        "oxidation_states": [2, 3, 4],
        "poissons_ratio": 0.24,
        "rigidity_modulus": 14,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.15, "ionic_radius": 1.01}},
                "VII": {"": {"crystal_radius": 1.21, "ionic_radius": 1.07}},
                "VIII": {"": {"crystal_radius": 1.283, "ionic_radius": 1.143}},
                "IX": {"": {"crystal_radius": 1.336, "ionic_radius": 1.196}},
                "electronegativity": {
                    "": {"crystal_radius": 1.39, "ionic_radius": 1.25}
                },
                "XII": {"": {"crystal_radius": 1.48, "ionic_radius": 1.34}},
            },
            "4": {
                "VI": {"": {"crystal_radius": 1.01, "ionic_radius": 0.87}},
                "VIII": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}},
                "electronegativity": {
                    "": {"crystal_radius": 1.21, "ionic_radius": 1.07}
                },
                "XII": {"": {"crystal_radius": 1.28, "ionic_radius": 1.14}},
            },
        },
        "superconduction_temperature": "0.022  (under pressure)K",
        "thermal_conductivity": 11,
        "velocity_of_sound": 2100,
        "vickers_hardness": 270,
        "electronegativity": 1.12,
        "youngs_modulus": 34,
        "metallic_radius": 1.707,
        "iupac_ordering": 46,
    },
    "Cf": {
        "atomic_mass": 251.0,
        "atomic_number": 98,
        "common_oxidation_states": [3],
        "density_of_solid": 15100,
        "electronic_structure": "[Rn].5f10.7s2",
        "ionic_radii": {"3": 1.09, "4": 0.961},
        "melting_point": 1173,
        "mendeleev_number": 39,
        "molar_volume": 16.50,
        "name": "Californium",
        "oxidation_states": [2, 3, 4],
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}}},
            "4": {
                "VI": {"": {"crystal_radius": 0.961, "ionic_radius": 0.821}},
                "VIII": {"": {"crystal_radius": 1.06, "ionic_radius": 0.92}},
            },
        },
        "electronegativity": 1.3,
        "metallic_radius": 1.86,
        "iupac_ordering": 23,
    },
    "Cl": {
        "atomic_mass": 35.453,
        "atomic_number": 17,
        "atomic_orbitals": {
            "1s": -100.369229,
            "2p": -7.039982,
            "2s": -9.187993,
            "3p": -0.32038,
            "3s": -0.754458,
        },
        "atomic_radius": 1.0,
        "atomic_radius_calculated": 0.79,
        "boiling_point": 239.11,
        "bulk_modulus": "1.1 (liquid)GPa",
        "common_oxidation_states": [-1, 1, 3, 5, 7],
        "critical_temperature": 417,
        "electrical_resistivity": 1e10,  # greater than
        "electronic_structure": "[Ne].3s2.3p5",
        "icsd_oxidation_states": [-1],
        "ionic_radii": {"-1": 1.67, "5": 0.26, "7": 0.41},
        "liquid_range": 67.51,
        "melting_point": 171.6,
        "mendeleev_number": 99,
        "molar_volume": 17.39,
        "name": "Chlorine",
        "oxidation_states": [-1, 1, 2, 3, 4, 5, 6, 7],
        "refractive_index": 1.000773,
        "shannon_radii": {
            "-1": {"VI": {"": {"crystal_radius": 1.67, "ionic_radius": 1.81}}},
            "5": {"IIIPY": {"": {"crystal_radius": 0.26, "ionic_radius": 0.12}}},
            "7": {
                "IV": {"": {"crystal_radius": 0.22, "ionic_radius": 0.08}},
                "VI": {"": {"crystal_radius": 0.41, "ionic_radius": 0.27}},
            },
        },
        "thermal_conductivity": 0.0089,
        "van_der_waals_radius": 1.75,
        "velocity_of_sound": 206,
        "electronegativity": 3.16,
        "nmr_quadrupole_moment": {"Cl-35": -81.65, "Cl-37": -64.35},
        "iupac_ordering": 101,
    },
    "Cm": {
        "atomic_mass": 247.0,
        "atomic_number": 96,
        "boiling_point": 3383,
        "common_oxidation_states": [3],
        "density_of_solid": 13510,
        "electronic_structure": "[Rn].5f7.6d1.7s2",
        "ionic_radii": {"3": 1.11, "4": 0.99},
        "liquid_range": 1770,
        "melting_point": 1613,
        "mendeleev_number": 41,
        "molar_volume": 18.05,
        "name": "Curium",
        "oxidation_states": [3, 4],
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}}},
            "4": {
                "VI": {"": {"crystal_radius": 0.99, "ionic_radius": 0.85}},
                "VIII": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
            },
        },
        "thermal_conductivity": 8.8,
        "electronegativity": 1.3,
        "metallic_radius": 1.743,
        "iupac_ordering": 25,
    },
    "Co": {
        "atomic_mass": 58.933195,
        "atomic_number": 27,
        "atomic_orbitals": {
            "1s": -275.616639,
            "2p": -28.152095,
            "2s": -32.379758,
            "3d": -0.322368,
            "3p": -2.388285,
            "3s": -3.651812,
            "4s": -0.204497,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.52,
        "boiling_point": 3200,
        "brinell_hardness": 700,
        "bulk_modulus": 180,
        "coefficient_of_linear_thermal_expansion": 13.0,
        "common_oxidation_states": [2, 3],
        "density_of_solid": 8900,
        "electrical_resistivity": 6,
        "electronic_structure": "[Ar].3d7.4s2",
        "icsd_oxidation_states": [1, 2, 3, 4],
        "ionic_radii": {"2": 0.885, "3": 0.75, "4": 0.67},
        "ionic_radii_hs": {"2": 0.885, "3": 0.75, "4": 0.67},
        "ionic_radii_ls": {"2": 0.79, "3": 0.685},
        "liquid_range": 1432,
        "melting_point": 1768,
        "mendeleev_number": 64,
        "mineral_hardness": 5.0,
        "molar_volume": 6.67,
        "name": "Cobalt",
        "oxidation_states": [-1, 1, 2, 3, 4, 5],
        "poissons_ratio": 0.31,
        "reflectivity": 67,
        "rigidity_modulus": 75,
        "shannon_radii": {
            "2": {
                "IV": {"High Spin": {"crystal_radius": 0.72, "ionic_radius": 0.58}},
                "V": {"": {"crystal_radius": 0.81, "ionic_radius": 0.67}},
                "VI": {
                    "Low Spin": {"crystal_radius": 0.79, "ionic_radius": 0.65},
                    "High Spin": {"crystal_radius": 0.885, "ionic_radius": 0.745},
                },
                "VIII": {"": {"crystal_radius": 1.04, "ionic_radius": 0.9}},
            },
            "3": {
                "VI": {
                    "High Spin": {"crystal_radius": 0.75, "ionic_radius": 0.61},
                    "Low Spin": {"crystal_radius": 0.685, "ionic_radius": 0.545},
                }
            },
            "4": {
                "IV": {"": {"crystal_radius": 0.54, "ionic_radius": 0.4}},
                "VI": {"High Spin": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
            },
        },
        "thermal_conductivity": 100,
        "velocity_of_sound": 4720,
        "vickers_hardness": 1043,
        "electronegativity": 1.88,
        "youngs_modulus": 209,
        "nmr_quadrupole_moment": {"Co-59": 420.3},
        "metallic_radius": 1.25,
        "iupac_ordering": 67,
    },
    "Cr": {
        "atomic_mass": 51.9961,
        "atomic_number": 24,
        "atomic_orbitals": {
            "1s": -213.881191,
            "2p": -20.526273,
            "2s": -24.113457,
            "3d": -0.118123,
            "3p": -1.65423,
            "3s": -2.649085,
            "4s": -0.150445,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.66,
        "boiling_point": 2944,
        "brinell_hardness": 1120,
        "bulk_modulus": 160,
        "coefficient_of_linear_thermal_expansion": 4.9,
        "common_oxidation_states": [3, 6],
        "density_of_solid": 7140,
        "electrical_resistivity": 12.7,
        "electronic_structure": "[Ar].3d5.4s1",
        "icsd_oxidation_states": [2, 3, 4, 5, 6],
        "ionic_radii": {"2": 0.94},
        "ionic_radii_hs": {"2": 0.94},
        "ionic_radii_ls": {"2": 0.87, "3": 0.755, "4": 0.69, "5": 0.63, "6": 0.58},
        "liquid_range": 764,
        "melting_point": 2180,
        "mendeleev_number": 57,
        "mineral_hardness": 8.5,
        "molar_volume": 7.23,
        "name": "Chromium",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.21,
        "rigidity_modulus": 115,
        "shannon_radii": {
            "2": {
                "VI": {
                    "Low Spin": {"crystal_radius": 0.87, "ionic_radius": 0.73},
                    "High Spin": {"crystal_radius": 0.94, "ionic_radius": 0.8},
                }
            },
            "3": {"VI": {"": {"crystal_radius": 0.755, "ionic_radius": 0.615}}},
            "4": {
                "IV": {"": {"crystal_radius": 0.55, "ionic_radius": 0.41}},
                "VI": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}},
            },
            "5": {
                "IV": {"": {"crystal_radius": 0.485, "ionic_radius": 0.345}},
                "VI": {"": {"crystal_radius": 0.63, "ionic_radius": 0.49}},
                "VIII": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}},
            },
            "6": {
                "IV": {"": {"crystal_radius": 0.4, "ionic_radius": 0.26}},
                "VI": {"": {"crystal_radius": 0.58, "ionic_radius": 0.44}},
            },
        },
        "thermal_conductivity": 94,
        "velocity_of_sound": 5940,
        "vickers_hardness": 1060,
        "electronegativity": 1.66,
        "youngs_modulus": 279,
        "nmr_quadrupole_moment": {"Cr-53": -150.5},
        "metallic_radius": 1.285,
        "iupac_ordering": 58,
    },
    "Cs": {
        "atomic_mass": 132.9054519,
        "atomic_number": 55,
        "atomic_orbitals": {
            "1s": -1256.738791,
            "2p": -180.995344,
            "2s": -191.981873,
            "3d": -26.418398,
            "3p": -35.166423,
            "3s": -39.851584,
            "4d": -2.848386,
            "4p": -5.769326,
            "4s": -7.455966,
            "5p": -0.504903,
            "5s": -0.915819,
            "6s": -0.078699,
        },
        "atomic_radius": 2.6,
        "atomic_radius_calculated": 2.98,
        "boiling_point": 944,
        "brinell_hardness": 0.14,
        "bulk_modulus": 1.6,
        "common_oxidation_states": [1],
        "critical_temperature": 1938,
        "density_of_solid": 1879,
        "electrical_resistivity": 21,
        "electronic_structure": "[Xe].6s1",
        "icsd_oxidation_states": [1],
        "ionic_radii": {"1": 1.81},
        "liquid_range": 642.41,
        "melting_point": 301.59,
        "mendeleev_number": 8,
        "mineral_hardness": 0.2,
        "molar_volume": 70.94,
        "name": "Cesium",
        "oxidation_states": [1],
        "shannon_radii": {
            "1": {
                "VI": {"": {"crystal_radius": 1.81, "ionic_radius": 1.67}},
                "VIII": {"": {"crystal_radius": 1.88, "ionic_radius": 1.74}},
                "IX": {"": {"crystal_radius": 1.92, "ionic_radius": 1.78}},
                "electronegativity": {
                    "": {"crystal_radius": 1.95, "ionic_radius": 1.81}
                },
                "XI": {"": {"crystal_radius": 1.99, "ionic_radius": 1.85}},
                "XII": {"": {"crystal_radius": 2.02, "ionic_radius": 1.88}},
            }
        },
        "thermal_conductivity": 36,
        "van_der_waals_radius": 3.43,
        "electronegativity": 0.79,
        "youngs_modulus": 1.7,
        "metallic_radius": 2.719,
        "iupac_ordering": 7,
    },
    "Cu": {
        "atomic_mass": 63.546,
        "atomic_number": 29,
        "atomic_orbitals": {
            "1s": -320.78852,
            "2p": -33.481247,
            "2s": -38.14131,
            "3d": -0.202272,
            "3p": -2.609244,
            "3s": -4.057453,
            "4s": -0.172056,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.45,
        "boiling_point": 3200,
        "brinell_hardness": 874,
        "bulk_modulus": 140,
        "coefficient_of_linear_thermal_expansion": 16.5,
        "common_oxidation_states": [2],
        "density_of_solid": 8920,
        "electrical_resistivity": 1.72,
        "electronic_structure": "[Ar].3d10.4s1",
        "icsd_oxidation_states": [1, 2, 3],
        "ionic_radii": {"1": 0.91, "2": 0.87, "3": 0.68},
        "liquid_range": 1842.23,
        "melting_point": 1357.77,
        "mendeleev_number": 72,
        "mineral_hardness": 3.0,
        "molar_volume": 7.11,
        "name": "Copper",
        "oxidation_states": [1, 2, 3, 4],
        "poissons_ratio": 0.34,
        "reflectivity": 90,
        "rigidity_modulus": 48,
        "shannon_radii": {
            "1": {
                "II": {"": {"crystal_radius": 0.6, "ionic_radius": 0.46}},
                "IV": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}},
                "VI": {"": {"crystal_radius": 0.91, "ionic_radius": 0.77}},
            },
            "2": {
                "IV": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}},
                "IVSQ": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}},
                "V": {"": {"crystal_radius": 0.79, "ionic_radius": 0.65}},
                "VI": {"": {"crystal_radius": 0.87, "ionic_radius": 0.73}},
            },
            "3": {"VI": {"Low Spin": {"crystal_radius": 0.68, "ionic_radius": 0.54}}},
        },
        "thermal_conductivity": 400,
        "van_der_waals_radius": 1.4,
        "velocity_of_sound": 3570,
        "vickers_hardness": 369,
        "electronegativity": 1.9,
        "youngs_modulus": 130,
        "nmr_quadrupole_moment": {"Cu-63": -220.15, "Cu-65": -204.14},
        "metallic_radius": 1.278,
        "iupac_ordering": 73,
    },
    "Dy": {
        "atomic_mass": 162.5,
        "atomic_number": 66,
        "atomic_orbitals": {
            "1s": -1843.229585,
            "2p": -281.558531,
            "2s": -295.342856,
            "3d": -47.4867,
            "3p": -59.091931,
            "3s": -65.299442,
            "4d": -5.686352,
            "4f": -0.265302,
            "4p": -10.094091,
            "4s": -12.551251,
            "5p": -0.90349,
            "5s": -1.547977,
            "6s": -0.132769,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.28,
        "boiling_point": 2840,
        "brinell_hardness": 500,
        "bulk_modulus": 41,
        "coefficient_of_linear_thermal_expansion": 9.9,
        "common_oxidation_states": [3],
        "density_of_solid": 8551,
        "electrical_resistivity": 92.6,
        "electronic_structure": "[Xe].4f10.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"2": 1.21, "3": 1.052},
        "liquid_range": 1160,
        "melting_point": 1680,
        "mendeleev_number": 24,
        "molar_volume": 19.01,
        "name": "Dysprosium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.25,
        "rigidity_modulus": 25,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.21, "ionic_radius": 1.07}},
                "VII": {"": {"crystal_radius": 1.27, "ionic_radius": 1.13}},
                "VIII": {"": {"crystal_radius": 1.33, "ionic_radius": 1.19}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.052, "ionic_radius": 0.912}},
                "VII": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}},
                "VIII": {"": {"crystal_radius": 1.167, "ionic_radius": 1.027}},
                "IX": {"": {"crystal_radius": 1.223, "ionic_radius": 1.083}},
            },
        },
        "thermal_conductivity": 11,
        "velocity_of_sound": 2710,
        "vickers_hardness": 540,
        "electronegativity": 1.22,
        "youngs_modulus": 61,
        "metallic_radius": 1.773,
        "iupac_ordering": 38,
    },
    "Er": {
        "atomic_mass": 167.259,
        "atomic_number": 68,
        "atomic_orbitals": {
            "1s": -1961.799176,
            "2p": -302.01827,
            "2s": -316.310631,
            "3d": -51.682149,
            "3p": -63.818655,
            "3s": -70.310142,
            "4d": -6.127443,
            "4f": -0.278577,
            "4p": -10.819574,
            "4s": -13.423547,
            "5p": -0.935202,
            "5s": -1.616073,
            "6s": -0.134905,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.26,
        "boiling_point": 3141,
        "brinell_hardness": 814,
        "bulk_modulus": 44,
        "coefficient_of_linear_thermal_expansion": 12.2,
        "common_oxidation_states": [3],
        "density_of_solid": 9066,
        "electrical_resistivity": 86.0,
        "electronic_structure": "[Xe].4f12.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 1.03},
        "liquid_range": 1371,
        "melting_point": 1802,
        "mendeleev_number": 22,
        "molar_volume": 18.46,
        "name": "Erbium",
        "oxidation_states": [3],
        "poissons_ratio": 0.24,
        "rigidity_modulus": 28,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.03, "ionic_radius": 0.89}},
                "VII": {"": {"crystal_radius": 1.085, "ionic_radius": 0.945}},
                "VIII": {"": {"crystal_radius": 1.144, "ionic_radius": 1.004}},
                "IX": {"": {"crystal_radius": 1.202, "ionic_radius": 1.062}},
            }
        },
        "thermal_conductivity": 15,
        "velocity_of_sound": 2830,
        "vickers_hardness": 589,
        "electronegativity": 1.24,
        "youngs_modulus": 70,
        "metallic_radius": 1.756,
        "iupac_ordering": 36,
    },
    "Es": {
        "atomic_mass": 252.0,
        "atomic_number": 99,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f11.7s2",
        "melting_point": 1133,
        "mendeleev_number": 38,
        "molar_volume": 28.52,
        "name": "Einsteinium",
        "oxidation_states": [2, 3],
        "electronegativity": 1.3,
        "metallic_radius": 1.86,
        "iupac_ordering": 22,
    },
    "Eu": {
        "atomic_mass": 151.964,
        "atomic_number": 63,
        "atomic_orbitals": {
            "1s": -1672.309322,
            "2p": -252.176697,
            "2s": -265.199534,
            "3d": -41.465518,
            "3p": -52.281987,
            "3s": -58.068128,
            "4d": -5.03242,
            "4f": -0.232773,
            "4p": -9.025455,
            "4s": -11.267747,
            "5p": -0.853575,
            "5s": -1.444087,
            "6s": -0.129426,
        },
        "atomic_radius": 1.85,
        "atomic_radius_calculated": 2.31,
        "boiling_point": 1800,
        "bulk_modulus": 8.3,
        "coefficient_of_linear_thermal_expansion": 35,
        "common_oxidation_states": [2, 3],
        "density_of_solid": 5244,
        "electrical_resistivity": 90,
        "electronic_structure": "[Xe].4f7.6s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"2": 1.31, "3": 1.087},
        "liquid_range": 701,
        "melting_point": 1099,
        "mendeleev_number": 18,
        "molar_volume": 28.97,
        "name": "Europium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.15,
        "rigidity_modulus": 7.9,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.31, "ionic_radius": 1.17}},
                "VII": {"": {"crystal_radius": 1.34, "ionic_radius": 1.2}},
                "VIII": {"": {"crystal_radius": 1.39, "ionic_radius": 1.25}},
                "IX": {"": {"crystal_radius": 1.44, "ionic_radius": 1.3}},
                "electronegativity": {
                    "": {"crystal_radius": 1.49, "ionic_radius": 1.35}
                },
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.087, "ionic_radius": 0.947}},
                "VII": {"": {"crystal_radius": 1.15, "ionic_radius": 1.01}},
                "VIII": {"": {"crystal_radius": 1.206, "ionic_radius": 1.066}},
                "IX": {"": {"crystal_radius": 1.26, "ionic_radius": 1.12}},
            },
        },
        "thermal_conductivity": 14,
        "vickers_hardness": 167,
        "electronegativity": 1.2,
        "youngs_modulus": 18,
        "metallic_radius": 2.041,
        "iupac_ordering": 41,
    },
    "F": {
        "atomic_mass": 18.9984032,
        "atomic_number": 9,
        "atomic_orbitals": {"1s": -24.189391, "2p": -0.415606, "2s": -1.086859},
        "atomic_radius": 0.5,
        "atomic_radius_calculated": 0.42,
        "boiling_point": 85.03,
        "common_oxidation_states": [-1],
        "critical_temperature": 144,
        "electronic_structure": "[He].2s2.2p5",
        "icsd_oxidation_states": [-1],
        "ionic_radii": {"-1": 1.19, "7": 0.22},
        "liquid_range": 31.5,
        "melting_point": 53.53,
        "mendeleev_number": 102,
        "molar_volume": 11.20,
        "name": "Fluorine",
        "oxidation_states": [-1],
        "refractive_index": 1.000195,
        "shannon_radii": {
            "-1": {
                "II": {"": {"crystal_radius": 1.145, "ionic_radius": 1.285}},
                "III": {"": {"crystal_radius": 1.16, "ionic_radius": 1.3}},
                "IV": {"": {"crystal_radius": 1.17, "ionic_radius": 1.31}},
                "VI": {"": {"crystal_radius": 1.19, "ionic_radius": 1.33}},
            },
            "7": {"VI": {"": {"crystal_radius": 0.22, "ionic_radius": 0.08}}},
        },
        "thermal_conductivity": 0.0277,
        "van_der_waals_radius": 1.47,
        "electronegativity": 3.98,
        "nmr_quadrupole_moment": {"F-19": -94.2},
        "iupac_ordering": 102,
    },
    "Fe": {
        "atomic_mass": 55.845,
        "atomic_number": 26,
        "atomic_orbitals": {
            "1s": -254.225505,
            "2p": -25.551766,
            "2s": -29.56486,
            "3d": -0.295049,
            "3p": -2.187523,
            "3s": -3.360621,
            "4s": -0.197978,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.56,
        "boiling_point": 3134,
        "brinell_hardness": 490,
        "bulk_modulus": 170,
        "coefficient_of_linear_thermal_expansion": 11.8,
        "common_oxidation_states": [2, 3],
        "density_of_solid": 7874,
        "electrical_resistivity": 10,
        "electronic_structure": "[Ar].3d6.4s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"2": 0.92, "3": 0.785},
        "ionic_radii_hs": {"2": 0.92, "3": 0.785},
        "ionic_radii_ls": {"2": 0.75, "3": 0.69, "4": 0.725, "6": 0.39},
        "liquid_range": 1323,
        "melting_point": 1811,
        "mendeleev_number": 61,
        "mineral_hardness": 4.0,
        "molar_volume": 7.09,
        "name": "Iron",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.29,
        "reflectivity": 65,
        "rigidity_modulus": 82,
        "shannon_radii": {
            "2": {
                "IV": {"High Spin": {"crystal_radius": 0.77, "ionic_radius": 0.63}},
                "IVSQ": {"High Spin": {"crystal_radius": 0.78, "ionic_radius": 0.64}},
                "VI": {
                    "Low Spin": {"crystal_radius": 0.75, "ionic_radius": 0.61},
                    "High Spin": {"crystal_radius": 0.92, "ionic_radius": 0.78},
                },
                "VIII": {"High Spin": {"crystal_radius": 1.06, "ionic_radius": 0.92}},
            },
            "3": {
                "IV": {"High Spin": {"crystal_radius": 0.63, "ionic_radius": 0.49}},
                "V": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}},
                "VI": {
                    "Low Spin": {"crystal_radius": 0.69, "ionic_radius": 0.55},
                    "High Spin": {"crystal_radius": 0.785, "ionic_radius": 0.645},
                },
                "VIII": {"High Spin": {"crystal_radius": 0.92, "ionic_radius": 0.78}},
            },
            "4": {"VI": {"": {"crystal_radius": 0.725, "ionic_radius": 0.585}}},
            "6": {"IV": {"": {"crystal_radius": 0.39, "ionic_radius": 0.25}}},
        },
        "thermal_conductivity": 80,
        "velocity_of_sound": 4910,
        "vickers_hardness": 608,
        "electronegativity": 1.83,
        "youngs_modulus": 211,
        "nmr_quadrupole_moment": {"Fe-57": 160.0},
        "metallic_radius": 1.277,
        "iupac_ordering": 64,
    },
    "Fm": {
        "atomic_mass": 257.0,
        "atomic_number": 100,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f12.7s2",
        "melting_point": 1800,  # roughly this temp
        "mendeleev_number": 37,
        "name": "Fermium",
        "oxidation_states": [2, 3],
        "electronegativity": 1.3,
        "iupac_ordering": 21,
    },
    "Fr": {
        "atomic_mass": 223.0,
        "atomic_number": 87,
        "atomic_orbitals": {
            "1s": -3283.263399,
            "2p": -542.41424,
            "2s": -561.73045,
            "3d": -111.085223,
            "3p": -128.607136,
            "3s": -137.959632,
            "4d": -20.812462,
            "4f": -10.050648,
            "4p": -28.648131,
            "4s": -32.861013,
            "5d": -2.360991,
            "5p": -4.97328,
            "5s": -6.509516,
            "6p": -0.466197,
            "6s": -0.841848,
            "7s": -0.076176,
        },
        "common_oxidation_states": [1],
        "electronic_structure": "[Rn].7s1",
        "ionic_radii": {"1": 1.94},
        "melting_point": 300,  # "maybe out" this temp
        "mendeleev_number": 7,
        "name": "Francium",
        "oxidation_states": [1],
        "shannon_radii": {
            "1": {"VI": {"": {"crystal_radius": 1.94, "ionic_radius": 1.8}}}
        },
        "van_der_waals_radius": 3.48,
        "electronegativity": 0.7,
        "iupac_ordering": 6,
    },
    "Ga": {
        "atomic_mass": 69.723,
        "atomic_number": 31,
        "atomic_orbitals": {
            "1s": -370.170639,
            "2p": -40.093339,
            "2s": -45.200869,
            "3d": -0.736204,
            "3p": -3.584666,
            "3s": -5.241645,
            "4p": -0.101634,
            "4s": -0.328019,
        },
        "atomic_radius": 1.3,
        "atomic_radius_calculated": 1.36,
        "boiling_point": 2477,
        "brinell_hardness": 60,
        "coefficient_of_linear_thermal_expansion": 120,
        "common_oxidation_states": [3],
        "density_of_solid": 5904,
        "electrical_resistivity": 14,  # roughly this value
        "electronic_structure": "[Ar].3d10.4s2.4p1",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"3": 0.76},
        "liquid_range": 2174.09,
        "melting_point": 302.91,
        "mendeleev_number": 81,
        "mineral_hardness": 1.5,
        "molar_volume": 11.80,
        "name": "Gallium",
        "oxidation_states": [1, 2, 3],
        "shannon_radii": {
            "3": {
                "IV": {"": {"crystal_radius": 0.61, "ionic_radius": 0.47}},
                "V": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}},
                "VI": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}},
            }
        },
        "superconduction_temperature": 1.083,
        "thermal_conductivity": 29,
        "van_der_waals_radius": 1.87,
        "velocity_of_sound": 2740,
        "electronegativity": 1.81,
        "metallic_radius": 1.35,
        "iupac_ordering": 79,
    },
    "Gd": {
        "atomic_mass": 157.25,
        "atomic_number": 64,
        "atomic_orbitals": {
            "1s": -1728.625195,
            "2p": -262.081616,
            "2s": -275.36313,
            "3d": -43.754556,
            "3p": -54.836922,
            "3s": -60.764408,
            "4d": -5.531835,
            "4f": -0.489012,
            "4p": -9.669866,
            "4s": -11.986486,
            "5d": -0.12722,
            "5p": -0.978749,
            "5s": -1.608477,
            "6s": -0.143627,
        },
        "atomic_radius": 1.8,
        "atomic_radius_calculated": 2.33,
        "boiling_point": 3523,
        "bulk_modulus": 38,
        "coefficient_of_linear_thermal_expansion": 9.4,
        "common_oxidation_states": [3],
        "density_of_solid": 7901,
        "electrical_resistivity": 131,
        "electronic_structure": "[Xe].4f7.5d1.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 1.075},
        "liquid_range": 1938,
        "melting_point": 1585,
        "mendeleev_number": 27,
        "molar_volume": 19.90,
        "name": "Gadolinium",
        "oxidation_states": [1, 2, 3],
        "poissons_ratio": 0.26,
        "rigidity_modulus": 22,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.078, "ionic_radius": 0.938}},
                "VII": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}},
                "VIII": {"": {"crystal_radius": 1.193, "ionic_radius": 1.053}},
                "IX": {"": {"crystal_radius": 1.247, "ionic_radius": 1.107}},
            }
        },
        "superconduction_temperature": 1.083,
        "thermal_conductivity": 11,
        "velocity_of_sound": 2680,
        "vickers_hardness": 570,
        "electronegativity": 1.2,
        "youngs_modulus": 55,
        "metallic_radius": 1.802,
        "iupac_ordering": 40,
    },
    "Ge": {
        "atomic_mass": 72.64,
        "atomic_number": 32,
        "atomic_orbitals": {
            "1s": -396.292991,
            "2p": -43.720129,
            "2s": -49.055282,
            "3d": -1.117316,
            "3p": -4.194822,
            "3s": -5.961472,
            "4p": -0.149882,
            "4s": -0.426523,
        },
        "atomic_radius": 1.25,
        "atomic_radius_calculated": 1.25,
        "boiling_point": 3093,
        "coefficient_of_linear_thermal_expansion": 6,
        "common_oxidation_states": [-4, 2, 4],
        "density_of_solid": 5323,
        "electrical_resistivity": 50000,  # roughly this value
        "electronic_structure": "[Ar].3d10.4s2.4p2",
        "icsd_oxidation_states": [2, 3, 4],
        "ionic_radii": {"2": 0.87, "4": 0.67},
        "liquid_range": 1881.6,
        "melting_point": 1211.4,
        "mendeleev_number": 84,
        "mineral_hardness": 6.0,
        "molar_volume": 13.63,
        "name": "Germanium",
        "oxidation_states": [-4, 1, 2, 3, 4],
        "shannon_radii": {
            "2": {"VI": {"": {"crystal_radius": 0.87, "ionic_radius": 0.73}}},
            "4": {
                "IV": {"": {"crystal_radius": 0.53, "ionic_radius": 0.39}},
                "VI": {"": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
            },
        },
        "thermal_conductivity": 60,
        "van_der_waals_radius": 2.11,
        "velocity_of_sound": 5400,
        "electronegativity": 2.01,
        "metallic_radius": 1.39,
        "iupac_ordering": 84,
    },
    "H": {
        "atomic_mass": 1.00794,
        "atomic_number": 1,
        "atomic_orbitals": {"1s": -0.233471},
        "atomic_radius": 0.25,
        "atomic_radius_calculated": 0.53,
        "boiling_point": 20.28,
        "common_oxidation_states": [-1, 1],
        "critical_temperature": 33,
        "electronic_structure": "1s1",
        "icsd_oxidation_states": [1, -1],
        "liquid_range": 6.27,
        "melting_point": 14.01,
        "mendeleev_number": 103,
        "molar_volume": 11.42,
        "name": "Hydrogen",
        "oxidation_states": [-1, 1],
        "refractive_index": 1.000132,  # for gas; liquid is 1.12"
        "shannon_radii": {
            "1": {
                "I": {"": {"crystal_radius": -0.24, "ionic_radius": -0.38}},
                "II": {"": {"crystal_radius": -0.04, "ionic_radius": -0.18}},
            }
        },
        "thermal_conductivity": 0.1805,
        "van_der_waals_radius": 1.2,
        "velocity_of_sound": 1270,
        "electronegativity": 2.2,
        "nmr_quadrupole_moment": {"H-2": 2.86},
        "iupac_ordering": 92,
    },
    "He": {
        "atomic_mass": 4.002602,
        "atomic_number": 2,
        "atomic_orbitals": {"1s": -0.570425},
        "atomic_radius_calculated": 0.31,
        "boiling_point": 4.22,
        "critical_temperature": 5.19,
        "electronic_structure": "1s2",
        "liquid_range": 3.27,
        "max_oxidation_state": 0.0,
        "melting_point": 0.95,
        "mendeleev_number": 1,
        "min_oxidation_state": 0.0,
        "molar_volume": 21.0,
        "name": "Helium",
        "refractive_index": 1.000035,  # for gas; liquid is 1.028
        "thermal_conductivity": 0.1513,
        "van_der_waals_radius": 1.4,
        "velocity_of_sound": 970,
        "iupac_ordering": 5,
    },
    "Hf": {
        "atomic_mass": 178.49,
        "atomic_number": 72,
        "atomic_orbitals": {
            "1s": -2210.65199,
            "2p": -345.687023,
            "2s": -361.006527,
            "3d": -61.231443,
            "3p": -74.452656,
            "3s": -81.522812,
            "4d": -7.676638,
            "4f": -0.871574,
            "4p": -12.971211,
            "4s": -15.883625,
            "5d": -0.143805,
            "5p": -1.246441,
            "5s": -2.049828,
            "6s": -0.166465,
        },
        "atomic_radius": 1.55,
        "atomic_radius_calculated": 2.08,
        "boiling_point": 4876,
        "brinell_hardness": 1700,
        "bulk_modulus": 110,
        "coefficient_of_linear_thermal_expansion": 5.9,
        "common_oxidation_states": [4],
        "density_of_solid": 13310,
        "electrical_resistivity": 34,
        "electronic_structure": "[Xe].4f14.5d2.6s2",
        "icsd_oxidation_states": [4],
        "ionic_radii": {"4": 0.85},
        "liquid_range": 2370,
        "melting_point": 2506,
        "mendeleev_number": 50,
        "mineral_hardness": 5.5,
        "molar_volume": 13.44,
        "name": "Hafnium",
        "oxidation_states": [2, 3, 4],
        "poissons_ratio": 0.37,
        "rigidity_modulus": 30,
        "shannon_radii": {
            "4": {
                "IV": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}},
                "VI": {"": {"crystal_radius": 0.85, "ionic_radius": 0.71}},
                "VII": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
                "VIII": {"": {"crystal_radius": 0.97, "ionic_radius": 0.83}},
            }
        },
        "superconduction_temperature": 0.128,
        "thermal_conductivity": 23,
        "velocity_of_sound": 3010,
        "vickers_hardness": 1760,
        "electronegativity": 1.3,
        "youngs_modulus": 78,
        "metallic_radius": 1.58,
        "iupac_ordering": 50,
    },
    "Hg": {
        "atomic_mass": 200.59,
        "atomic_number": 80,
        "atomic_orbitals": {
            "1s": -2755.022637,
            "2p": -443.848676,
            "2s": -461.27864,
            "3d": -84.845492,
            "3p": -100.328031,
            "3s": -108.597921,
            "4d": -13.019221,
            "4f": -4.110291,
            "4p": -19.636187,
            "4s": -23.222921,
            "5d": -0.452552,
            "5p": -2.261975,
            "5s": -3.423486,
            "6s": -0.205137,
        },
        "atomic_radius": 1.5,
        "atomic_radius_calculated": 1.71,
        "boiling_point": 629.88,
        "bulk_modulus": 25,
        "common_oxidation_states": [1, 2],
        "critical_temperature": 1750,
        "electrical_resistivity": 96,
        "electronic_structure": "[Xe].4f14.5d10.6s2",
        "icsd_oxidation_states": [1, 2],
        "ionic_radii": {"1": 1.33, "2": 1.16},
        "liquid_range": 395.56,
        "melting_point": 234.32,
        "mendeleev_number": 74,
        "mineral_hardness": 1.5,
        "molar_volume": 14.09,
        "name": "Mercury",
        "oxidation_states": [1, 2, 4],
        "reflectivity": 73,
        "refractive_index": 1.000933,
        "shannon_radii": {
            "1": {
                "III": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}},
                "VI": {"": {"crystal_radius": 1.33, "ionic_radius": 1.19}},
            },
            "2": {
                "II": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}},
                "IV": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
                "VI": {"": {"crystal_radius": 1.16, "ionic_radius": 1.02}},
                "VIII": {"": {"crystal_radius": 1.28, "ionic_radius": 1.14}},
            },
        },
        "superconduction_temperature": 3.95,
        "thermal_conductivity": 8.3,
        "van_der_waals_radius": 1.55,
        "velocity_of_sound": 1407,
        "electronegativity": 2.0,
        "nmr_quadrupole_moment": {"Hg-201": 387.6},
        "metallic_radius": 1.51,
        "iupac_ordering": 74,
    },
    "Ho": {
        "atomic_mass": 164.93032,
        "atomic_number": 67,
        "atomic_orbitals": {
            "1s": -1902.051908,
            "2p": -291.700994,
            "2s": -305.739294,
            "3d": -49.565996,
            "3p": -61.436304,
            "3s": -67.785492,
            "4d": -5.906195,
            "4f": -0.272677,
            "4p": -10.455303,
            "4s": -12.985498,
            "5p": -0.919463,
            "5s": -1.582088,
            "6s": -0.133845,
        },
        "atomic_radius": 1.75,
        "boiling_point": 2993,
        "brinell_hardness": 746,
        "bulk_modulus": 40,
        "coefficient_of_linear_thermal_expansion": 11.2,
        "common_oxidation_states": [3],
        "density_of_solid": 8795,
        "electrical_resistivity": 81.4,
        "electronic_structure": "[Xe].4f11.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 1.041},
        "liquid_range": 1259,
        "melting_point": 1734,
        "mendeleev_number": 23,
        "molar_volume": 18.74,
        "name": "Holmium",
        "oxidation_states": [3],
        "poissons_ratio": 0.23,
        "rigidity_modulus": 26,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.041, "ionic_radius": 0.901}},
                "VIII": {"": {"crystal_radius": 1.155, "ionic_radius": 1.015}},
                "IX": {"": {"crystal_radius": 1.212, "ionic_radius": 1.072}},
                "electronegativity": {
                    "": {"crystal_radius": 1.26, "ionic_radius": 1.12}
                },
            }
        },
        "thermal_conductivity": 16,
        "velocity_of_sound": 2760,
        "vickers_hardness": 481,
        "electronegativity": 1.23,
        "youngs_modulus": 65,
        "metallic_radius": 1.765,
        "iupac_ordering": 37,
    },
    "I": {
        "atomic_mass": 126.90447,
        "atomic_number": 53,
        "atomic_orbitals": {
            "1s": -1161.787047,
            "2p": -164.603788,
            "2s": -175.073804,
            "3d": -22.600693,
            "3p": -30.831092,
            "3s": -35.243351,
            "4d": -1.938179,
            "4p": -4.572522,
            "4s": -6.115811,
            "5p": -0.267904,
            "5s": -0.596339,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.15,
        "boiling_point": 457.4,
        "bulk_modulus": 7.7,
        "common_oxidation_states": [-1, 1, 3, 5, 7],
        "critical_temperature": 819,
        "density_of_solid": 4940,
        "electrical_resistivity": 1e15,  # greater than
        "electronic_structure": "[Kr].4d10.5s2.5p5",
        "icsd_oxidation_states": [5, -1],
        "ionic_radii": {"-1": 2.06, "5": 1.09, "7": 0.67},
        "liquid_range": 70.55,
        "melting_point": 386.85,
        "mendeleev_number": 97,
        "molar_volume": 25.72,
        "name": "Iodine",
        "oxidation_states": [-1, 1, 3, 5, 7],
        "shannon_radii": {
            "-1": {"VI": {"": {"crystal_radius": 2.06, "ionic_radius": 2.2}}},
            "5": {
                "IIIPY": {"": {"crystal_radius": 0.58, "ionic_radius": 0.44}},
                "VI": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
            },
            "7": {
                "IV": {"": {"crystal_radius": 0.56, "ionic_radius": 0.42}},
                "VI": {"": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
            },
        },
        "thermal_conductivity": 0.449,
        "van_der_waals_radius": 1.98,
        "electronegativity": 2.66,
        "nmr_quadrupole_moment": {"I-127": -696.12, "I-129": -604.1},
        "iupac_ordering": 99,
    },
    "In": {
        "atomic_mass": 114.818,
        "atomic_number": 49,
        "atomic_orbitals": {
            "1s": -983.647445,
            "2p": -134.628845,
            "2s": -144.078357,
            "3d": -16.139823,
            "3p": -23.345778,
            "3s": -27.2206,
            "4d": -0.730481,
            "4p": -2.795832,
            "4s": -4.062639,
            "5p": -0.101782,
            "5s": -0.290497,
        },
        "atomic_radius": 1.55,
        "atomic_radius_calculated": 1.56,
        "boiling_point": 2345,
        "brinell_hardness": 8.83,
        "coefficient_of_linear_thermal_expansion": 32.1,
        "common_oxidation_states": [3],
        "density_of_solid": 7310,
        "electrical_resistivity": 8,
        "electronic_structure": "[Kr].4d10.5s2.5p1",
        "icsd_oxidation_states": [1, 2, 3],
        "ionic_radii": {"3": 0.94},
        "liquid_range": 1915.25,
        "melting_point": 429.75,
        "mendeleev_number": 79,
        "mineral_hardness": 1.2,
        "molar_volume": 15.76,
        "name": "Indium",
        "oxidation_states": [1, 2, 3],
        "shannon_radii": {
            "3": {
                "IV": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}},
                "VI": {"": {"crystal_radius": 0.94, "ionic_radius": 0.8}},
                "VIII": {"": {"crystal_radius": 1.06, "ionic_radius": 0.92}},
            }
        },
        "superconduction_temperature": 3.41,
        "thermal_conductivity": 82,
        "van_der_waals_radius": 1.93,
        "velocity_of_sound": 1215,
        "electronegativity": 1.78,
        "youngs_modulus": 11,
        "nmr_quadrupole_moment": {"In-113": 759.8, "In-115": 770.8},
        "metallic_radius": 1.67,
        "iupac_ordering": 78,
    },
    "Ir": {
        "atomic_mass": 192.217,
        "atomic_number": 77,
        "atomic_orbitals": {
            "1s": -2543.761342,
            "2p": -405.526834,
            "2s": -422.159424,
            "3d": -75.485027,
            "3p": -90.108427,
            "3s": -97.923081,
            "4d": -10.856593,
            "4f": -2.738339,
            "4p": -16.966578,
            "4s": -20.29429,
            "5d": -0.335189,
            "5p": -1.883349,
            "5s": -2.909174,
            "6s": -0.195511,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.8,
        "boiling_point": 4701,
        "brinell_hardness": 1670,
        "bulk_modulus": 320,
        "coefficient_of_linear_thermal_expansion": 6.4,
        "common_oxidation_states": [3, 4],
        "density_of_solid": 22650,
        "electrical_resistivity": 4.7,
        "electronic_structure": "[Xe].4f14.5d7.6s2",
        "icsd_oxidation_states": [3, 4, 5],
        "ionic_radii": {"3": 0.82, "4": 0.765, "5": 0.71},
        "liquid_range": 1962,
        "melting_point": 2739,
        "mendeleev_number": 66,
        "mineral_hardness": 6.5,
        "molar_volume": 8.52,
        "name": "Iridium",
        "oxidation_states": [-3, -1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.26,
        "reflectivity": 78,
        "rigidity_modulus": 210,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}}},
            "4": {"VI": {"": {"crystal_radius": 0.765, "ionic_radius": 0.625}}},
            "5": {"VI": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}}},
        },
        "superconduction_temperature": 0.11,
        "thermal_conductivity": 150,
        "velocity_of_sound": 4825,
        "vickers_hardness": 1760,
        "electronegativity": 2.2,
        "youngs_modulus": 528,
        "metallic_radius": 1.357,
        "iupac_ordering": 65,
    },
    "K": {
        "atomic_mass": 39.0983,
        "atomic_number": 19,
        "atomic_orbitals": {
            "1s": -128.414957,
            "2p": -10.283851,
            "2s": -12.839001,
            "3p": -0.693776,
            "3s": -1.281897,
            "4s": -0.088815,
        },
        "atomic_radius": 2.2,
        "atomic_radius_calculated": 2.43,
        "boiling_point": 1032,
        "brinell_hardness": 0.363,
        "bulk_modulus": 3.1,
        "common_oxidation_states": [1],
        "critical_temperature": 2223,
        "density_of_solid": 856,
        "electrical_resistivity": 7.5,
        "electronic_structure": "[Ar].4s1",
        "icsd_oxidation_states": [1],
        "ionic_radii": {"1": 1.52},
        "liquid_range": 695.47,
        "melting_point": 336.53,
        "mendeleev_number": 10,
        "mineral_hardness": 0.4,
        "molar_volume": 45.94,
        "name": "Potassium",
        "oxidation_states": [-1, 1],
        "rigidity_modulus": 1.3,
        "shannon_radii": {
            "1": {
                "IV": {"": {"crystal_radius": 1.51, "ionic_radius": 1.37}},
                "VI": {"": {"crystal_radius": 1.52, "ionic_radius": 1.38}},
                "VII": {"": {"crystal_radius": 1.6, "ionic_radius": 1.46}},
                "VIII": {"": {"crystal_radius": 1.65, "ionic_radius": 1.51}},
                "IX": {"": {"crystal_radius": 1.69, "ionic_radius": 1.55}},
                "electronegativity": {
                    "": {"crystal_radius": 1.73, "ionic_radius": 1.59}
                },
                "XII": {"": {"crystal_radius": 1.78, "ionic_radius": 1.64}},
            }
        },
        "thermal_conductivity": 100,
        "van_der_waals_radius": 2.75,
        "velocity_of_sound": 2000,
        "electronegativity": 0.82,
        "nmr_quadrupole_moment": {"K-39": 58.5, "K-40": -73.0, "K-41": 71.1},
        "metallic_radius": 2.381,
        "iupac_ordering": 9,
    },
    "Kr": {
        "atomic_mass": 83.798,
        "atomic_number": 36,
        "atomic_orbitals": {
            "1s": -509.982989,
            "2p": -60.017328,
            "2s": -66.285953,
            "3d": -3.074109,
            "3p": -7.086634,
            "3s": -9.315192,
            "4p": -0.34634,
            "4s": -0.820574,
        },
        "atomic_radius_calculated": 0.88,
        "boiling_point": 119.93,
        "critical_temperature": 209.4,
        "electronic_structure": "[Ar].3d10.4s2.4p6",
        "liquid_range": 4.14,
        "max_oxidation_state": 0.0,
        "melting_point": 115.79,
        "mendeleev_number": 4,
        "min_oxidation_state": 0.0,
        "molar_volume": 27.99,
        "name": "Krypton",
        "refractive_index": 1.000427,
        "thermal_conductivity": 0.00943,
        "van_der_waals_radius": 2.02,
        "velocity_of_sound": 1120,
        "electronegativity": 3.0,
        "iupac_ordering": 2,
    },
    "La": {
        "atomic_mass": 138.90547,
        "atomic_number": 57,
        "atomic_orbitals": {
            "1s": -1355.622446,
            "2p": -198.325243,
            "2s": -209.831151,
            "3d": -30.626696,
            "3p": -39.895838,
            "3s": -44.856283,
            "4d": -3.95801,
            "4p": -7.167724,
            "4s": -9.000543,
            "5d": -0.141085,
            "5p": -0.824498,
            "5s": -1.324936,
            "6s": -0.132233,
        },
        "atomic_radius": 1.95,
        "boiling_point": 3743,
        "brinell_hardness": 363,
        "bulk_modulus": 28,
        "coefficient_of_linear_thermal_expansion": 12.1,
        "common_oxidation_states": [3],
        "density_of_solid": 6146,
        "electrical_resistivity": 61.5,
        "electronic_structure": "[Xe].5d1.6s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"3": 1.172},
        "liquid_range": 2550,
        "melting_point": 1193,
        "mendeleev_number": 33,
        "mineral_hardness": 2.5,
        "molar_volume": 22.39,
        "name": "Lanthanum",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 14,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.172, "ionic_radius": 1.032}},
                "VII": {"": {"crystal_radius": 1.24, "ionic_radius": 1.1}},
                "VIII": {"": {"crystal_radius": 1.3, "ionic_radius": 1.16}},
                "IX": {"": {"crystal_radius": 1.356, "ionic_radius": 1.216}},
                "electronegativity": {
                    "": {"crystal_radius": 1.41, "ionic_radius": 1.27}
                },
                "XII": {"": {"crystal_radius": 1.5, "ionic_radius": 1.36}},
            }
        },
        "superconduction_temperature": 6.00,
        "thermal_conductivity": 13,
        "velocity_of_sound": 2475,
        "vickers_hardness": 491,
        "electronegativity": 1.1,
        "youngs_modulus": 37,
        "nmr_quadrupole_moment": {"La-139": 200.6},
        "metallic_radius": 1.877,
        "iupac_ordering": 47,
    },
    "Li": {
        "atomic_mass": 6.941,
        "atomic_number": 3,
        "atomic_orbitals": {"1s": -1.878564, "2s": -0.10554},
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 1.67,
        "boiling_point": 1615,
        "bulk_modulus": 11,
        "coefficient_of_linear_thermal_expansion": 46,
        "common_oxidation_states": [1],
        "critical_temperature": 3223,
        "density_of_solid": 535,
        "electrical_resistivity": 9.5,
        "electronic_structure": "[He].2s1",
        "icsd_oxidation_states": [1],
        "ionic_radii": {"1": 0.9},
        "liquid_range": 1161.31,
        "melting_point": 453.69,
        "mendeleev_number": 12,
        "mineral_hardness": 0.6,
        "molar_volume": 13.02,
        "name": "Lithium",
        "oxidation_states": [1],
        "rigidity_modulus": 4.2,
        "shannon_radii": {
            "1": {
                "IV": {"": {"crystal_radius": 0.73, "ionic_radius": 0.59}},
                "VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
                "VIII": {"": {"crystal_radius": 1.06, "ionic_radius": 0.92}},
            }
        },
        "thermal_conductivity": 85,
        "van_der_waals_radius": 1.82,
        "velocity_of_sound": 6000,
        "electronegativity": 0.98,
        "youngs_modulus": 4.9,
        "nmr_quadrupole_moment": {"Li-6": -0.808, "Li-7": -40.1},
        "metallic_radius": 1.52,
        "iupac_ordering": 11,
    },
    "Lr": {
        "atomic_mass": 262.0,
        "atomic_number": 103,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f14.7s2.7p1 (tentative)",
        "melting_point": 1900,  # about this temp
        "mendeleev_number": 34,
        "name": "Lawrencium",
        "oxidation_states": [3],
        "electronegativity": 1.3,
        "iupac_ordering": 18,
    },
    "Lu": {
        "atomic_mass": 174.967,
        "atomic_number": 71,
        "atomic_orbitals": {
            "1s": -2146.885351,
            "2p": -334.330902,
            "2s": -349.390492,
            "3d": -58.592982,
            "3p": -71.538779,
            "3s": -78.462398,
            "4d": -7.113364,
            "4f": -0.568096,
            "4p": -12.250904,
            "4s": -15.08337,
            "5d": -0.103686,
            "5p": -1.111991,
            "5s": -1.872086,
            "6s": -0.155112,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.17,
        "boiling_point": 3675,
        "brinell_hardness": 893,
        "bulk_modulus": 48,
        "coefficient_of_linear_thermal_expansion": 9.9,
        "common_oxidation_states": [3],
        "density_of_solid": 9841,
        "electrical_resistivity": 58,
        "electronic_structure": "[Xe].4f14.5d1.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 1.001},
        "liquid_range": 1750,
        "melting_point": 1925,
        "mendeleev_number": 20,
        "molar_volume": 17.78,
        "name": "Lutetium",
        "oxidation_states": [3],
        "poissons_ratio": 0.26,
        "rigidity_modulus": 27,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.001, "ionic_radius": 0.861}},
                "VIII": {"": {"crystal_radius": 1.117, "ionic_radius": 0.977}},
                "IX": {"": {"crystal_radius": 1.172, "ionic_radius": 1.032}},
            }
        },
        "superconduction_temperature": 0.022,
        "thermal_conductivity": 16,
        "vickers_hardness": 1160,
        "electronegativity": 1.27,
        "youngs_modulus": 69,
        "metallic_radius": 1.735,
        "iupac_ordering": 33,
    },
    "Md": {
        "atomic_mass": 258.0,
        "atomic_number": 101,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f13.7s2",
        "melting_point": 1100,  # about this temp
        "mendeleev_number": 36,
        "name": "Mendelevium",
        "oxidation_states": [2, 3],
        "electronegativity": 1.3,
        "iupac_ordering": 20,
    },
    "Mg": {
        "atomic_mass": 24.305,
        "atomic_number": 12,
        "atomic_orbitals": {
            "1s": -45.973167,
            "2p": -1.71897,
            "2s": -2.903746,
            "3s": -0.175427,
        },
        "atomic_radius": 1.5,
        "atomic_radius_calculated": 1.45,
        "boiling_point": 1363,
        "brinell_hardness": 260,
        "bulk_modulus": 45,
        "coefficient_of_linear_thermal_expansion": 8.2,
        "common_oxidation_states": [2],
        "density_of_solid": 1738,
        "electrical_resistivity": 4.4,
        "electronic_structure": "[Ne].3s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 0.86},
        "liquid_range": 440,
        "melting_point": 923,
        "mendeleev_number": 73,
        "mineral_hardness": 2.5,
        "molar_volume": 14.00,
        "name": "Magnesium",
        "oxidation_states": [1, 2],
        "poissons_ratio": 0.29,
        "reflectivity": 74,
        "rigidity_modulus": 17,
        "shannon_radii": {
            "2": {
                "IV": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}},
                "V": {"": {"crystal_radius": 0.8, "ionic_radius": 0.66}},
                "VI": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}},
                "VIII": {"": {"crystal_radius": 1.03, "ionic_radius": 0.89}},
            }
        },
        "thermal_conductivity": 160,
        "van_der_waals_radius": 1.73,
        "velocity_of_sound": 4602,
        "electronegativity": 1.31,
        "youngs_modulus": 45,
        "nmr_quadrupole_moment": {"Mg-25": 199.4},
        "metallic_radius": 1.6,
        "iupac_ordering": 16,
    },
    "Mn": {
        "atomic_mass": 54.938045,
        "atomic_number": 25,
        "atomic_orbitals": {
            "1s": -233.696912,
            "2p": -23.066297,
            "2s": -26.866646,
            "3d": -0.26654,
            "3p": -1.99145,
            "3s": -3.076637,
            "4s": -0.191136,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.61,
        "boiling_point": 2334,
        "brinell_hardness": 196,
        "bulk_modulus": 120,
        "coefficient_of_linear_thermal_expansion": 21.7,
        "common_oxidation_states": [2, 4, 7],
        "density_of_solid": 7470,
        "electrical_resistivity": 144,
        "electronic_structure": "[Ar].3d5.4s2",
        "icsd_oxidation_states": [2, 3, 4, 7],
        "ionic_radii": {
            "2": 0.97,
            "3": 0.785,
            "4": 0.67,
            "5": 0.47,
            "6": 0.395,
            "7": 0.6,
        },
        "ionic_radii_hs": {"2": 0.97, "3": 0.785},
        "ionic_radii_ls": {
            "2": 0.81,
            "3": 0.72,
            "4": 0.67,
            "5": 0.47,
            "6": 0.395,
            "7": 0.6,
        },
        "liquid_range": 815,
        "melting_point": 1519,
        "mendeleev_number": 60,
        "mineral_hardness": 6.0,
        "molar_volume": 7.35,
        "name": "Manganese",
        "oxidation_states": [-3, -2, -1, 1, 2, 3, 4, 5, 6, 7],
        "shannon_radii": {
            "2": {
                "IV": {"High Spin": {"crystal_radius": 0.8, "ionic_radius": 0.66}},
                "V": {"High Spin": {"crystal_radius": 0.89, "ionic_radius": 0.75}},
                "VI": {
                    "Low Spin": {"crystal_radius": 0.81, "ionic_radius": 0.67},
                    "High Spin": {"crystal_radius": 0.97, "ionic_radius": 0.83},
                },
                "VII": {"High Spin": {"crystal_radius": 1.04, "ionic_radius": 0.9}},
                "VIII": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
            },
            "3": {
                "V": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}},
                "VI": {
                    "Low Spin": {"crystal_radius": 0.72, "ionic_radius": 0.58},
                    "High Spin": {"crystal_radius": 0.785, "ionic_radius": 0.645},
                },
            },
            "4": {
                "IV": {"": {"crystal_radius": 0.53, "ionic_radius": 0.39}},
                "VI": {"": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
            },
            "5": {"IV": {"": {"crystal_radius": 0.47, "ionic_radius": 0.33}}},
            "6": {"IV": {"": {"crystal_radius": 0.395, "ionic_radius": 0.255}}},
            "7": {
                "IV": {"": {"crystal_radius": 0.39, "ionic_radius": 0.25}},
                "VI": {"": {"crystal_radius": 0.6, "ionic_radius": 0.46}},
            },
        },
        "thermal_conductivity": 7.8,
        "velocity_of_sound": 5150,
        "electronegativity": 1.55,
        "youngs_modulus": 198,
        "nmr_quadrupole_moment": {"Mn-55": 330.1},
        "metallic_radius": 1.292,
        "iupac_ordering": 61,
    },
    "Mo": {
        "atomic_mass": 95.94,
        "atomic_number": 42,
        "atomic_orbitals": {
            "1s": -709.232119,
            "2p": -90.791541,
            "2s": -98.503638,
            "3d": -8.257721,
            "3p": -13.71481,
            "3s": -16.681545,
            "4d": -0.153347,
            "4p": -1.39005,
            "4s": -2.234824,
            "5s": -0.14788,
        },
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 1.9,
        "boiling_point": 4912,
        "brinell_hardness": 1500,
        "bulk_modulus": 230,
        "coefficient_of_linear_thermal_expansion": 4.8,
        "common_oxidation_states": [4, 6],
        "density_of_solid": 10280,
        "electrical_resistivity": 5.5,
        "electronic_structure": "[Kr].4d5.5s1",
        "icsd_oxidation_states": [2, 3, 4, 5, 6],
        "ionic_radii": {"3": 0.83, "4": 0.79, "5": 0.75, "6": 0.73},
        "liquid_range": 2016,
        "melting_point": 2896,
        "mendeleev_number": 56,
        "mineral_hardness": 5.5,
        "molar_volume": 9.38,
        "name": "Molybdenum",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.31,
        "reflectivity": 58,
        "rigidity_modulus": 20,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}}},
            "4": {"VI": {"": {"crystal_radius": 0.79, "ionic_radius": 0.65}}},
            "5": {
                "IV": {"": {"crystal_radius": 0.6, "ionic_radius": 0.46}},
                "VI": {"": {"crystal_radius": 0.75, "ionic_radius": 0.61}},
            },
            "6": {
                "IV": {"": {"crystal_radius": 0.55, "ionic_radius": 0.41}},
                "V": {"": {"crystal_radius": 0.64, "ionic_radius": 0.5}},
                "VI": {"": {"crystal_radius": 0.73, "ionic_radius": 0.59}},
                "VII": {"": {"crystal_radius": 0.87, "ionic_radius": 0.73}},
            },
        },
        "superconduction_temperature": 0.915,
        "thermal_conductivity": 139,
        "velocity_of_sound": 6190,
        "vickers_hardness": 1530,
        "electronegativity": 2.16,
        "youngs_modulus": 329,
        "metallic_radius": 1.402,
        "iupac_ordering": 57,
    },
    "N": {
        "atomic_mass": 14.0067,
        "atomic_number": 7,
        "atomic_orbitals": {"1s": -14.011501, "2p": -0.266297, "2s": -0.676151},
        "atomic_radius": 0.65,
        "atomic_radius_calculated": 0.56,
        "boiling_point": 77.36,
        "common_oxidation_states": [-3, 3, 5],
        "critical_temperature": 126.2,
        "electronic_structure": "[He].2s2.2p3",
        "icsd_oxidation_states": [1, 3, 5, -1, -3, -2],
        "ionic_radii": {"-3": 1.32, "3": 0.3, "5": 0.27},
        "liquid_range": 14.31,
        "melting_point": 63.05,
        "mendeleev_number": 100,
        "molar_volume": 13.54,
        "name": "Nitrogen",
        "oxidation_states": [-3, -2, -1, 1, 2, 3, 4, 5],
        "refractive_index": 1.000298,  # for gas; liquid is 1.197
        "shannon_radii": {
            "-3": {"IV": {"": {"crystal_radius": 1.32, "ionic_radius": 1.46}}},
            "3": {"VI": {"": {"crystal_radius": 0.3, "ionic_radius": 0.16}}},
            "5": {
                "III": {"": {"crystal_radius": 0.044, "ionic_radius": -0.104}},
                "VI": {"": {"crystal_radius": 0.27, "ionic_radius": 0.13}},
            },
        },
        "thermal_conductivity": 0.02583,
        "van_der_waals_radius": 1.55,
        "velocity_of_sound": 333.6,
        "electronegativity": 3.04,
        "nmr_quadrupole_moment": {"N-14": 20.44},
        "iupac_ordering": 91,
    },
    "Na": {
        "atomic_mass": 22.98976928,
        "atomic_number": 11,
        "atomic_orbitals": {
            "1s": -37.719975,
            "2p": -1.060636,
            "2s": -2.063401,
            "3s": -0.103415,
        },
        "atomic_radius": 1.8,
        "atomic_radius_calculated": 1.9,
        "boiling_point": 1156,
        "brinell_hardness": 0.69,
        "bulk_modulus": 6.3,
        "coefficient_of_linear_thermal_expansion": 71,
        "common_oxidation_states": [1],
        "critical_temperature": 2573,
        "density_of_solid": 968,
        "electrical_resistivity": 4.9,
        "electronic_structure": "[Ne].3s1",
        "icsd_oxidation_states": [1],
        "ionic_radii": {"1": 1.16},
        "liquid_range": 785.13,
        "melting_point": 370.87,
        "mendeleev_number": 11,
        "mineral_hardness": 0.5,
        "molar_volume": 23.78,
        "name": "Sodium",
        "oxidation_states": [-1, 1],
        "rigidity_modulus": 3.3,
        "shannon_radii": {
            "1": {
                "IV": {"": {"crystal_radius": 1.13, "ionic_radius": 0.99}},
                "V": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}},
                "VI": {"": {"crystal_radius": 1.16, "ionic_radius": 1.02}},
                "VII": {"": {"crystal_radius": 1.26, "ionic_radius": 1.12}},
                "VIII": {"": {"crystal_radius": 1.32, "ionic_radius": 1.18}},
                "IX": {"": {"crystal_radius": 1.38, "ionic_radius": 1.24}},
                "XII": {"": {"crystal_radius": 1.53, "ionic_radius": 1.39}},
            }
        },
        "thermal_conductivity": 140,
        "van_der_waals_radius": 2.27,
        "velocity_of_sound": 3200,
        "electronegativity": 0.93,
        "youngs_modulus": 10,
        "nmr_quadrupole_moment": {"Na-23": 104.1},
        "metallic_radius": 1.86,
        "iupac_ordering": 10,
    },
    "Nb": {
        "atomic_mass": 92.90638,
        "atomic_number": 41,
        "atomic_orbitals": {
            "1s": -673.76253,
            "2p": -85.272175,
            "2s": -92.74086,
            "3d": -7.339839,
            "3p": -12.552855,
            "3s": -15.393727,
            "4d": -0.125252,
            "4p": -1.250049,
            "4s": -2.036693,
            "5s": -0.144272,
        },
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 1.98,
        "boiling_point": 5017,
        "brinell_hardness": 736,
        "bulk_modulus": 170,
        "coefficient_of_linear_thermal_expansion": 7.3,
        "common_oxidation_states": [5],
        "density_of_solid": 8570,
        "electrical_resistivity": 15.2,
        "electronic_structure": "[Kr].4d4.5s1",
        "icsd_oxidation_states": [2, 3, 4, 5],
        "ionic_radii": {"3": 0.86, "4": 0.82, "5": 0.78},
        "liquid_range": 2267,
        "melting_point": 2750,
        "mendeleev_number": 53,
        "mineral_hardness": 6.0,
        "molar_volume": 10.83,
        "name": "Niobium",
        "oxidation_states": [-1, 2, 3, 4, 5],
        "poissons_ratio": 0.40,
        "rigidity_modulus": 38,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}}},
            "4": {
                "VI": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}},
                "VIII": {"": {"crystal_radius": 0.93, "ionic_radius": 0.79}},
            },
            "5": {
                "IV": {"": {"crystal_radius": 0.62, "ionic_radius": 0.48}},
                "VI": {"": {"crystal_radius": 0.78, "ionic_radius": 0.64}},
                "VII": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}},
                "VIII": {"": {"crystal_radius": 0.88, "ionic_radius": 0.74}},
            },
        },
        "superconduction_temperature": 9.25,
        "thermal_conductivity": 54,
        "velocity_of_sound": 3480,
        "vickers_hardness": 1320,
        "electronegativity": 1.6,
        "youngs_modulus": 105,
        "metallic_radius": 1.473,
        "iupac_ordering": 54,
    },
    "Nd": {
        "atomic_mass": 144.242,
        "atomic_number": 60,
        "atomic_orbitals": {
            "1s": -1509.698955,
            "2p": -224.351816,
            "2s": -236.613572,
            "3d": -35.754515,
            "3p": -45.791219,
            "3s": -51.161263,
            "4d": -4.377027,
            "4f": -0.179508,
            "4p": -7.96782,
            "4s": -10.000891,
            "5p": -0.798503,
            "5s": -1.334934,
            "6s": -0.125796,
        },
        "atomic_radius": 1.85,
        "atomic_radius_calculated": 2.06,
        "boiling_point": 3373,
        "brinell_hardness": 265,
        "bulk_modulus": 32,
        "coefficient_of_linear_thermal_expansion": 9.6,
        "common_oxidation_states": [3],
        "density_of_solid": 6800,
        "electrical_resistivity": 64.3,
        "electronic_structure": "[Xe].4f4.6s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"2": 1.43, "3": 1.123},
        "liquid_range": 2076,
        "melting_point": 1297,
        "mendeleev_number": 30,
        "molar_volume": 20.59,
        "name": "Neodymium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 16,
        "shannon_radii": {
            "2": {
                "VIII": {"": {"crystal_radius": 1.43, "ionic_radius": 1.29}},
                "IX": {"": {"crystal_radius": 1.49, "ionic_radius": 1.35}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.123, "ionic_radius": 0.983}},
                "VIII": {"": {"crystal_radius": 1.249, "ionic_radius": 1.109}},
                "IX": {"": {"crystal_radius": 1.303, "ionic_radius": 1.163}},
                "XII": {"": {"crystal_radius": 1.41, "ionic_radius": 1.27}},
            },
        },
        "thermal_conductivity": 17,
        "velocity_of_sound": 2330,
        "vickers_hardness": 343,
        "electronegativity": 1.14,
        "youngs_modulus": 41,
        "metallic_radius": 1.821,
        "iupac_ordering": 44,
    },
    "Ne": {
        "atomic_mass": 20.1797,
        "atomic_number": 10,
        "atomic_orbitals": {"1s": -30.305855, "2p": -0.498034, "2s": -1.322809},
        "atomic_radius_calculated": 0.38,
        "boiling_point": 27.07,
        "critical_temperature": 44.4,
        "electronic_structure": "[He].2s2.2p6",
        "liquid_range": 2.51,
        "max_oxidation_state": 0.0,
        "melting_point": 24.56,
        "mendeleev_number": 2,
        "min_oxidation_state": 0.0,
        "molar_volume": 13.23,
        "name": "Neon",
        "refractive_index": 1.000067,
        "thermal_conductivity": 0.0491,
        "van_der_waals_radius": 1.54,
        "velocity_of_sound": 936,
        "nmr_quadrupole_moment": {"Ne-21": 101.55},
        "iupac_ordering": 4,
    },
    "Ni": {
        "atomic_mass": 58.6934,
        "atomic_number": 28,
        "atomic_orbitals": {
            "1s": -297.870824,
            "2p": -30.868027,
            "2s": -35.312112,
            "3d": -0.348699,
            "3p": -2.594158,
            "3s": -3.950717,
            "4s": -0.210764,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.49,
        "boiling_point": 3186,
        "brinell_hardness": 700,
        "bulk_modulus": 180,
        "coefficient_of_linear_thermal_expansion": 13.4,
        "common_oxidation_states": [2],
        "density_of_solid": 8908,
        "electrical_resistivity": 7.2,
        "electronic_structure": "[Ar].3d8.4s2",
        "icsd_oxidation_states": [1, 2, 3, 4],
        "ionic_radii": {"3": 0.74},
        "ionic_radii_hs": {"3": 0.74},
        "ionic_radii_ls": {"2": 0.83, "3": 0.7, "4": 0.62},
        "liquid_range": 1458,
        "melting_point": 1728,
        "mendeleev_number": 67,
        "mineral_hardness": 4.0,
        "molar_volume": 6.59,
        "name": "Nickel",
        "oxidation_states": [-1, 1, 2, 3, 4],
        "poissons_ratio": 0.31,
        "reflectivity": 72,
        "rigidity_modulus": 76,
        "shannon_radii": {
            "2": {
                "IV": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}},
                "IVSQ": {"": {"crystal_radius": 0.63, "ionic_radius": 0.49}},
                "V": {"": {"crystal_radius": 0.77, "ionic_radius": 0.63}},
                "VI": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}},
            },
            "3": {
                "VI": {
                    "Low Spin": {"crystal_radius": 0.7, "ionic_radius": 0.56},
                    "High Spin": {"crystal_radius": 0.74, "ionic_radius": 0.6},
                }
            },
            "4": {"VI": {"Low Spin": {"crystal_radius": 0.62, "ionic_radius": 0.48}}},
        },
        "thermal_conductivity": 91,
        "van_der_waals_radius": 1.63,
        "velocity_of_sound": 4970,
        "vickers_hardness": 638,
        "electronegativity": 1.91,
        "youngs_modulus": 200,
        "nmr_quadrupole_moment": {"Ni-61": 162.15},
        "metallic_radius": 1.246,
        "iupac_ordering": 70,
    },
    "No": {
        "atomic_mass": 259.0,
        "atomic_number": 102,
        "common_oxidation_states": [3],
        "electronic_structure": "[Rn].5f14.7s2",
        "melting_point": 1100,  # about this temp
        "mendeleev_number": 35,
        "name": "Nobelium",
        "oxidation_states": [2, 3],
        "shannon_radii": {
            "2": {"VI": {"": {"crystal_radius": 1.24, "ionic_radius": 1.1}}}
        },
        "electronegativity": 1.3,
        "iupac_ordering": 19,
    },
    "Np": {
        "atomic_mass": 237.0,
        "atomic_number": 93,
        "atomic_radius": 1.75,
        "boiling_point": 4273,
        "common_oxidation_states": [5],
        "density_of_solid": 20450,
        "electrical_resistivity": 120,
        "electronic_structure": "[Rn].5f4.6d1.7s2",
        "ionic_radii": {
            "2": 1.24,
            "3": 1.15,
            "4": 1.01,
            "5": 0.89,
            "6": 0.86,
            "7": 0.85,
        },
        "liquid_range": 3363,
        "melting_point": 910,
        "mendeleev_number": 44,
        "molar_volume": 11.59,
        "name": "Neptunium",
        "oxidation_states": [3, 4, 5, 6, 7],
        "shannon_radii": {
            "2": {"VI": {"": {"crystal_radius": 1.24, "ionic_radius": 1.1}}},
            "3": {"VI": {"": {"crystal_radius": 1.15, "ionic_radius": 1.01}}},
            "4": {
                "VI": {"": {"crystal_radius": 1.01, "ionic_radius": 0.87}},
                "VIII": {"": {"crystal_radius": 1.12, "ionic_radius": 0.98}},
            },
            "5": {"VI": {"": {"crystal_radius": 0.89, "ionic_radius": 0.75}}},
            "6": {"VI": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}}},
            "7": {"VI": {"": {"crystal_radius": 0.85, "ionic_radius": 0.71}}},
        },
        "thermal_conductivity": 6,
        "electronegativity": 1.36,
        "metallic_radius": 1.503,
        "iupac_ordering": 28,
    },
    "O": {
        "atomic_mass": 15.9994,
        "atomic_number": 8,
        "atomic_orbitals": {"1s": -18.758245, "2p": -0.338381, "2s": -0.871362},
        "atomic_radius": 0.6,
        "atomic_radius_calculated": 0.48,
        "boiling_point": 90.2,
        "common_oxidation_states": [-2],
        "critical_temperature": 154.6,
        "electronic_structure": "[He].2s2.2p4",
        "icsd_oxidation_states": [-2],
        "ionic_radii": {"-2": 1.26},
        "liquid_range": 35.4,
        "melting_point": 54.8,
        "mendeleev_number": 101,
        "molar_volume": 17.36,
        "name": "Oxygen",
        "oxidation_states": [-2, -1, 1, 2],
        "refractive_index": 1.000271,  # for gas; liquid is 1.221
        "shannon_radii": {
            "-2": {
                "II": {"": {"crystal_radius": 1.21, "ionic_radius": 1.35}},
                "III": {"": {"crystal_radius": 1.22, "ionic_radius": 1.36}},
                "IV": {"": {"crystal_radius": 1.24, "ionic_radius": 1.38}},
                "VI": {"": {"crystal_radius": 1.26, "ionic_radius": 1.4}},
                "VIII": {"": {"crystal_radius": 1.28, "ionic_radius": 1.42}},
            }
        },
        "thermal_conductivity": 0.02658,
        "van_der_waals_radius": 1.52,
        "velocity_of_sound": 317.5,
        "electronegativity": 3.44,
        "nmr_quadrupole_moment": {"O-17": -25.58},
        "iupac_ordering": 97,
    },
    "Os": {
        "atomic_mass": 190.23,
        "atomic_number": 76,
        "atomic_orbitals": {
            "1s": -2475.238617,
            "2p": -393.15408,
            "2s": -409.522396,
            "3d": -72.497183,
            "3p": -86.837047,
            "3s": -94.501324,
            "4d": -10.176082,
            "4f": -2.321175,
            "4p": -16.119671,
            "4s": -19.362527,
            "5d": -0.296791,
            "5p": -1.757404,
            "5s": -2.738293,
            "6s": -0.191489,
        },
        "atomic_radius": 1.3,
        "atomic_radius_calculated": 1.85,
        "boiling_point": 5285,
        "brinell_hardness": 3920,
        "coefficient_of_linear_thermal_expansion": 5.1,
        "common_oxidation_states": [4],
        "density_of_solid": 22610,
        "electrical_resistivity": 8.1,
        "electronic_structure": "[Xe].4f14.5d6.6s2",
        "ionic_radii": {"4": 0.77, "5": 0.715, "6": 0.685, "7": 0.665, "8": 0.53},
        "liquid_range": 1979,
        "melting_point": 3306,
        "mendeleev_number": 63,
        "mineral_hardness": 7.0,
        "molar_volume": 8.42,
        "name": "Osmium",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6, 7, 8],
        "poissons_ratio": 0.25,
        "rigidity_modulus": 222,
        "shannon_radii": {
            "4": {"VI": {"": {"crystal_radius": 0.77, "ionic_radius": 0.63}}},
            "5": {"VI": {"": {"crystal_radius": 0.715, "ionic_radius": 0.575}}},
            "6": {
                "V": {"": {"crystal_radius": 0.63, "ionic_radius": 0.49}},
                "VI": {"": {"crystal_radius": 0.685, "ionic_radius": 0.545}},
            },
            "7": {"VI": {"": {"crystal_radius": 0.665, "ionic_radius": 0.525}}},
            "8": {"IV": {"": {"crystal_radius": 0.53, "ionic_radius": 0.39}}},
        },
        "superconduction_temperature": 0.66,
        "thermal_conductivity": 88,
        "velocity_of_sound": 4940,
        "electronegativity": 2.2,
        "metallic_radius": 1.352,
        "iupac_ordering": 62,
    },
    "P": {
        "atomic_mass": 30.973762,
        "atomic_number": 15,
        "atomic_orbitals": {
            "1s": -76.061897,
            "2p": -4.576617,
            "2s": -6.329346,
            "3p": -0.20608,
            "3s": -0.512364,
        },
        "atomic_radius": 1.0,
        "atomic_radius_calculated": 0.98,
        "boiling_point": 550,
        "bulk_modulus": 11,
        "common_oxidation_states": [-3, 3, 5],
        "critical_temperature": 994,
        "density_of_solid": 1823,
        "electrical_resistivity": 10,  # roughly this value
        "electronic_structure": "[Ne].3s2.3p3",
        "icsd_oxidation_states": [3, 4, 5, -2, -3, -1],
        "ionic_radii": {"3": 0.58, "5": 0.52},
        "liquid_range": 232.7,
        "melting_point": 317.3,  # white P
        "mendeleev_number": 90,
        "molar_volume": 17.02,
        "name": "Phosphorus",
        "oxidation_states": [-3, -2, -1, 1, 2, 3, 4, 5],
        "refractive_index": 1.001212,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.58, "ionic_radius": 0.44}}},
            "5": {
                "IV": {"": {"crystal_radius": 0.31, "ionic_radius": 0.17}},
                "V": {"": {"crystal_radius": 0.43, "ionic_radius": 0.29}},
                "VI": {"": {"crystal_radius": 0.52, "ionic_radius": 0.38}},
            },
        },
        "thermal_conductivity": 0.236,
        "van_der_waals_radius": 1.8,
        "electronegativity": 2.19,
        "iupac_ordering": 90,
    },
    "Pa": {
        "atomic_mass": 231.03588,
        "atomic_number": 91,
        "atomic_orbitals": {
            "1s": -3606.333629,
            "2p": -603.470278,
            "2s": -623.870431,
            "3d": -127.781168,
            "3p": -146.485678,
            "3s": -156.466742,
            "4d": -25.933121,
            "4f": -14.105747,
            "4p": -34.48293,
            "4s": -39.064507,
            "5d": -3.659928,
            "5f": -0.316813,
            "5p": -6.709821,
            "5s": -8.463463,
            "6d": -0.142481,
            "6p": -0.799756,
            "6s": -1.287232,
            "7s": -0.129653,
        },
        "atomic_radius": 1.8,
        "common_oxidation_states": [5],
        "density_of_solid": 15370,
        "electrical_resistivity": 18,
        "electronic_structure": "[Rn].5f2.6d1.7s2",
        "ionic_radii": {"3": 1.16, "4": 1.04, "5": 0.92},
        "melting_point": 1841,
        "mendeleev_number": 46,
        "molar_volume": 15.18,
        "name": "Protactinium",
        "oxidation_states": [3, 4, 5],
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.18, "ionic_radius": 1.04}}},
            "4": {
                "VI": {"": {"crystal_radius": 1.04, "ionic_radius": 0.9}},
                "VIII": {"": {"crystal_radius": 1.15, "ionic_radius": 1.01}},
            },
            "5": {
                "VI": {"": {"crystal_radius": 0.92, "ionic_radius": 0.78}},
                "VIII": {"": {"crystal_radius": 1.05, "ionic_radius": 0.91}},
                "IX": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
            },
        },
        "superconduction_temperature": 1.4,
        "thermal_conductivity": 47,
        "electronegativity": 1.5,
        "metallic_radius": 1.642,
        "iupac_ordering": 30,
    },
    "Pb": {
        "atomic_mass": 207.2,
        "atomic_number": 82,
        "atomic_orbitals": {
            "1s": -2901.078061,
            "2p": -470.877785,
            "2s": -488.843335,
            "3d": -91.889924,
            "3p": -107.950391,
            "3s": -116.526852,
            "4d": -15.030026,
            "4f": -5.592532,
            "4p": -21.990564,
            "4s": -25.75333,
            "5d": -0.902393,
            "5p": -2.941657,
            "5s": -4.206797,
            "6p": -0.141831,
            "6s": -0.357187,
        },
        "atomic_radius": 1.8,
        "atomic_radius_calculated": 1.54,
        "boiling_point": 2022,
        "brinell_hardness": 38.3,
        "bulk_modulus": 46,
        "coefficient_of_linear_thermal_expansion": 28.9,
        "common_oxidation_states": [2, 4],
        "density_of_solid": 11340,
        "electrical_resistivity": 21,
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p2",
        "icsd_oxidation_states": [2, 4],
        "ionic_radii": {"2": 1.33, "4": 0.915},
        "liquid_range": 1421.39,
        "melting_point": 600.61,
        "mendeleev_number": 82,
        "mineral_hardness": 1.5,
        "molar_volume": 18.26,
        "name": "Lead",
        "oxidation_states": [-4, 2, 4],
        "poissons_ratio": 0.44,
        "rigidity_modulus": 5.6,
        "shannon_radii": {
            "2": {
                "IVPY": {"": {"crystal_radius": 1.12, "ionic_radius": 0.98}},
                "VI": {"": {"crystal_radius": 1.33, "ionic_radius": 1.19}},
                "VII": {"": {"crystal_radius": 1.37, "ionic_radius": 1.23}},
                "VIII": {"": {"crystal_radius": 1.43, "ionic_radius": 1.29}},
                "IX": {"": {"crystal_radius": 1.49, "ionic_radius": 1.35}},
                "electronegativity": {
                    "": {"crystal_radius": 1.54, "ionic_radius": 1.4}
                },
                "XI": {"": {"crystal_radius": 1.59, "ionic_radius": 1.45}},
                "XII": {"": {"crystal_radius": 1.63, "ionic_radius": 1.49}},
            },
            "4": {
                "IV": {"": {"crystal_radius": 0.79, "ionic_radius": 0.65}},
                "V": {"": {"crystal_radius": 0.87, "ionic_radius": 0.73}},
                "VI": {"": {"crystal_radius": 0.915, "ionic_radius": 0.775}},
                "VIII": {"": {"crystal_radius": 1.08, "ionic_radius": 0.94}},
            },
        },
        "superconduction_temperature": 7.2,
        "thermal_conductivity": 35,
        "van_der_waals_radius": 2.02,
        "velocity_of_sound": 1260,
        "electronegativity": 2.33,
        "youngs_modulus": 16,
        "metallic_radius": 1.75,
        "iupac_ordering": 82,
    },
    "Pd": {
        "atomic_mass": 106.42,
        "atomic_number": 46,
        "atomic_orbitals": {
            "1s": -860.134909,
            "2p": -114.408286,
            "2s": -123.105078,
            "3d": -12.132197,
            "3p": -18.580798,
            "3s": -22.060898,
            "4d": -0.160771,
            "4p": -1.815215,
            "4s": -2.889173,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.69,
        "boiling_point": 3236,
        "brinell_hardness": 37.3,
        "bulk_modulus": 180,
        "coefficient_of_linear_thermal_expansion": 11.8,
        "common_oxidation_states": [2, 4],
        "density_of_solid": 12023,
        "electrical_resistivity": 10.8,
        "electronic_structure": "[Kr].4d10",
        "icsd_oxidation_states": [2, 4],
        "ionic_radii": {"1": 0.73, "2": 1.0, "3": 0.9, "4": 0.755},
        "liquid_range": 1407.95,
        "melting_point": 1828.05,
        "mendeleev_number": 69,
        "mineral_hardness": 4.75,
        "molar_volume": 8.56,
        "name": "Palladium",
        "oxidation_states": [2, 4],
        "poissons_ratio": 0.39,
        "reflectivity": 72,
        "rigidity_modulus": 44,
        "shannon_radii": {
            "1": {"II": {"": {"crystal_radius": 0.73, "ionic_radius": 0.59}}},
            "2": {
                "IVSQ": {"": {"crystal_radius": 0.78, "ionic_radius": 0.64}},
                "VI": {"": {"crystal_radius": 1.0, "ionic_radius": 0.86}},
            },
            "3": {"VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}}},
            "4": {"VI": {"": {"crystal_radius": 0.755, "ionic_radius": 0.615}}},
        },
        "thermal_conductivity": 72,
        "van_der_waals_radius": 1.63,
        "velocity_of_sound": 3070,
        "vickers_hardness": 461,
        "electronegativity": 2.2,
        "youngs_modulus": 121,
        "metallic_radius": 1.376,
        "iupac_ordering": 69,
    },
    "Pm": {
        "atomic_mass": 145.0,
        "atomic_number": 61,
        "atomic_orbitals": {
            "1s": -1562.980284,
            "2p": -233.455114,
            "2s": -245.970548,
            "3d": -37.625433,
            "3p": -47.921132,
            "3s": -53.429311,
            "4d": -4.596822,
            "4f": -0.200159,
            "4p": -8.320495,
            "4s": -10.422756,
            "5p": -0.817702,
            "5s": -1.372265,
            "6s": -0.127053,
        },
        "atomic_radius": 1.85,
        "atomic_radius_calculated": 2.05,
        "boiling_point": 3273,
        "bulk_modulus": 33,
        "coefficient_of_linear_thermal_expansion": 11,
        "common_oxidation_states": [3],
        "density_of_solid": 7264,
        "electrical_resistivity": 75,  # roughly this value
        "electronic_structure": "[Xe].4f5.6s2",
        "ionic_radii": {"3": 1.11},
        "liquid_range": 1900,
        "melting_point": 1373,
        "mendeleev_number": 29,
        "molar_volume": 20.23,
        "name": "Promethium",
        "oxidation_states": [3],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 18,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}},
                "VIII": {"": {"crystal_radius": 1.233, "ionic_radius": 1.093}},
                "IX": {"": {"crystal_radius": 1.284, "ionic_radius": 1.144}},
            }
        },
        "thermal_conductivity": 15,
        "electronegativity": 1.13,
        "youngs_modulus": 46,
        "metallic_radius": 1.811,
        "iupac_ordering": 43,
    },
    "Po": {
        "atomic_mass": 210.0,
        "atomic_number": 84,
        "atomic_orbitals": {
            "1s": -3050.988417,
            "2p": -498.77192,
            "2s": -517.275843,
            "3d": -99.256068,
            "3p": -115.898384,
            "3s": -124.783683,
            "4d": -17.173307,
            "4f": -7.206499,
            "4p": -24.481337,
            "4s": -28.42254,
            "5d": -1.386458,
            "5p": -3.655382,
            "5s": -5.027447,
            "6p": -0.217889,
            "6s": -0.493528,
        },
        "atomic_radius": 1.9,
        "atomic_radius_calculated": 1.35,
        "boiling_point": 1235,
        "common_oxidation_states": [-2, 2, 4],
        "density_of_solid": 9196,
        "electrical_resistivity": 40,
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p4",
        "ionic_radii": {"4": 1.08, "6": 0.81},
        "liquid_range": 708,
        "melting_point": 527,
        "mendeleev_number": 91,
        "molar_volume": 22.97,
        "name": "Polonium",
        "oxidation_states": [-2, 2, 4, 6],
        "shannon_radii": {
            "4": {
                "VI": {"": {"crystal_radius": 1.08, "ionic_radius": 0.94}},
                "VIII": {"": {"crystal_radius": 1.22, "ionic_radius": 1.08}},
            },
            "6": {"VI": {"": {"crystal_radius": 0.81, "ionic_radius": 0.67}}},
        },
        "thermal_conductivity": 20,
        "van_der_waals_radius": 1.97,
        "electronegativity": 2.0,
        "metallic_radius": 1.53,
        "iupac_ordering": 93,
    },
    "Pr": {
        "atomic_mass": 140.90765,
        "atomic_number": 59,
        "atomic_orbitals": {
            "1s": -1457.338067,
            "2p": -215.418313,
            "2s": -227.426363,
            "3d": -33.913996,
            "3p": -43.692548,
            "3s": -48.924994,
            "4d": -4.154228,
            "4f": -0.155138,
            "4p": -7.613108,
            "4s": -9.577447,
            "5p": -0.778046,
            "5s": -1.296106,
            "6s": -0.124465,
        },
        "atomic_radius": 1.85,
        "atomic_radius_calculated": 2.47,
        "boiling_point": 3563,
        "brinell_hardness": 481,
        "bulk_modulus": 29,
        "coefficient_of_linear_thermal_expansion": 6.7,
        "common_oxidation_states": [3],
        "density_of_solid": 6640,
        "electrical_resistivity": 70,
        "electronic_structure": "[Xe].4f3.6s2",
        "icsd_oxidation_states": [3, 4],
        "ionic_radii": {"3": 1.13, "4": 0.99},
        "liquid_range": 2355,
        "melting_point": 1208,
        "mendeleev_number": 31,
        "molar_volume": 20.80,
        "name": "Praseodymium",
        "oxidation_states": [2, 3, 4],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 15,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.13, "ionic_radius": 0.99}},
                "VIII": {"": {"crystal_radius": 1.266, "ionic_radius": 1.126}},
                "IX": {"": {"crystal_radius": 1.319, "ionic_radius": 1.179}},
            },
            "4": {
                "VI": {"": {"crystal_radius": 0.99, "ionic_radius": 0.85}},
                "VIII": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
            },
        },
        "thermal_conductivity": 13,
        "velocity_of_sound": 2280,
        "vickers_hardness": 400,
        "electronegativity": 1.13,
        "youngs_modulus": 37,
        "metallic_radius": 1.828,
        "iupac_ordering": 45,
    },
    "Pt": {
        "atomic_mass": 195.084,
        "atomic_number": 78,
        "atomic_orbitals": {
            "1s": -2613.096532,
            "2p": -417.96053,
            "2s": -434.858003,
            "3d": -78.400271,
            "3p": -93.309108,
            "3s": -101.274869,
            "4d": -11.419476,
            "4f": -3.038049,
            "4p": -17.697297,
            "4s": -21.110651,
            "5d": -0.273634,
            "5p": -1.884256,
            "5s": -2.950526,
            "6s": -0.161308,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.77,
        "boiling_point": 4098,
        "brinell_hardness": 392,
        "bulk_modulus": 230,
        "coefficient_of_linear_thermal_expansion": 8.8,
        "common_oxidation_states": [2, 4],
        "density_of_solid": 21090,
        "electrical_resistivity": 10.6,
        "electronic_structure": "[Xe].4f14.5d9.6s1",
        "ionic_radii": {"2": 0.94, "4": 0.765, "5": 0.71},
        "liquid_range": 2056.6,
        "melting_point": 2041.4,
        "mendeleev_number": 68,
        "mineral_hardness": 3.5,
        "molar_volume": 9.09,
        "name": "Platinum",
        "oxidation_states": [-2, 2, 4, 5, 6],
        "poissons_ratio": 0.38,
        "reflectivity": 73,
        "rigidity_modulus": 61,
        "shannon_radii": {
            "2": {
                "IVSQ": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}},
                "VI": {"": {"crystal_radius": 0.94, "ionic_radius": 0.8}},
            },
            "4": {"VI": {"": {"crystal_radius": 0.765, "ionic_radius": 0.625}}},
            "5": {"VI": {"": {"crystal_radius": 0.71, "ionic_radius": 0.57}}},
        },
        "thermal_conductivity": 72,
        "van_der_waals_radius": 1.75,
        "velocity_of_sound": 2680,
        "vickers_hardness": 549,
        "electronegativity": 2.28,
        "youngs_modulus": 168,
        "metallic_radius": 1.387,
        "iupac_ordering": 68,
    },
    "Pu": {
        "atomic_mass": 244.0,
        "atomic_number": 94,
        "atomic_radius": 1.75,
        "boiling_point": 3503,
        "common_oxidation_states": [4],
        "density_of_solid": 19816,
        "electrical_resistivity": 150,
        "electronic_structure": "[Rn].5f6.7s2",
        "ionic_radii": {"3": 1.14, "4": 1.0, "5": 0.88, "6": 0.85},
        "liquid_range": 2590.5,
        "melting_point": 912.5,
        "mendeleev_number": 43,
        "molar_volume": 12.29,
        "name": "Plutonium",
        "oxidation_states": [3, 4, 5, 6, 7],
        "poissons_ratio": 0.21,
        "rigidity_modulus": 43,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}}},
            "4": {
                "VI": {"": {"crystal_radius": 1.0, "ionic_radius": 0.86}},
                "VIII": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
            },
            "5": {"VI": {"": {"crystal_radius": 0.88, "ionic_radius": 0.74}}},
            "6": {"VI": {"": {"crystal_radius": 0.85, "ionic_radius": 0.71}}},
        },
        "thermal_conductivity": 6,
        "velocity_of_sound": 2260,
        "electronegativity": 1.28,
        "youngs_modulus": 96,
        "metallic_radius": 1.523,
        "iupac_ordering": 27,
    },
    "Ra": {
        "atomic_mass": 226.0,
        "atomic_number": 88,
        "atomic_orbitals": {
            "1s": -3362.736563,
            "2p": -557.513214,
            "2s": -577.101208,
            "3d": -115.306476,
            "3p": -133.12325,
            "3s": -142.632426,
            "4d": -22.208125,
            "4f": -11.181066,
            "4p": -30.221208,
            "4s": -34.525628,
            "5d": -2.819853,
            "5p": -5.547203,
            "5s": -7.139137,
            "6p": -0.634674,
            "6s": -1.05135,
            "7s": -0.113732,
        },
        "atomic_radius": 2.15,
        "boiling_point": 2010,
        "common_oxidation_states": [2],
        "density_of_solid": 5000,
        "electrical_resistivity": 100,
        "electronic_structure": "[Rn].7s2",
        "ionic_radii": {"2": 1.62},
        "liquid_range": 1037,
        "melting_point": 973,
        "mendeleev_number": 13,
        "molar_volume": 41.09,
        "name": "Radium",
        "oxidation_states": [2],
        "shannon_radii": {
            "2": {
                "VIII": {"": {"crystal_radius": 1.62, "ionic_radius": 1.48}},
                "XII": {"": {"crystal_radius": 1.84, "ionic_radius": 1.7}},
            }
        },
        "thermal_conductivity": 19,
        "van_der_waals_radius": 2.83,
        "electronegativity": 0.9,
        "nmr_quadrupole_moment": {"Ra-223": 1210.3},
        "metallic_radius": 2.293,
        "iupac_ordering": 12,
    },
    "Rb": {
        "atomic_mass": 85.4678,
        "atomic_number": 37,
        "atomic_orbitals": {
            "1s": -540.957115,
            "2p": -64.784678,
            "2s": -71.291202,
            "3d": -3.915508,
            "3p": -8.165416,
            "3s": -10.513861,
            "4p": -0.59217,
            "4s": -1.135051,
            "5s": -0.085375,
        },
        "atomic_radius": 2.35,
        "atomic_radius_calculated": 2.65,
        "boiling_point": 961,
        "brinell_hardness": 0.216,
        "bulk_modulus": 2.5,
        "common_oxidation_states": [1],
        "critical_temperature": 2093,
        "density_of_solid": 1532,
        "electrical_resistivity": 13.3,
        "electronic_structure": "[Kr].5s1",
        "icsd_oxidation_states": [1],
        "ionic_radii": {"1": 1.66},
        "liquid_range": 648.54,
        "melting_point": 312.46,
        "mendeleev_number": 9,
        "mineral_hardness": 0.3,
        "molar_volume": 55.76,
        "name": "Rubidium",
        "oxidation_states": [1],
        "shannon_radii": {
            "1": {
                "VI": {"": {"crystal_radius": 1.66, "ionic_radius": 1.52}},
                "VII": {"": {"crystal_radius": 1.7, "ionic_radius": 1.56}},
                "VIII": {"": {"crystal_radius": 1.75, "ionic_radius": 1.61}},
                "IX": {"": {"crystal_radius": 1.77, "ionic_radius": 1.63}},
                "electronegativity": {
                    "": {"crystal_radius": 1.8, "ionic_radius": 1.66}
                },
                "XI": {"": {"crystal_radius": 1.83, "ionic_radius": 1.69}},
                "XII": {"": {"crystal_radius": 1.86, "ionic_radius": 1.72}},
                "XIV": {"": {"crystal_radius": 1.97, "ionic_radius": 1.83}},
            }
        },
        "thermal_conductivity": 58,
        "van_der_waals_radius": 3.03,
        "velocity_of_sound": 1300,
        "electronegativity": 0.82,
        "youngs_modulus": 2.4,
        "metallic_radius": 2.537,
        "iupac_ordering": 8,
    },
    "Re": {
        "atomic_mass": 186.207,
        "atomic_number": 75,
        "atomic_orbitals": {
            "1s": -2407.665572,
            "2p": -380.982869,
            "2s": -397.087707,
            "3d": -69.57676,
            "3p": -83.634578,
            "3s": -91.149193,
            "4d": -9.516816,
            "4f": -1.92508,
            "4p": -15.295495,
            "4s": -18.454325,
            "5d": -0.258639,
            "5p": -1.631227,
            "5s": -2.567348,
            "6s": -0.186859,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.88,
        "boiling_point": 5869,
        "brinell_hardness": 1320,
        "bulk_modulus": 370,
        "coefficient_of_linear_thermal_expansion": 6.2,
        "common_oxidation_states": [4],
        "density_of_solid": 21020,
        "electrical_resistivity": 18,
        "electronic_structure": "[Xe].4f14.5d5.6s2",
        "icsd_oxidation_states": [3, 4, 5, 6, 7],
        "ionic_radii": {"4": 0.77, "5": 0.72, "6": 0.69, "7": 0.67},
        "liquid_range": 2410,
        "melting_point": 3459,
        "mendeleev_number": 58,
        "mineral_hardness": 7.0,
        "molar_volume": 8.86,
        "name": "Rhenium",
        "oxidation_states": [-3, -1, 1, 2, 3, 4, 5, 6, 7],
        "poissons_ratio": 0.30,
        "rigidity_modulus": 178,
        "shannon_radii": {
            "4": {"VI": {"": {"crystal_radius": 0.77, "ionic_radius": 0.63}}},
            "5": {"VI": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}}},
            "6": {"VI": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}}},
            "7": {
                "IV": {"": {"crystal_radius": 0.52, "ionic_radius": 0.38}},
                "VI": {"": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
            },
        },
        "superconduction_temperature": 1.70,
        "thermal_conductivity": 48,
        "velocity_of_sound": 4700,
        "vickers_hardness": 2450,
        "electronegativity": 1.9,
        "youngs_modulus": 463,
        "metallic_radius": 1.375,
        "iupac_ordering": 59,
    },
    "Rh": {
        "atomic_mass": 102.9055,
        "atomic_number": 45,
        "atomic_orbitals": {
            "1s": -821.136773,
            "2p": -108.357665,
            "2s": -116.80695,
            "3d": -11.21725,
            "3p": -17.415299,
            "3s": -20.765603,
            "4d": -0.239422,
            "4p": -1.806456,
            "4s": -2.825505,
            "5s": -0.154624,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.73,
        "boiling_point": 3968,
        "brinell_hardness": 1100,
        "bulk_modulus": 380,
        "coefficient_of_linear_thermal_expansion": 8.2,
        "common_oxidation_states": [3],
        "density_of_solid": 12450,
        "electrical_resistivity": 4.3,
        "electronic_structure": "[Kr].4d8.5s1",
        "icsd_oxidation_states": [3, 4],
        "ionic_radii": {"3": 0.805, "4": 0.74, "5": 0.69},
        "liquid_range": 1731,
        "melting_point": 2237,
        "mendeleev_number": 65,
        "mineral_hardness": 6.0,
        "molar_volume": 8.28,
        "name": "Rhodium",
        "oxidation_states": [-1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.26,
        "reflectivity": 84,
        "rigidity_modulus": 150,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.805, "ionic_radius": 0.665}}},
            "4": {"VI": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}}},
            "5": {"VI": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}}},
        },
        "thermal_conductivity": 150,
        "velocity_of_sound": 4700,
        "vickers_hardness": 1246,
        "electronegativity": 2.28,
        "youngs_modulus": 275,
        "metallic_radius": 1.345,
        "iupac_ordering": 66,
    },
    "Rn": {
        "atomic_mass": 220.0,
        "atomic_number": 86,
        "atomic_orbitals": {
            "1s": -3204.756288,
            "2p": -527.533025,
            "2s": -546.57796,
            "3d": -106.945006,
            "3p": -124.172862,
            "3s": -133.369144,
            "4d": -19.449994,
            "4f": -8.953318,
            "4p": -27.108985,
            "4s": -31.230804,
            "5d": -1.911329,
            "5p": -4.408702,
            "5s": -5.889683,
            "6p": -0.29318,
            "6s": -0.62657,
        },
        "atomic_radius_calculated": 1.2,
        "boiling_point": 211.3,
        "critical_temperature": 377,
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p6",
        "liquid_range": 9.3,
        "max_oxidation_state": 0.0,
        "melting_point": 202,
        "mendeleev_number": 6,
        "min_oxidation_state": 0.0,
        "molar_volume": 50.50,
        "name": "Radon",
        "thermal_conductivity": 0.00361,
        "van_der_waals_radius": 2.2,
        "electronegativity": 2.2,
        "iupac_ordering": 0,
    },
    "Ru": {
        "atomic_mass": 101.07,
        "atomic_number": 44,
        "atomic_orbitals": {
            "1s": -782.918621,
            "2p": -102.333649,
            "2s": -110.536054,
            "3d": -10.195668,
            "3p": -16.145217,
            "3s": -19.366692,
            "4d": -0.210375,
            "4p": -1.667549,
            "4s": -2.628363,
            "5s": -0.152834,
        },
        "atomic_radius": 1.3,
        "atomic_radius_calculated": 1.78,
        "boiling_point": 4423,
        "brinell_hardness": 2160,
        "bulk_modulus": 220,
        "coefficient_of_linear_thermal_expansion": 6.4,
        "common_oxidation_states": [3, 4],
        "density_of_solid": 12370,
        "electrical_resistivity": 7.1,
        "electronic_structure": "[Kr].4d7.5s1",
        "icsd_oxidation_states": [2, 3, 4, 5, 6],
        "ionic_radii": {"3": 0.82, "4": 0.76, "5": 0.705, "7": 0.52, "8": 0.5},
        "liquid_range": 1816,
        "melting_point": 2607,
        "mendeleev_number": 62,
        "mineral_hardness": 6.5,
        "molar_volume": 8.17,
        "name": "Ruthenium",
        "oxidation_states": [-2, 1, 2, 3, 4, 5, 6, 7, 8],
        "poissons_ratio": 0.30,
        "rigidity_modulus": 173,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}}},
            "4": {"VI": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}}},
            "5": {"VI": {"": {"crystal_radius": 0.705, "ionic_radius": 0.565}}},
            "7": {"IV": {"": {"crystal_radius": 0.52, "ionic_radius": 0.38}}},
            "8": {"IV": {"": {"crystal_radius": 0.5, "ionic_radius": 0.36}}},
        },
        "superconduction_temperature": 0.49,
        "thermal_conductivity": 120,
        "velocity_of_sound": 5970,
        "electronegativity": 2.2,
        "youngs_modulus": 447,
        "metallic_radius": 1.339,
        "iupac_ordering": 63,
    },
    "S": {
        "atomic_mass": 32.065,
        "atomic_number": 16,
        "atomic_orbitals": {
            "1s": -87.789937,
            "2p": -5.751257,
            "2s": -7.69994,
            "3p": -0.261676,
            "3s": -0.630912,
        },
        "atomic_radius": 1.0,
        "atomic_radius_calculated": 0.88,
        "boiling_point": 717.87,
        "bulk_modulus": 7.7,
        "common_oxidation_states": [-2, 2, 4, 6],
        "critical_temperature": 1314,
        "density_of_solid": 1960,
        "electrical_resistivity": 1e23,  # greater than
        "electronic_structure": "[Ne].3s2.3p4",
        "icsd_oxidation_states": [-1, 2, 4, -2, 6],
        "ionic_radii": {"-2": 1.7, "4": 0.51, "6": 0.43},
        "liquid_range": 329.51,
        "melting_point": 388.36,
        "mendeleev_number": 94,
        "mineral_hardness": 2.0,
        "molar_volume": 15.53,
        "name": "Sulfur",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6],
        "refractive_index": 1.001111,
        "shannon_radii": {
            "-2": {"VI": {"": {"crystal_radius": 1.7, "ionic_radius": 1.84}}},
            "4": {"VI": {"": {"crystal_radius": 0.51, "ionic_radius": 0.37}}},
            "6": {
                "IV": {"": {"crystal_radius": 0.26, "ionic_radius": 0.12}},
                "VI": {"": {"crystal_radius": 0.43, "ionic_radius": 0.29}},
            },
        },
        "thermal_conductivity": 0.205,
        "van_der_waals_radius": 1.8,
        "electronegativity": 2.58,
        "nmr_quadrupole_moment": {"S-33": -67.8, "S-35": 47.1},
        "iupac_ordering": 96,
    },
    "Sb": {
        "atomic_mass": 121.76,
        "atomic_number": 51,
        "atomic_orbitals": {
            "1s": -1070.823495,
            "2p": -149.214271,
            "2s": -159.171745,
            "3d": -19.239895,
            "3p": -26.956184,
            "3s": -31.098242,
            "4d": -1.297338,
            "4p": -3.646579,
            "4s": -5.04964,
            "5p": -0.185623,
            "5s": -0.445605,
        },
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 1.33,
        "boiling_point": 1860,
        "brinell_hardness": 294,
        "bulk_modulus": 42,
        "coefficient_of_linear_thermal_expansion": 11,
        "common_oxidation_states": [-3, 3, 5],
        "density_of_solid": 6697,
        "electrical_resistivity": 40,
        "electronic_structure": "[Kr].4d10.5s2.5p3",
        "icsd_oxidation_states": [-2, 3, 5, -1, -3],
        "ionic_radii": {"3": 0.9, "5": 0.76},
        "liquid_range": 956.22,
        "melting_point": 903.78,
        "mendeleev_number": 88,
        "mineral_hardness": 3.0,
        "molar_volume": 18.19,
        "name": "Antimony",
        "oxidation_states": [-3, 3, 5],
        "reflectivity": 55,
        "rigidity_modulus": 20,
        "shannon_radii": {
            "3": {
                "IVPY": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
                "V": {"": {"crystal_radius": 0.94, "ionic_radius": 0.8}},
                "VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
            },
            "5": {"VI": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}}},
        },
        "thermal_conductivity": 24,
        "van_der_waals_radius": 2.06,
        "velocity_of_sound": 3420,
        "electronegativity": 2.05,
        "youngs_modulus": 55,
        "nmr_quadrupole_moment": {"Sb-121": -543.11, "Sb-123": -692.14},
        "metallic_radius": 1.61,
        "iupac_ordering": 88,
    },
    "Sc": {
        "atomic_mass": 44.955912,
        "atomic_number": 21,
        "atomic_orbitals": {
            "1s": -160.184109,
            "2p": -14.240006,
            "2s": -17.206464,
            "3d": -0.13108,
            "3p": -1.233165,
            "3s": -1.988378,
            "4s": -0.156478,
        },
        "atomic_radius": 1.6,
        "atomic_radius_calculated": 1.84,
        "boiling_point": 3103,
        "brinell_hardness": 750,
        "bulk_modulus": 57,
        "coefficient_of_linear_thermal_expansion": 10.2,
        "common_oxidation_states": [3],
        "density_of_solid": 2985,
        "electrical_resistivity": 55,  # roughly this value
        "electronic_structure": "[Ar].3d1.4s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"3": 0.885},
        "liquid_range": 1289,
        "melting_point": 1814,
        "mendeleev_number": 19,
        "molar_volume": 15.00,
        "name": "Scandium",
        "oxidation_states": [1, 2, 3],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 29,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 0.885, "ionic_radius": 0.745}},
                "VIII": {"": {"crystal_radius": 1.01, "ionic_radius": 0.87}},
            }
        },
        "superconduction_temperature": "0.05  (under pressure)K",
        "thermal_conductivity": 16,
        "van_der_waals_radius": 2.11,
        "electronegativity": 1.36,
        "youngs_modulus": 74,
        "nmr_quadrupole_moment": {"Sc-45": -220.2},
        "metallic_radius": 1.641,
        "iupac_ordering": 49,
    },
    "Se": {
        "atomic_mass": 78.96,
        "atomic_number": 34,
        "atomic_orbitals": {
            "1s": -451.300258,
            "2p": -51.514388,
            "2s": -57.311948,
            "3d": -2.011392,
            "3p": -5.553517,
            "3s": -7.547186,
            "4p": -0.245806,
            "4s": -0.621248,
        },
        "atomic_radius": 1.15,
        "atomic_radius_calculated": 1.03,
        "boiling_point": 958,
        "brinell_hardness": 736,
        "bulk_modulus": 8.3,
        "common_oxidation_states": [-2, 2, 4, 6],
        "critical_temperature": 1766,
        "density_of_solid": 4819,
        # "electrical_resistivity": "high",  # lmaoooo this doesn't help
        "electronic_structure": "[Ar].3d10.4s2.4p4",
        "icsd_oxidation_states": [-1, 4, -2, 6],
        "ionic_radii": {"-2": 1.84, "4": 0.64, "6": 0.56},
        "liquid_range": 464,
        "melting_point": 494,
        "mendeleev_number": 93,
        "mineral_hardness": 2.0,
        "molar_volume": 16.42,
        "name": "Selenium",
        "oxidation_states": [-2, 2, 4, 6],
        "poissons_ratio": 0.33,
        "refractive_index": 1.000895,
        "rigidity_modulus": 3.7,
        "shannon_radii": {
            "-2": {"VI": {"": {"crystal_radius": 1.84, "ionic_radius": 1.98}}},
            "4": {"VI": {"": {"crystal_radius": 0.64, "ionic_radius": 0.5}}},
            "6": {
                "IV": {"": {"crystal_radius": 0.42, "ionic_radius": 0.28}},
                "VI": {"": {"crystal_radius": 0.56, "ionic_radius": 0.42}},
            },
        },
        "thermal_conductivity": 0.52,
        "van_der_waals_radius": 1.9,
        "velocity_of_sound": 3350,
        "electronegativity": 2.55,
        "youngs_modulus": 10,
        "iupac_ordering": 95,
    },
    "Si": {
        "atomic_mass": 28.0855,
        "atomic_number": 14,
        "atomic_orbitals": {
            "1s": -65.184426,
            "2p": -3.514938,
            "2s": -5.075056,
            "3p": -0.153293,
            "3s": -0.398139,
        },
        "atomic_radius": 1.1,
        "atomic_radius_calculated": 1.11,
        "boiling_point": 3173,
        "bulk_modulus": 100,
        "coefficient_of_linear_thermal_expansion": 2.6,
        "common_oxidation_states": [-4, 4],
        "density_of_solid": 2330,
        "electrical_resistivity": 100000,  # roughly this value
        "electronic_structure": "[Ne].3s2.3p2",
        "icsd_oxidation_states": [-4, 4],
        "ionic_radii": {"4": 0.54},
        "liquid_range": 1486,
        "melting_point": 1687,
        "mendeleev_number": 85,
        "mineral_hardness": 6.5,
        "molar_volume": 12.06,
        "name": "Silicon",
        "oxidation_states": [-4, -3, -2, -1, 1, 2, 3, 4],
        "reflectivity": 28,
        "shannon_radii": {
            "4": {
                "IV": {"": {"crystal_radius": 0.4, "ionic_radius": 0.26}},
                "VI": {"": {"crystal_radius": 0.54, "ionic_radius": 0.4}},
            }
        },
        "thermal_conductivity": 150,
        "van_der_waals_radius": 2.1,
        "velocity_of_sound": 2200,
        "electronegativity": 1.9,
        "youngs_modulus": 47,
        "iupac_ordering": 85,
    },
    "Sm": {
        "atomic_mass": 150.36,
        "atomic_number": 62,
        "atomic_orbitals": {
            "1s": -1617.183426,
            "2p": -242.729726,
            "2s": -255.498846,
            "3d": -39.528656,
            "3p": -50.08426,
            "3s": -55.731133,
            "4d": -4.814978,
            "4f": -0.21776,
            "4p": -8.672685,
            "4s": -10.844667,
            "5p": -0.835987,
            "5s": -1.408552,
            "6s": -0.128259,
        },
        "atomic_radius": 1.85,
        "atomic_radius_calculated": 2.38,
        "boiling_point": 2076,
        "brinell_hardness": 441,
        "bulk_modulus": 38,
        "coefficient_of_linear_thermal_expansion": 12.7,
        "common_oxidation_states": [3],
        "density_of_solid": 7353,
        "electrical_resistivity": 94,
        "electronic_structure": "[Xe].4f6.6s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"2": 1.36, "3": 1.0979999999999999},
        "liquid_range": 731,
        "melting_point": 1345,
        "mendeleev_number": 28,
        "molar_volume": 19.98,
        "name": "Samarium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.27,
        "rigidity_modulus": 20,
        "shannon_radii": {
            "2": {
                "VII": {"": {"crystal_radius": 1.36, "ionic_radius": 1.22}},
                "VIII": {"": {"crystal_radius": 1.41, "ionic_radius": 1.27}},
                "IX": {"": {"crystal_radius": 1.46, "ionic_radius": 1.32}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.098, "ionic_radius": 0.958}},
                "VII": {"": {"crystal_radius": 1.16, "ionic_radius": 1.02}},
                "VIII": {"": {"crystal_radius": 1.219, "ionic_radius": 1.079}},
                "IX": {"": {"crystal_radius": 1.272, "ionic_radius": 1.132}},
                "XII": {"": {"crystal_radius": 1.38, "ionic_radius": 1.24}},
            },
        },
        "thermal_conductivity": 13,
        "velocity_of_sound": 2130,
        "vickers_hardness": 412,
        "electronegativity": 1.17,
        "youngs_modulus": 50,
        "metallic_radius": 1.804,
        "iupac_ordering": 42,
    },
    "Sn": {
        "atomic_mass": 118.71,
        "atomic_number": 50,
        "atomic_orbitals": {
            "1s": -1026.762169,
            "2p": -141.821093,
            "2s": -151.523991,
            "3d": -17.657276,
            "3p": -25.117913,
            "3s": -29.125969,
            "4d": -1.004952,
            "4p": -3.211998,
            "4s": -4.546335,
            "5p": -0.14445,
            "5s": -0.369349,
        },
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 1.45,
        "boiling_point": 2875,
        "brinell_hardness": 51,
        "bulk_modulus": 58,
        "coefficient_of_linear_thermal_expansion": 22,
        "common_oxidation_states": [-4, 2, 4],
        "density_of_solid": 7310,
        "electrical_resistivity": 11.5,
        "electronic_structure": "[Kr].4d10.5s2.5p2",
        "icsd_oxidation_states": [2, 3, 4],
        "ionic_radii": {"4": 0.83},
        "liquid_range": 2369.92,
        "melting_point": 505.08,
        "mendeleev_number": 83,
        "mineral_hardness": 1.5,
        "molar_volume": 16.29,
        "name": "Tin",
        "oxidation_states": [-4, 2, 4],
        "poissons_ratio": 0.36,
        "reflectivity": 54,
        "rigidity_modulus": 18,
        "shannon_radii": {
            "4": {
                "IV": {"": {"crystal_radius": 0.69, "ionic_radius": 0.55}},
                "V": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}},
                "VI": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}},
                "VII": {"": {"crystal_radius": 0.89, "ionic_radius": 0.75}},
                "VIII": {"": {"crystal_radius": 0.95, "ionic_radius": 0.81}},
            }
        },
        "superconduction_temperature": 3.72,
        "thermal_conductivity": 67,
        "van_der_waals_radius": 2.17,
        "velocity_of_sound": 2500,
        "electronegativity": 1.96,
        "youngs_modulus": 50,
        "nmr_quadrupole_moment": {"Sn-119": -132.1},
        "metallic_radius": 1.58,
        "iupac_ordering": 83,
    },
    "Sr": {
        "atomic_mass": 87.62,
        "atomic_number": 38,
        "atomic_orbitals": {
            "1s": -572.870169,
            "2p": -69.745941,
            "2s": -76.491823,
            "3d": -4.813498,
            "3p": -9.301863,
            "3s": -11.771585,
            "4p": -0.844489,
            "4s": -1.455317,
            "5s": -0.131793,
        },
        "atomic_radius": 2.0,
        "atomic_radius_calculated": 2.19,
        "boiling_point": 1655,
        "coefficient_of_linear_thermal_expansion": 22.5,
        "common_oxidation_states": [2],
        "density_of_solid": 2630,
        "electrical_resistivity": 13.5,
        "electronic_structure": "[Kr].5s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 1.32},
        "liquid_range": 605,
        "melting_point": 1050,
        "mendeleev_number": 15,
        "mineral_hardness": 1.5,
        "molar_volume": 33.94,
        "name": "Strontium",
        "oxidation_states": [2],
        "poissons_ratio": 0.28,
        "rigidity_modulus": 6.1,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.32, "ionic_radius": 1.18}},
                "VII": {"": {"crystal_radius": 1.35, "ionic_radius": 1.21}},
                "VIII": {"": {"crystal_radius": 1.4, "ionic_radius": 1.26}},
                "IX": {"": {"crystal_radius": 1.45, "ionic_radius": 1.31}},
                "electronegativity": {
                    "": {"crystal_radius": 1.5, "ionic_radius": 1.36}
                },
                "XII": {"": {"crystal_radius": 1.58, "ionic_radius": 1.44}},
            }
        },
        "thermal_conductivity": 35,
        "van_der_waals_radius": 2.49,
        "electronegativity": 0.95,
        "nmr_quadrupole_moment": {"Sr-87": 305.2},
        "metallic_radius": 2.151,
        "iupac_ordering": 14,
    },
    "Ta": {
        "atomic_mass": 180.94788,
        "atomic_number": 73,
        "atomic_orbitals": {
            "1s": -2275.371387,
            "2p": -357.248334,
            "2s": -372.828724,
            "3d": -63.942521,
            "3p": -77.440942,
            "3s": -84.658467,
            "4d": -8.265848,
            "4f": -1.199347,
            "4p": -13.71981,
            "4s": -16.713337,
            "5d": -0.182464,
            "5p": -1.37653,
            "5s": -2.223807,
            "6s": -0.174814,
        },
        "atomic_radius": 1.45,
        "atomic_radius_calculated": 2.0,
        "boiling_point": 5731,
        "brinell_hardness": 800,
        "bulk_modulus": 200,
        "coefficient_of_linear_thermal_expansion": 6.3,
        "common_oxidation_states": [5],
        "density_of_solid": 16650,
        "electrical_resistivity": 13.5,
        "electronic_structure": "[Xe].4f14.5d3.6s2",
        "icsd_oxidation_states": [3, 4, 5],
        "ionic_radii": {"3": 0.86, "4": 0.82, "5": 0.78},
        "liquid_range": 2441,
        "melting_point": 3290,
        "mendeleev_number": 52,
        "mineral_hardness": 6.5,
        "molar_volume": 10.85,
        "name": "Tantalum",
        "oxidation_states": [-1, 2, 3, 4, 5],
        "poissons_ratio": 0.34,
        "reflectivity": 78,
        "rigidity_modulus": 69,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}}},
            "4": {"VI": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}}},
            "5": {
                "VI": {"": {"crystal_radius": 0.78, "ionic_radius": 0.64}},
                "VII": {"": {"crystal_radius": 0.83, "ionic_radius": 0.69}},
                "VIII": {"": {"crystal_radius": 0.88, "ionic_radius": 0.74}},
            },
        },
        "superconduction_temperature": 4.47,
        "thermal_conductivity": 57,
        "velocity_of_sound": 3400,
        "vickers_hardness": 873,
        "electronegativity": 1.5,
        "youngs_modulus": 186,
        "metallic_radius": 1.47,
        "iupac_ordering": 53,
    },
    "Tb": {
        "atomic_mass": 158.92535,
        "atomic_number": 65,
        "atomic_orbitals": {
            "1s": -1785.331942,
            "2p": -271.590585,
            "2s": -285.121013,
            "3d": -45.443863,
            "3p": -56.785113,
            "3s": -62.851563,
            "4d": -5.467662,
            "4f": -0.256311,
            "4p": -9.735637,
            "4s": -12.120486,
            "5p": -0.88723,
            "5s": -1.513669,
            "6s": -0.131677,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.25,
        "boiling_point": 3503,
        "brinell_hardness": 677,
        "bulk_modulus": 38.7,
        "coefficient_of_linear_thermal_expansion": 10.3,
        "common_oxidation_states": [3],
        "density_of_solid": 8219,
        "electrical_resistivity": 115,
        "electronic_structure": "[Xe].4f9.6s2",
        "icsd_oxidation_states": [3, 4],
        "ionic_radii": {"3": 1.063, "4": 0.9},
        "liquid_range": 1874,
        "melting_point": 1629,
        "mendeleev_number": 26,
        "molar_volume": 19.30,
        "name": "Terbium",
        "oxidation_states": [1, 3, 4],
        "poissons_ratio": 0.26,
        "rigidity_modulus": 22,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.063, "ionic_radius": 0.923}},
                "VII": {"": {"crystal_radius": 1.12, "ionic_radius": 0.98}},
                "VIII": {"": {"crystal_radius": 1.18, "ionic_radius": 1.04}},
                "IX": {"": {"crystal_radius": 1.235, "ionic_radius": 1.095}},
            },
            "4": {
                "VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
                "VIII": {"": {"crystal_radius": 1.02, "ionic_radius": 0.88}},
            },
        },
        "thermal_conductivity": 11,
        "velocity_of_sound": 2620,
        "vickers_hardness": 863,
        "electronegativity": 1.1,
        "youngs_modulus": 56,
        "metallic_radius": 1.781,
        "iupac_ordering": 39,
    },
    "Tc": {
        "atomic_mass": 98.0,
        "atomic_number": 43,
        "atomic_orbitals": {
            "1s": -745.742024,
            "2p": -96.61021,
            "2s": -104.567508,
            "3d": -9.33986,
            "3p": -15.041738,
            "3s": -18.135303,
            "4d": -0.270262,
            "4p": -1.64323,
            "4s": -2.550712,
            "5s": -0.183636,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.83,
        "boiling_point": 4538,
        "common_oxidation_states": [4, 7],
        "density_of_solid": 11500,
        "electrical_resistivity": 22,  # roughly this value
        "electronic_structure": "[Kr].4d5.5s2",
        "ionic_radii": {"4": 0.785, "5": 0.74, "7": 0.7},
        "liquid_range": 2108,
        "melting_point": 2430,
        "mendeleev_number": 59,
        "molar_volume": 8.63,
        "name": "Technetium",
        "oxidation_states": [-3, -1, 1, 2, 3, 4, 5, 6, 7],
        "shannon_radii": {
            "4": {"VI": {"": {"crystal_radius": 0.785, "ionic_radius": 0.645}}},
            "5": {"VI": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}}},
            "7": {
                "IV": {"": {"crystal_radius": 0.51, "ionic_radius": 0.37}},
                "VI": {"": {"crystal_radius": 0.7, "ionic_radius": 0.56}},
            },
        },
        "superconduction_temperature": 7.8,
        "thermal_conductivity": 51,
        "electronegativity": 1.9,
        "metallic_radius": 1.363,
        "iupac_ordering": 60,
    },
    "Te": {
        "atomic_mass": 127.6,
        "atomic_number": 52,
        "atomic_orbitals": {
            "1s": -1115.831819,
            "2p": -156.808583,
            "2s": -167.021776,
            "3d": -20.887801,
            "3p": -28.860685,
            "3s": -33.137485,
            "4d": -1.608381,
            "4p": -4.100084,
            "4s": -5.572846,
            "5p": -0.226594,
            "5s": -0.520997,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.23,
        "boiling_point": 1261,
        "brinell_hardness": 180,
        "bulk_modulus": 65,
        "common_oxidation_states": [-2, 2, 4, 6],
        "density_of_solid": 6240,
        "electrical_resistivity": 10000,  # roughly this value
        "electronic_structure": "[Kr].4d10.5s2.5p4",
        "icsd_oxidation_states": [-2, 4, -1, 6],
        "ionic_radii": {"-2": 2.07, "4": 1.11, "6": 0.7},
        "liquid_range": 538.34,
        "melting_point": 722.66,
        "mendeleev_number": 92,
        "mineral_hardness": 2.25,
        "molar_volume": 20.46,
        "name": "Tellurium",
        "oxidation_states": [-2, 2, 4, 5, 6],
        "reflectivity": 50,
        "refractive_index": 1.000991,
        "rigidity_modulus": 16,
        "shannon_radii": {
            "-2": {"VI": {"": {"crystal_radius": 2.07, "ionic_radius": 2.21}}},
            "4": {
                "III": {"": {"crystal_radius": 0.66, "ionic_radius": 0.52}},
                "IV": {"": {"crystal_radius": 0.8, "ionic_radius": 0.66}},
                "VI": {"": {"crystal_radius": 1.11, "ionic_radius": 0.97}},
            },
            "6": {
                "IV": {"": {"crystal_radius": 0.57, "ionic_radius": 0.43}},
                "VI": {"": {"crystal_radius": 0.7, "ionic_radius": 0.56}},
            },
        },
        "thermal_conductivity": 3,
        "van_der_waals_radius": 2.06,
        "velocity_of_sound": 2610,
        "electronegativity": 2.1,
        "youngs_modulus": 43,
        "iupac_ordering": 94,
    },
    "Th": {
        "atomic_mass": 232.03806,
        "atomic_number": 90,
        "atomic_orbitals": {
            "1s": -3524.439052,
            "2p": -588.218112,
            "2s": -608.350958,
            "3d": -123.846396,
            "3p": -142.25581,
            "3s": -152.079741,
            "4d": -24.955184,
            "4f": -13.397389,
            "4p": -33.325252,
            "4s": -37.814094,
            "5d": -3.625729,
            "5p": -6.58281,
            "5s": -8.287057,
            "6d": -0.172896,
            "6p": -0.846921,
            "6s": -1.333769,
            "7s": -0.135872,
        },
        "atomic_radius": 1.8,
        "boiling_point": 5093,
        "brinell_hardness": 400,
        "bulk_modulus": 54,
        "coefficient_of_linear_thermal_expansion": 11.0,
        "common_oxidation_states": [4],
        "density_of_solid": 11724,
        "electrical_resistivity": 15,
        "electronic_structure": "[Rn].6d2.7s2",
        "icsd_oxidation_states": [4],
        "ionic_radii": {"4": 1.08},
        "liquid_range": 2978,
        "melting_point": 2115,
        "mendeleev_number": 47,
        "mineral_hardness": 3.0,
        "molar_volume": 19.80,
        "name": "Thorium",
        "oxidation_states": [2, 3, 4],
        "poissons_ratio": 0.27,
        "rigidity_modulus": 31,
        "shannon_radii": {
            "4": {
                "VI": {"": {"crystal_radius": 1.08, "ionic_radius": 0.94}},
                "VIII": {"": {"crystal_radius": 1.19, "ionic_radius": 1.05}},
                "IX": {"": {"crystal_radius": 1.23, "ionic_radius": 1.09}},
                "electronegativity": {
                    "": {"crystal_radius": 1.27, "ionic_radius": 1.13}
                },
                "XI": {"": {"crystal_radius": 1.32, "ionic_radius": 1.18}},
                "XII": {"": {"crystal_radius": 1.35, "ionic_radius": 1.21}},
            }
        },
        "superconduction_temperature": 1.38,
        "thermal_conductivity": 54,
        "velocity_of_sound": 2490,
        "vickers_hardness": 350,
        "electronegativity": 1.3,
        "youngs_modulus": 79,
        "metallic_radius": 1.798,
        "iupac_ordering": 31,
    },
    "Ti": {
        "atomic_mass": 47.867,
        "atomic_number": 22,
        "atomic_orbitals": {
            "1s": -177.276643,
            "2p": -16.285339,
            "2s": -19.457901,
            "3d": -0.17001,
            "3p": -1.422947,
            "3s": -2.258007,
            "4s": -0.167106,
        },
        "atomic_radius": 1.4,
        "atomic_radius_calculated": 1.76,
        "boiling_point": 3560,
        "brinell_hardness": 716,
        "bulk_modulus": 110,
        "coefficient_of_linear_thermal_expansion": 8.6,
        "common_oxidation_states": [4],
        "density_of_solid": 4507,
        "electrical_resistivity": 40,  # roughly this value
        "electronic_structure": "[Ar].3d2.4s2",
        "icsd_oxidation_states": [2, 3, 4],
        "ionic_radii": {"2": 1.0, "3": 0.81, "4": 0.745},
        "liquid_range": 1619,
        "melting_point": 1941,
        "mendeleev_number": 51,
        "mineral_hardness": 6.0,
        "molar_volume": 10.64,
        "name": "Titanium",
        "oxidation_states": [-1, 2, 3, 4],
        "poissons_ratio": 0.32,
        "rigidity_modulus": 44,
        "shannon_radii": {
            "2": {"VI": {"": {"crystal_radius": 1.0, "ionic_radius": 0.86}}},
            "3": {"VI": {"": {"crystal_radius": 0.81, "ionic_radius": 0.67}}},
            "4": {
                "IV": {"": {"crystal_radius": 0.56, "ionic_radius": 0.42}},
                "V": {"": {"crystal_radius": 0.65, "ionic_radius": 0.51}},
                "VI": {"": {"crystal_radius": 0.745, "ionic_radius": 0.605}},
                "VIII": {"": {"crystal_radius": 0.88, "ionic_radius": 0.74}},
            },
        },
        "superconduction_temperature": 0.40,
        "thermal_conductivity": 22,
        "velocity_of_sound": 4140,
        "vickers_hardness": 970,
        "electronegativity": 1.54,
        "youngs_modulus": 116,
        "nmr_quadrupole_moment": {"Ti-47": 302.1, "Ti-49": 247.11},
        "metallic_radius": 1.462,
        "iupac_ordering": 52,
    },
    "Tl": {
        "atomic_mass": 204.3833,
        "atomic_number": 81,
        "atomic_orbitals": {
            "1s": -2827.569408,
            "2p": -457.255971,
            "2s": -474.953368,
            "3d": -88.328299,
            "3p": -104.099296,
            "3s": -112.52218,
            "4d": -14.008848,
            "4f": -4.835747,
            "4p": -20.797078,
            "4s": -24.471512,
            "5d": -0.674544,
            "5p": -2.59873,
            "5s": -3.811512,
            "6p": -0.101507,
            "6s": -0.28502,
        },
        "atomic_radius": 1.9,
        "atomic_radius_calculated": 1.56,
        "boiling_point": 1746,
        "brinell_hardness": 26.4,
        "bulk_modulus": 43,
        "coefficient_of_linear_thermal_expansion": 29.9,
        "common_oxidation_states": [1, 3],
        "density_of_solid": 11850,
        "electrical_resistivity": 15,
        "electronic_structure": "[Xe].4f14.5d10.6s2.6p1",
        "icsd_oxidation_states": [1, 3],
        "ionic_radii": {"1": 1.64, "3": 1.025},
        "liquid_range": 1169,
        "melting_point": 577,
        "mendeleev_number": 78,
        "mineral_hardness": 1.2,
        "molar_volume": 17.22,
        "name": "Thallium",
        "oxidation_states": [1, 3],
        "poissons_ratio": 0.45,
        "rigidity_modulus": 2.8,
        "shannon_radii": {
            "1": {
                "VI": {"": {"crystal_radius": 1.64, "ionic_radius": 1.5}},
                "VIII": {"": {"crystal_radius": 1.73, "ionic_radius": 1.59}},
                "XII": {"": {"crystal_radius": 1.84, "ionic_radius": 1.7}},
            },
            "3": {
                "IV": {"": {"crystal_radius": 0.89, "ionic_radius": 0.75}},
                "VI": {"": {"crystal_radius": 1.025, "ionic_radius": 0.885}},
                "VIII": {"": {"crystal_radius": 1.12, "ionic_radius": 0.98}},
            },
        },
        "superconduction_temperature": 2.38,
        "thermal_conductivity": 46,
        "van_der_waals_radius": 1.96,
        "velocity_of_sound": 818,
        "electronegativity": 1.62,
        "youngs_modulus": 8,
        "metallic_radius": 1.7,
        "iupac_ordering": 77,
    },
    "Tm": {
        "atomic_mass": 168.93421,
        "atomic_number": 69,
        "atomic_orbitals": {
            "1s": -2022.471608,
            "2p": -312.510608,
            "2s": -327.05712,
            "3d": -53.835494,
            "3p": -66.239338,
            "3s": -72.873753,
            "4d": -6.350307,
            "4f": -0.28312,
            "4p": -11.187151,
            "4s": -13.865665,
            "5p": -0.950748,
            "5s": -1.64999,
            "6s": -0.135953,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.22,
        "boiling_point": 2223,
        "brinell_hardness": 471,
        "bulk_modulus": 45,
        "coefficient_of_linear_thermal_expansion": 13.3,
        "common_oxidation_states": [3],
        "density_of_solid": 9321,
        "electrical_resistivity": 67.6,
        "electronic_structure": "[Xe].4f13.6s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"2": 1.17, "3": 1.02},
        "liquid_range": 405,
        "melting_point": 1818,
        "mendeleev_number": 21,
        "molar_volume": 19.1,
        "name": "Thulium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.21,
        "rigidity_modulus": 31,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.17, "ionic_radius": 1.03}},
                "VII": {"": {"crystal_radius": 1.23, "ionic_radius": 1.09}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.02, "ionic_radius": 0.88}},
                "VIII": {"": {"crystal_radius": 1.134, "ionic_radius": 0.994}},
                "IX": {"": {"crystal_radius": 1.192, "ionic_radius": 1.052}},
            },
        },
        "thermal_conductivity": 17,
        "vickers_hardness": 520,
        "electronegativity": 1.25,
        "youngs_modulus": 74,
        "metallic_radius": 1.747,
        "iupac_ordering": 35,
    },
    "U": {
        "atomic_mass": 238.02891,
        "atomic_number": 92,
        "atomic_orbitals": {
            "1s": -3689.355141,
            "2p": -619.10855,
            "2s": -639.778728,
            "3d": -131.977358,
            "3p": -150.97898,
            "3s": -161.118073,
            "4d": -27.123212,
            "4f": -15.02746,
            "4p": -35.853321,
            "4s": -40.528084,
            "5d": -3.866175,
            "5f": -0.366543,
            "5p": -7.018092,
            "5s": -8.824089,
            "6d": -0.14319,
            "6p": -0.822538,
            "6s": -1.325976,
            "7s": -0.130948,
        },
        "atomic_radius": 1.75,
        "boiling_point": 4200,
        "brinell_hardness": 2400,
        "bulk_modulus": 100,
        "coefficient_of_linear_thermal_expansion": 13.9,
        "common_oxidation_states": [6],
        "density_of_solid": 19050,
        "electrical_resistivity": 28,
        "electronic_structure": "[Rn].5f3.6d1.7s2",
        "icsd_oxidation_states": [3, 4, 5, 6],
        "ionic_radii": {"3": 1.165, "4": 1.03, "5": 0.9, "6": 0.87},
        "liquid_range": 2794.7,
        "melting_point": 1405.3,
        "mendeleev_number": 45,
        "mineral_hardness": 6.0,
        "molar_volume": 12.49,
        "name": "Uranium",
        "oxidation_states": [3, 4, 5, 6],
        "poissons_ratio": 0.23,
        "rigidity_modulus": 111,
        "shannon_radii": {
            "3": {"VI": {"": {"crystal_radius": 1.165, "ionic_radius": 1.025}}},
            "4": {
                "VI": {"": {"crystal_radius": 1.03, "ionic_radius": 0.89}},
                "VII": {"": {"crystal_radius": 1.09, "ionic_radius": 0.95}},
                "VIII": {"": {"crystal_radius": 1.14, "ionic_radius": 1.0}},
                "IX": {"": {"crystal_radius": 1.19, "ionic_radius": 1.05}},
                "XII": {"": {"crystal_radius": 1.31, "ionic_radius": 1.17}},
            },
            "5": {
                "VI": {"": {"crystal_radius": 0.9, "ionic_radius": 0.76}},
                "VII": {"": {"crystal_radius": 0.98, "ionic_radius": 0.84}},
            },
            "6": {
                "II": {"": {"crystal_radius": 0.59, "ionic_radius": 0.45}},
                "IV": {"": {"crystal_radius": 0.66, "ionic_radius": 0.52}},
                "VI": {"": {"crystal_radius": 0.87, "ionic_radius": 0.73}},
                "VII": {"": {"crystal_radius": 0.95, "ionic_radius": 0.81}},
                "VIII": {"": {"crystal_radius": 1.0, "ionic_radius": 0.86}},
            },
        },
        "superconduction_temperature": 0.2,
        "thermal_conductivity": 27,
        "van_der_waals_radius": 1.86,
        "velocity_of_sound": 3155,
        "vickers_hardness": 1960,
        "electronegativity": 1.38,
        "youngs_modulus": 208,
        "metallic_radius": 1.542,
        "iupac_ordering": 29,
    },
    "V": {
        "atomic_mass": 50.9415,
        "atomic_number": 23,
        "atomic_orbitals": {
            "1s": -195.224014,
            "2p": -18.435189,
            "2s": -21.815346,
            "3d": -0.204634,
            "3p": -1.610516,
            "3s": -2.526904,
            "4s": -0.175968,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.71,
        "boiling_point": 3680,
        "brinell_hardness": 628,
        "bulk_modulus": 160,
        "coefficient_of_linear_thermal_expansion": 8.4,
        "common_oxidation_states": [5],
        "density_of_solid": 6110,
        "electrical_resistivity": 20,
        "electronic_structure": "[Ar].3d3.4s2",
        "icsd_oxidation_states": [2, 3, 4, 5],
        "ionic_radii": {"2": 0.93, "3": 0.78, "4": 0.72, "5": 0.68},
        "liquid_range": 1497,
        "melting_point": 2183,
        "mendeleev_number": 54,
        "mineral_hardness": 7.0,
        "molar_volume": 8.32,
        "name": "Vanadium",
        "oxidation_states": [-1, 1, 2, 3, 4, 5],
        "poissons_ratio": 0.37,
        "reflectivity": 61,
        "rigidity_modulus": 47,
        "shannon_radii": {
            "2": {"VI": {"": {"crystal_radius": 0.93, "ionic_radius": 0.79}}},
            "3": {"VI": {"": {"crystal_radius": 0.78, "ionic_radius": 0.64}}},
            "4": {
                "V": {"": {"crystal_radius": 0.67, "ionic_radius": 0.53}},
                "VI": {"": {"crystal_radius": 0.72, "ionic_radius": 0.58}},
                "VIII": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}},
            },
            "5": {
                "IV": {"": {"crystal_radius": 0.495, "ionic_radius": 0.355}},
                "V": {"": {"crystal_radius": 0.6, "ionic_radius": 0.46}},
                "VI": {"": {"crystal_radius": 0.68, "ionic_radius": 0.54}},
            },
        },
        "superconduction_temperature": 5.40,
        "thermal_conductivity": 31,
        "velocity_of_sound": 4560,
        "vickers_hardness": 628,
        "electronegativity": 1.63,
        "youngs_modulus": 128,
        "nmr_quadrupole_moment": {"V-50": 210.4, "V-51": -52.1},
        "metallic_radius": 1.347,
        "iupac_ordering": 55,
    },
    "W": {
        "atomic_mass": 183.84,
        "atomic_number": 74,
        "atomic_orbitals": {
            "1s": -2341.042887,
            "2p": -369.013973,
            "2s": -384.856157,
            "3d": -66.724787,
            "3p": -80.502102,
            "3s": -87.867792,
            "4d": -8.879693,
            "4f": -1.550835,
            "4p": -14.495102,
            "4s": -17.570797,
            "5d": -0.220603,
            "5p": -1.504457,
            "5s": -2.396018,
            "6s": -0.181413,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.93,
        "boiling_point": 5828,
        "brinell_hardness": 2570,
        "bulk_modulus": 310,
        "coefficient_of_linear_thermal_expansion": 4.5,
        "common_oxidation_states": [4, 6],
        "density_of_solid": 19250,
        "electrical_resistivity": 5.4,
        "electronic_structure": "[Xe].4f14.5d4.6s2",
        "icsd_oxidation_states": [2, 3, 4, 5, 6],
        "ionic_radii": {"4": 0.8, "5": 0.76, "6": 0.74},
        "liquid_range": 2133,
        "melting_point": 3695,
        "mendeleev_number": 55,
        "mineral_hardness": 7.5,
        "molar_volume": 9.47,
        "name": "Tungsten",
        "oxidation_states": [-2, -1, 1, 2, 3, 4, 5, 6],
        "poissons_ratio": 0.28,
        "reflectivity": 62,
        "rigidity_modulus": 161,
        "shannon_radii": {
            "4": {"VI": {"": {"crystal_radius": 0.8, "ionic_radius": 0.66}}},
            "5": {"VI": {"": {"crystal_radius": 0.76, "ionic_radius": 0.62}}},
            "6": {
                "IV": {"": {"crystal_radius": 0.56, "ionic_radius": 0.42}},
                "V": {"": {"crystal_radius": 0.65, "ionic_radius": 0.51}},
                "VI": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}},
            },
        },
        "superconduction_temperature": 0.015,
        "thermal_conductivity": 170,
        "velocity_of_sound": 5174,
        "vickers_hardness": 3430,
        "electronegativity": 2.36,
        "youngs_modulus": 411,
        "metallic_radius": 1.41,
        "iupac_ordering": 56,
    },
    "Xe": {
        "atomic_mass": 131.293,
        "atomic_number": 54,
        "atomic_orbitals": {
            "1s": -1208.688993,
            "2p": -172.599583,
            "2s": -183.327495,
            "3d": -24.37823,
            "3p": -32.867042,
            "3s": -37.415454,
            "4d": -2.286666,
            "4p": -5.063802,
            "4s": -6.67834,
            "5p": -0.309835,
            "5s": -0.672086,
        },
        "atomic_radius_calculated": 1.08,
        "boiling_point": 165.1,
        "critical_temperature": 289.7,
        "electronic_structure": "[Kr].4d10.5s2.5p6",
        "ionic_radii": {"8": 0.62},
        "liquid_range": 3.7,
        "max_oxidation_state": 8.0,
        "melting_point": 161.4,
        "mendeleev_number": 5,
        "min_oxidation_state": 2.0,
        "molar_volume": 35.92,
        "name": "Xenon",
        "refractive_index": 1.000702,
        "shannon_radii": {
            "8": {
                "IV": {"": {"crystal_radius": 0.54, "ionic_radius": 0.4}},
                "VI": {"": {"crystal_radius": 0.62, "ionic_radius": 0.48}},
            }
        },
        "thermal_conductivity": 0.00565,
        "van_der_waals_radius": 2.16,
        "velocity_of_sound": 1090,
        "electronegativity": 2.6,
        "iupac_ordering": 1,
    },
    "Y": {
        "atomic_mass": 88.90585,
        "atomic_number": 39,
        "atomic_orbitals": {
            "1s": -605.631981,
            "2p": -74.803201,
            "2s": -81.789102,
            "3d": -5.671499,
            "3p": -10.399926,
            "3s": -12.992217,
            "4d": -0.108691,
            "4p": -1.02449,
            "4s": -1.697124,
            "5s": -0.150727,
        },
        "atomic_radius": 1.8,
        "atomic_radius_calculated": 2.12,
        "boiling_point": 3609,
        "brinell_hardness": 589,
        "bulk_modulus": 41,
        "coefficient_of_linear_thermal_expansion": 10.6,
        "common_oxidation_states": [3],
        "density_of_solid": 4472,
        "electrical_resistivity": 60,  # roughly this value
        "electronic_structure": "[Kr].4d1.5s2",
        "icsd_oxidation_states": [3],
        "ionic_radii": {"3": 1.04},
        "liquid_range": 1810,
        "melting_point": 1799,
        "mendeleev_number": 25,
        "molar_volume": 19.88,
        "name": "Yttrium",
        "oxidation_states": [1, 2, 3],
        "poissons_ratio": 0.24,
        "rigidity_modulus": 26,
        "shannon_radii": {
            "3": {
                "VI": {"": {"crystal_radius": 1.04, "ionic_radius": 0.9}},
                "VII": {"": {"crystal_radius": 1.1, "ionic_radius": 0.96}},
                "VIII": {"": {"crystal_radius": 1.159, "ionic_radius": 1.019}},
                "IX": {"": {"crystal_radius": 1.215, "ionic_radius": 1.075}},
            }
        },
        "superconduction_temperature": "1.3  (under pressure)K",
        "thermal_conductivity": 17,
        "velocity_of_sound": 3300,
        "electronegativity": 1.22,
        "youngs_modulus": 64,
        "metallic_radius": 1.8,
        "iupac_ordering": 48,
    },
    "Yb": {
        "atomic_mass": 173.04,
        "atomic_number": 70,
        "atomic_orbitals": {
            "1s": -2084.069389,
            "2p": -323.178219,
            "2s": -337.978976,
            "3d": -56.026315,
            "3p": -68.698655,
            "3s": -75.47663,
            "4d": -6.574963,
            "4f": -0.286408,
            "4p": -11.558246,
            "4s": -14.312076,
            "5p": -0.966137,
            "5s": -1.683886,
            "6s": -0.136989,
        },
        "atomic_radius": 1.75,
        "atomic_radius_calculated": 2.22,
        "boiling_point": 1469,
        "brinell_hardness": 343,
        "bulk_modulus": 31,
        "coefficient_of_linear_thermal_expansion": 26.3,
        "common_oxidation_states": [3],
        "density_of_solid": 6570,
        "electrical_resistivity": 25.0,
        "electronic_structure": "[Xe].4f14.6s2",
        "icsd_oxidation_states": [2, 3],
        "ionic_radii": {"2": 1.16, "3": 1.008},
        "liquid_range": 372,
        "melting_point": 1097,
        "mendeleev_number": 17,
        "molar_volume": 24.84,
        "name": "Ytterbium",
        "oxidation_states": [2, 3],
        "poissons_ratio": 0.21,
        "rigidity_modulus": 9.9,
        "shannon_radii": {
            "2": {
                "VI": {"": {"crystal_radius": 1.16, "ionic_radius": 1.02}},
                "VII": {"": {"crystal_radius": 1.22, "ionic_radius": 1.08}},
                "VIII": {"": {"crystal_radius": 1.28, "ionic_radius": 1.14}},
            },
            "3": {
                "VI": {"": {"crystal_radius": 1.008, "ionic_radius": 0.868}},
                "VII": {"": {"crystal_radius": 1.065, "ionic_radius": 0.925}},
                "VIII": {"": {"crystal_radius": 1.125, "ionic_radius": 0.985}},
                "IX": {"": {"crystal_radius": 1.182, "ionic_radius": 1.042}},
            },
        },
        "thermal_conductivity": 39,
        "velocity_of_sound": 1590,
        "vickers_hardness": 206,
        "electronegativity": 1.1,
        "youngs_modulus": 24,
        "metallic_radius": 1.94,
        "iupac_ordering": 34,
    },
    "Zn": {
        "atomic_mass": 65.409,
        "atomic_number": 30,
        "atomic_orbitals": {
            "1s": -344.969756,
            "2p": -36.648765,
            "2s": -41.531323,
            "3d": -0.398944,
            "3p": -3.022363,
            "3s": -4.573041,
            "4s": -0.222725,
        },
        "atomic_radius": 1.35,
        "atomic_radius_calculated": 1.42,
        "boiling_point": 1180,
        "brinell_hardness": 412,
        "bulk_modulus": 70,
        "coefficient_of_linear_thermal_expansion": 30.2,
        "common_oxidation_states": [2],
        "density_of_solid": 7140,
        "electrical_resistivity": 6.0,
        "electronic_structure": "[Ar].3d10.4s2",
        "icsd_oxidation_states": [2],
        "ionic_radii": {"2": 0.88},
        "liquid_range": 487.32,
        "melting_point": 692.68,
        "mendeleev_number": 76,
        "mineral_hardness": 2.5,
        "molar_volume": 9.16,
        "name": "Zinc",
        "oxidation_states": [1, 2],
        "poissons_ratio": 0.25,
        "reflectivity": 80,
        "refractive_index": 1.002050,
        "rigidity_modulus": 43,
        "shannon_radii": {
            "2": {
                "IV": {"": {"crystal_radius": 0.74, "ionic_radius": 0.6}},
                "V": {"": {"crystal_radius": 0.82, "ionic_radius": 0.68}},
                "VI": {"": {"crystal_radius": 0.88, "ionic_radius": 0.74}},
                "VIII": {"": {"crystal_radius": 1.04, "ionic_radius": 0.9}},
            }
        },
        "superconduction_temperature": 0.85,
        "thermal_conductivity": 120,
        "van_der_waals_radius": 1.39,
        "velocity_of_sound": 3700,
        "electronegativity": 1.65,
        "youngs_modulus": 108,
        "nmr_quadrupole_moment": {"Zn-67": 150.15},
        "metallic_radius": 1.34,
        "iupac_ordering": 76,
    },
    "Zr": {
        "atomic_mass": 91.224,
        "atomic_number": 40,
        "atomic_orbitals": {
            "1s": -639.292236,
            "2p": -80.010043,
            "2s": -87.237062,
            "3d": -6.544643,
            "3p": -11.514415,
            "3s": -14.230432,
            "4d": -0.150673,
            "4p": -1.186597,
            "4s": -1.918971,
            "5s": -0.162391,
        },
        "atomic_radius": 1.55,
        "atomic_radius_calculated": 2.06,
        "boiling_point": 4682,
        "brinell_hardness": 650,
        "coefficient_of_linear_thermal_expansion": 5.7,
        "common_oxidation_states": [4],
        "density_of_solid": 6511,
        "electrical_resistivity": 43.3,
        "electronic_structure": "[Kr].4d2.5s2",
        "icsd_oxidation_states": [2, 3, 4],
        "ionic_radii": {"4": 0.86},
        "liquid_range": 2554,
        "melting_point": 2128,
        "mendeleev_number": 49,
        "mineral_hardness": 5.0,
        "molar_volume": 14.02,
        "name": "Zirconium",
        "oxidation_states": [1, 2, 3, 4],
        "poissons_ratio": 0.34,
        "rigidity_modulus": 33,
        "shannon_radii": {
            "4": {
                "IV": {"": {"crystal_radius": 0.73, "ionic_radius": 0.59}},
                "V": {"": {"crystal_radius": 0.8, "ionic_radius": 0.66}},
                "VI": {"": {"crystal_radius": 0.86, "ionic_radius": 0.72}},
                "VII": {"": {"crystal_radius": 0.92, "ionic_radius": 0.78}},
                "VIII": {"": {"crystal_radius": 0.98, "ionic_radius": 0.84}},
                "IX": {"": {"crystal_radius": 1.03, "ionic_radius": 0.89}},
            }
        },
        "superconduction_temperature": 0.61,
        "thermal_conductivity": 23,
        "velocity_of_sound": 3800,
        "vickers_hardness": 903,
        "electronegativity": 1.33,
        "youngs_modulus": 68,
        "metallic_radius": 1.602,
        "iupac_ordering": 51,
    },
    "Rf": {
        "atomic_mass": 267,
        "atomic_number": 104,
        "name": "Rutherfordium",
    },
    "Db": {
        "atomic_mass": 268,
        "atomic_number": 105,
        "name": "Dubnium",
    },
    "Sg": {
        "atomic_mass": 269,
        "atomic_number": 106,
        "name": "Seaborgium",
    },
    "Bh": {
        "atomic_mass": 270,
        "atomic_number": 107,
        "name": "Bohrium",
    },
    "Hs": {
        "atomic_mass": 270,
        "atomic_number": 108,
        "name": "Hassium",
    },
    "Mt": {
        "atomic_mass": 278,
        "atomic_number": 109,
        "name": "Meitnerium",
    },
    "Ds": {
        "atomic_mass": 281,
        "atomic_number": 110,
        "name": "Darmstadtium",
    },
    "Rg": {
        "atomic_mass": 282,
        "atomic_number": 111,
        "name": "Roentgenium",
    },
    "Cn": {
        "atomic_mass": 285,
        "atomic_number": 112,
        "name": "Copernicium",
    },
    "Nh": {
        "atomic_mass": 286,
        "atomic_number": 113,
        "name": "Nihonium",
    },
    "Fl": {
        "atomic_mass": 289,
        "atomic_number": 114,
        "name": "Flerovium",
    },
    "Mc": {
        "atomic_mass": 290,
        "atomic_number": 115,
        "name": "Moscovium",
    },
    "Lv": {
        "atomic_mass": 293,
        "atomic_number": 116,
        "name": "Livermorium",
    },
    "Ts": {
        "atomic_mass": 294,
        "atomic_number": 117,
        "name": "Tennessine",
    },
    "Og": {
        "atomic_mass": 294,
        "atomic_number": 118,
        "name": "Oganesson",
    },
    # This is a dummy element.
    "X": {"atomic_number": 0, "name": "Unknown"},
}
