import json
from os import path
import re

from django import template
from django.conf import settings
from django.utils.safestring import mark_safe
from django.templatetags.static import static

register = template.Library()

is_absolute_url = lambda url: re.match("^https?://", url)


def vite_manifest(app_name, entries_names):
    manifest_filepath = path.join(app_name, 'static/manifest.json')
    VITE_SERVER_URL = getattr(settings, 'VITE_SERVER_URL', None)

    if VITE_SERVER_URL:
        scripts = [
            f"{VITE_SERVER_URL}/@vite/client",
        ]
        for name in entries_names:
            scripts.append(f'{VITE_SERVER_URL}/{name}')
        styles = []
        return scripts, styles
    else:
        with open(manifest_filepath) as fp:
            manifest = json.load(fp)
        _processed = set()

        def _process_entries(names):
            scripts = []
            styles = []

            for name in names:
                if name in _processed:
                    continue

                chunk = manifest[name]

                import_scripts, import_styles = _process_entries(chunk.get('imports', []))
                scripts += import_scripts
                styles += import_styles

                scripts += [chunk['file']]
                styles += [css for css in chunk.get('css', [])]

                _processed.add(name)

            return scripts, styles

        return _process_entries(entries_names)


@register.simple_tag(takes_context=True)
def vite_styles(context, *entries_names):
    """
    Populate an html template with styles generated by vite

    Usage::

        {% vite_styles 'main.js' 'other-entry.js' %}

    Examples::
        <head>
            ...
            {% vite_styles 'main.js' 'other-entry.js' %}
        </head>
    """
    app_name = context['request'].resolver_match.app_name
    _, styles = vite_manifest(app_name, entries_names)
    styles = map(lambda href: href if is_absolute_url(href) else static(href), styles)
    as_link_tag = lambda href: f'<link rel="stylesheet" href="{href}" />'
    return mark_safe("\n".join(map(as_link_tag, styles)))


@register.simple_tag(takes_context=True)
def vite_scripts(context, *entries_names):
    """
    Populate an html template with script tags generated by vite

    Usage::

        {% vite_scripts 'main.js' 'other-entry.js' %}

    Examples::
        <body>
            <!-- Your HTML -->
            {% vite_scripts 'main.js' 'other-entry.js' %}
        </body>
    """
    app_name = context['request'].resolver_match.app_name
    scripts, _ = vite_manifest(app_name, entries_names)
    scripts = map(lambda src: src if is_absolute_url(src) else static(src), scripts)
    as_script_tag = lambda src: f'<script type="module" src="{src}"></script>'
    return mark_safe("\n".join(map(as_script_tag, scripts)))
