"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.replaceListenerRulePriority = exports.destroyListenerRule = exports.fetchListenerRulePriority = exports.createNewListenerRule = exports.fetchFreePriority = exports.deallocPriority = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_dynamodb_1 = require("@aws-sdk/client-dynamodb");
const errors_1 = require("./errors");
const PRIORITY_INIT = parseInt(process.env.PRIORITY_INIT || '1', 10);
const PRIORITY_STEP = 1;
async function createMaxPriority(ctx) {
    try {
        await ctx.client.send(new client_dynamodb_1.PutItemCommand({
            TableName: ctx.tableName,
            Item: {
                pk: { S: 'max' },
                priority: { N: PRIORITY_INIT.toString() },
            },
            ConditionExpression: 'attribute_not_exists(pk)',
        }));
        return PRIORITY_INIT;
    }
    catch (e) {
        if (e.name === 'ConditionalCheckFailedException') {
            throw new errors_1.PriorityAlreadySetError();
        }
        throw e;
    }
}
async function updateMaxPriority(ctx) {
    const { Attributes } = await ctx.client.send(new client_dynamodb_1.UpdateItemCommand({
        TableName: ctx.tableName,
        Key: {
            pk: { S: 'max' },
        },
        UpdateExpression: 'ADD priority :one',
        ExpressionAttributeValues: {
            ':one': { N: PRIORITY_STEP.toString() },
        },
        ReturnValues: 'UPDATED_NEW',
    }));
    return parseInt(Attributes?.priority.N);
}
async function upsertFreePrioritiesList(ctx) {
    try {
        await ctx.client.send(new client_dynamodb_1.PutItemCommand({
            TableName: ctx.tableName,
            Item: {
                pk: { S: 'free' },
                priorities: { L: [] },
            },
            ConditionExpression: 'attribute_not_exists(pk)',
        }));
    }
    catch (e) {
        if (e.name !== 'ConditionalCheckFailedException') {
            throw e;
        }
        // ignore error if free priorities list already exists
    }
}
async function fetchFreePriorityFromList(ctx) {
    try {
        const { Attributes } = await ctx.client.send(new client_dynamodb_1.UpdateItemCommand({
            TableName: ctx.tableName,
            Key: {
                pk: { S: 'free' },
            },
            UpdateExpression: 'REMOVE priorities[0]',
            ReturnValues: 'ALL_OLD',
        }));
        return Attributes?.priorities?.L?.[0]?.N ? parseInt(Attributes.priorities.L[0].N) : undefined;
    }
    catch (e) {
        if (e.name === 'ValidationException') {
            // It's likely to be "The document path provided in the update expression is invalid for update"
            // returned when the free priorities list has not been created yet
            return undefined;
        }
        throw e;
    }
}
const deallocPriorityPayload = (ctx, priority) => ({
    TableName: ctx.tableName,
    Key: {
        pk: { S: 'free' },
    },
    UpdateExpression: 'SET priorities = list_append(priorities, :priorities)',
    ExpressionAttributeValues: {
        ':priorities': { L: [{ N: priority.toString() }] },
    },
});
async function deallocPriority(ctx, priority) {
    await upsertFreePrioritiesList(ctx);
    await ctx.client.send(new client_dynamodb_1.UpdateItemCommand(deallocPriorityPayload(ctx, priority)));
}
exports.deallocPriority = deallocPriority;
async function incrMaxPriority(ctx) {
    try {
        return await createMaxPriority(ctx);
    }
    catch (e) {
        if (e instanceof errors_1.PriorityAlreadySetError) {
            return updateMaxPriority(ctx);
        }
        throw e;
    }
}
async function fetchFreePriority(ctx) {
    return (await fetchFreePriorityFromList(ctx)) ?? (await incrMaxPriority(ctx));
}
exports.fetchFreePriority = fetchFreePriority;
const createListenerRulePriorityPayload = (ctx, props) => ({
    TableName: ctx.tableName,
    Item: {
        pk: { S: `priority#${props.priority}` },
        rule: { S: props.rulePath },
        priority: { N: props.priority.toString() },
    },
    ConditionExpression: 'attribute_not_exists(pk)',
});
async function createNewListenerRule(ctx, props) {
    try {
        await ctx.client.send(new client_dynamodb_1.TransactWriteItemsCommand({
            TransactItems: [
                {
                    Put: {
                        TableName: ctx.tableName,
                        Item: {
                            pk: { S: `rule#${props.rulePath}` },
                            rule: { S: props.rulePath },
                            priority: { N: props.priority.toString() },
                        },
                        ConditionExpression: 'attribute_not_exists(pk)',
                    },
                },
                {
                    Put: createListenerRulePriorityPayload(ctx, props),
                },
            ],
        }));
    }
    catch (e) {
        if (e instanceof client_dynamodb_1.TransactionCanceledException) {
            const codes = e.CancellationReasons ? e.CancellationReasons?.map((reason) => reason?.Code) : [];
            if (codes[0] === 'ConditionalCheckFailed') {
                throw new errors_1.ListenerRuleAlreadyExistsError({ rulePath: props.rulePath });
            }
            if (codes[1] === 'ConditionalCheckFailed') {
                throw new errors_1.ListenerRulePriorityAlreadyTakenError({ rulePath: props.rulePath }, props.priority);
            }
            throw e;
        }
        throw e;
    }
}
exports.createNewListenerRule = createNewListenerRule;
async function fetchListenerRulePriority(ctx, props) {
    const { Item } = await ctx.client.send(new client_dynamodb_1.GetItemCommand({
        TableName: ctx.tableName,
        Key: {
            pk: { S: `rule#${props.rulePath}` },
        },
    }));
    if (!Item?.priority?.N) {
        throw new errors_1.ListenerRuleNotFoundError(props);
    }
    return parseInt(Item.priority.N);
}
exports.fetchListenerRulePriority = fetchListenerRulePriority;
const deleteListenerRulePriorityPayload = (ctx, props) => ({
    TableName: ctx.tableName,
    Key: {
        pk: { S: `priority#${props.priority}` },
    },
});
async function destroyListenerRule(ctx, props) {
    await upsertFreePrioritiesList(ctx);
    await ctx.client.send(new client_dynamodb_1.TransactWriteItemsCommand({
        TransactItems: [
            {
                Delete: {
                    TableName: ctx.tableName,
                    Key: {
                        pk: { S: `rule#${props.rulePath}` },
                    },
                },
            },
            {
                Delete: deleteListenerRulePriorityPayload(ctx, props),
            },
            {
                Update: deallocPriorityPayload(ctx, props.priority),
            },
        ],
    }));
}
exports.destroyListenerRule = destroyListenerRule;
async function replaceListenerRulePriority(ctx, props) {
    await upsertFreePrioritiesList(ctx);
    try {
        await ctx.client.send(new client_dynamodb_1.TransactWriteItemsCommand({
            TransactItems: [
                {
                    Update: {
                        TableName: ctx.tableName,
                        Key: {
                            pk: { S: `rule#${props.rulePath}` },
                        },
                        UpdateExpression: 'SET priority = :priority',
                        ExpressionAttributeValues: {
                            ':priority': { N: props.priority.toString() },
                        },
                    },
                },
                {
                    Put: createListenerRulePriorityPayload(ctx, props),
                },
                {
                    Delete: deleteListenerRulePriorityPayload(ctx, {
                        priority: props.oldPriority,
                    }),
                },
                {
                    Update: deallocPriorityPayload(ctx, props.oldPriority),
                },
            ],
        }));
    }
    catch (e) {
        if (e instanceof client_dynamodb_1.TransactionCanceledException) {
            const codes = e.CancellationReasons ? e.CancellationReasons?.map((reason) => reason?.Code) : [];
            if (codes[1] === 'ConditionalCheckFailed') {
                throw new errors_1.ListenerRulePriorityAlreadyTakenError({ rulePath: props.rulePath }, props.priority);
            }
            throw e;
        }
    }
}
exports.replaceListenerRulePriority = replaceListenerRulePriority;
//# sourceMappingURL=data:application/json;base64,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