"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Networking = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
/**
 * The Networking construct creates a VPC which can have public, private, and isolated subnets.
 * It enforces to define a CIDR block for the VPC, which is a best practice.
 *
 * If the `natGateways` property is set to a positive integer, the VPC will be created with private subnets that have access to the internet through NAT gateways.
 * If instead the `natGateways` property is set to 0, the VPC will have only public and isolated subnets. In this case, the subnets will anyway use a cidrMask of `24`, so that changing the number of NAT gateways will not require to re-provision the VPC.
 *
 * In addition, this construct can also take care of creating a bastion host in the VPC by using the latest Amazon Linux AMI with the smallest available instance (t4g.nano), if the `bastionHostEnabled` property is set to `true`.
 */
class Networking extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.hasPrivateSubnets = props.natGateways !== 0;
        this.vpc = this.buildVpc(props);
        if (props.bastionHostEnabled) {
            this.bastionHost = new ec2.BastionHostLinux(scope, 'Bastion', {
                vpc: this.vpc,
                instanceName: props.bastionName ?? (props.vpcName ? `${props.vpcName}-bastion` : undefined),
                machineImage: props.bastionHostAmi ??
                    ec2.MachineImage.latestAmazonLinux2023({
                        cpuType: ec2.AmazonLinuxCpuType.ARM_64,
                    }),
                instanceType: props.bastionHostInstanceType ?? ec2.InstanceType.of(ec2.InstanceClass.T4G, ec2.InstanceSize.NANO),
                subnetSelection: this.privateSubnets ?? this.publicSubnets,
            });
            cdk.Tags.of(this.bastionHost.instance).add('Resource', 'Bastion');
        }
    }
    get publicSubnets() {
        return { subnetType: ec2.SubnetType.PUBLIC };
    }
    get privateSubnets() {
        if (!this.hasPrivateSubnets) {
            return undefined;
        }
        return { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS };
    }
    get isolatedSubnets() {
        return { subnetType: ec2.SubnetType.PRIVATE_ISOLATED };
    }
    buildVpc(props) {
        const subnetConfiguration = [
            {
                cidrMask: 24,
                name: 'public',
                subnetType: ec2.SubnetType.PUBLIC,
            },
            {
                cidrMask: 24,
                name: 'isolated',
                subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
            },
        ];
        if (this.hasPrivateSubnets) {
            subnetConfiguration.push({
                cidrMask: 24,
                name: 'private',
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            });
        }
        return new ec2.Vpc(this, 'VPC', {
            ipAddresses: props.ipAddresses,
            vpcName: props.vpcName,
            subnetConfiguration,
            natGateways: props.natGateways,
            maxAzs: props.maxAzs,
        });
    }
}
exports.Networking = Networking;
_a = JSII_RTTI_SYMBOL_1;
Networking[_a] = { fqn: "@condensetech/cdk-constructs.Networking", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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