import * as ec2 from 'aws-cdk-lib/aws-ec2';
import { Construct } from 'constructs';
import { INetworking } from '../interfaces';
/**
 * Properties for the Networking construct.
 */
export interface NetworkingProps {
    readonly ipAddresses: ec2.IIpAddresses;
    readonly vpcName?: string;
    readonly natGateways?: number;
    readonly bastionName?: string;
    readonly bastionHostEnabled?: boolean;
    readonly bastionHostAmi?: ec2.IMachineImage;
    readonly bastionHostInstanceType?: ec2.InstanceType;
    readonly maxAzs?: number;
}
/**
 * The Networking construct creates a VPC which can have public, private, and isolated subnets.
 * It enforces to define a CIDR block for the VPC, which is a best practice.
 *
 * If the `natGateways` property is set to a positive integer, the VPC will be created with private subnets that have access to the internet through NAT gateways.
 * If instead the `natGateways` property is set to 0, the VPC will have only public and isolated subnets. In this case, the subnets will anyway use a cidrMask of `24`, so that changing the number of NAT gateways will not require to re-provision the VPC.
 *
 * In addition, this construct can also take care of creating a bastion host in the VPC by using the latest Amazon Linux AMI with the smallest available instance (t4g.nano), if the `bastionHostEnabled` property is set to `true`.
 */
export declare class Networking extends Construct implements INetworking {
    readonly vpc: ec2.IVpc;
    readonly bastionHost?: ec2.BastionHostLinux;
    readonly hasPrivateSubnets: boolean;
    constructor(scope: Construct, id: string, props: NetworkingProps);
    get publicSubnets(): ec2.SubnetSelection;
    get privateSubnets(): ec2.SubnetSelection | undefined;
    get isolatedSubnets(): ec2.SubnetSelection;
    private buildVpc;
}
