import * as cdk from 'aws-cdk-lib';
import { aws_cloudwatch as cw, aws_elasticache as elasticache, aws_sns as sns, aws_elasticloadbalancingv2 as elbv2, aws_ecs as ecs, aws_rds as rds } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { RdsInstanceMonitoringConfig, TargetGroupMonitoringConfig, CacheClusterMonitoringConfig, FargateServiceMonitoringConfig, ApplicationLoadBalancerMonitoringConfig, RdsClusterMonitoringConfig } from './aspects';
import { ICondenseMonitoringFacade } from './interfaces';
/**
 * Properties for the MonitoringFacade.
 */
export interface MonitoringFacadeProps {
    /**
     * The name of the Cloudwatch dashboard to create.
     * @default - A name is generated by CDK.
     */
    readonly dashboardName?: string;
    /**
     * The ARN of the SNS topic to use for alarms.
     */
    readonly topicArn: string;
}
/**
 * The MonitoringFacade creates a Cloudwatch dashboard and applies monitoring aspects to resources.
 * These aspects will scan for resources, create alarms and add metrics to the MonitoringFacade dashboard.
 *
 * This allow to have a centralized monitoring configuration for all resources in the stack.
 *
 * Additionally, the `config*` methods allow to override the default configuration for a specific resource.
 *
 * @example
 * class MyStack extends cdk.Stack {
 *   constructor(scope: Construct, id: string, props: cdk.StackProps) {
 *     super(scope, id, props);
 *
 *     const cluster = new AuroraCluster(this, 'DatabaseCluster', { ... });
 *
 *     // Even if the MonitoringFacade is built after the AuroraCluster, the cluster will be monitored, because the aspects are executed after the stack is built.
 *     const monitoring = new MonitoringFacade(this, { topicArn: 'arn:aws:sns:us-east-1:123456789012:MyTopic' });
 *
 *     const cluster2 = new AuroraCluster(this, 'DatabaseCluster2', { ... });
 *     // The monitoring configuration for the second cluster is modified so that the CPU utilization alarm is triggered when the utilization is over the 10%.
 *     monitoring.configRdsCluster(cluster2, {
 *       cpuUtilizationThreshold: 0.1,
 *     });
 *   }
 * }
 */
export declare class MonitoringFacade implements ICondenseMonitoringFacade {
    static of(scope: Construct): MonitoringFacade | undefined;
    private static instances;
    readonly dashboard: cw.Dashboard;
    readonly alarmTopic: sns.ITopic;
    private readonly cacheClusterAspect;
    private readonly rdsInstanceAspect;
    private readonly targetGroupAspect;
    private readonly fargateServiceAspect;
    private readonly albAspect;
    private readonly rdsClusterAspect;
    constructor(scope: cdk.Stack, props: MonitoringFacadeProps);
    addAlarm(alarm: cw.Alarm): void;
    /**
     * Overrides the default configuration for a specific Elasticache cluster.
     * @param resource The elasticache cluster to monitor
     * @param config The configuration to apply
     */
    configCacheCluster(resource: elasticache.CfnCacheCluster, config: CacheClusterMonitoringConfig): void;
    /**
     * Overrides the default configuration for a specific ELBv2 Target Group.
     * @param resource The target group to monitor
     * @param config The configuration to apply
     */
    configTargetGroup(resource: elbv2.ApplicationTargetGroup, config: TargetGroupMonitoringConfig): void;
    /**
     * Overrides the default configuration for a specific RDS instance.
     * @param resource The RDS instance to monitor
     * @param config The configuration to apply
     */
    configRdsInstance(resource: rds.DatabaseInstance, config: RdsInstanceMonitoringConfig): void;
    /**
     * Overrides the default configuration for a specific ECS Fargate service.
     * @param resource The Fargate service to monitor
     * @param config The configuration to apply
     */
    configFargateService(resource: ecs.FargateService, config: FargateServiceMonitoringConfig): void;
    /**
     * Overrides the default configuration for a specific Application Load Balancer.
     * @param resource The ALB to monitor
     * @param config The configuration to apply
     */
    configApplicationLoadBalancer(resource: elbv2.ApplicationLoadBalancer, config: ApplicationLoadBalancerMonitoringConfig): void;
    /**
     * Overrides the default configuration for a specific RDS cluster.
     * @param resource The RDS cluster to monitor
     * @param config The configuration to apply
     */
    configRdsCluster(resource: rds.DatabaseCluster, config: RdsClusterMonitoringConfig): void;
}
