"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RdsClusterMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The RdsClusterMonitoringAspect iterates over the RDS clusters and adds monitoring widgets and alarms.
 */
class RdsClusterMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            cpuUtilizationThreshold: 90,
            maxConnectionsThreshold: 50,
            ebsByteBalanceThreshold: 10,
            ebsIoBalanceThreshold: 10,
            freeableMemoryThreshold: cdk.Size.mebibytes(100),
            readLatencyThreshold: 20,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_rds.DatabaseCluster)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for a specific RDS cluster.
     * @param node The RDS cluster to monitor.
     * @param config The configuration to apply.
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        return [
            (0, widgets_1.dashboardSectionTitle)(`RDS ${node.node.path}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'CPU Utilization',
                left: [metrics.cpuUtilization],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.cpuUtilizationThreshold }]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Connections',
                left: [metrics.maxConnections],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.maxConnectionsThreshold }]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'EBS Byte & IO Balance',
                left: [metrics.ebsByteBalance, metrics.ebsIOBalance],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([
                    { value: config.ebsByteBalanceThreshold, label: 'EBS Byte Balance' },
                    { value: config.ebsIoBalanceThreshold, label: 'EBS IO Balance' },
                ]),
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Read Latency',
                left: [metrics.readLatency],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.readLatencyThreshold }]),
            }),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.clusterIdentifier,
            alarms: [
                {
                    alarmId: 'RdsCluster-CpuUsageAlarm',
                    metric: metrics.cpuUtilization,
                    evaluationPeriods: 5,
                    threshold: config.cpuUtilizationThreshold,
                    alarmDescription: `CPU Utilization high on ${node.clusterIdentifier}`,
                },
                {
                    alarmId: 'RdsCluster-MaxConnectionsAlarm',
                    metric: metrics.maxConnections,
                    evaluationPeriods: 5,
                    threshold: config.maxConnectionsThreshold, // FIXME: This depends on the database
                    alarmDescription: `Max Connections high on ${node.clusterIdentifier}`,
                },
                {
                    alarmId: 'RdsCluster-EBSByteBalanceAlarm',
                    metric: metrics.ebsByteBalance,
                    evaluationPeriods: 3,
                    threshold: config.ebsByteBalanceThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `EBS Byte Balance too low on ${node.clusterIdentifier}`,
                },
                {
                    alarmId: 'RdsCluster-EBSIOBalanceAlarm',
                    metric: metrics.ebsIOBalance,
                    evaluationPeriods: 3,
                    threshold: config.ebsIoBalanceThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `EBS IO Balance too low on ${node.clusterIdentifier}`,
                },
                {
                    alarmId: 'RdsCluster-FreeableMemoryAlarm',
                    metric: metrics.freeableMemory,
                    evaluationPeriods: 15,
                    threshold: config.freeableMemoryThreshold?.toBytes(),
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
                    alarmDescription: `Freeable Memory low on ${node.clusterIdentifier}`,
                },
                {
                    alarmId: 'RdsCluster-ReadLatencyAlarm',
                    metric: metrics.readLatency,
                    evaluationPeriods: 5,
                    threshold: config.readLatencyThreshold,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                    alarmDescription: `Read Latency high on ${node.clusterIdentifier}`,
                },
            ],
        });
    }
    metrics(node) {
        return {
            cpuUtilization: node.metricCPUUtilization({
                period: cdk.Duration.minutes(1),
            }),
            maxConnections: node.metricDatabaseConnections({
                period: cdk.Duration.minutes(1),
            }),
            freeableMemory: node.metricFreeableMemory({
                period: cdk.Duration.minutes(1),
            }),
            readLatency: node.metric('ReadLatency', {
                period: cdk.Duration.minutes(1),
                statistic: 'p90',
            }),
            ebsIOBalance: node.metric('EBSIOBalance%', {
                period: cdk.Duration.minutes(1),
            }),
            ebsByteBalance: node.metric('EBSByteBalance%', {
                period: cdk.Duration.minutes(1),
            }),
        };
    }
}
exports.RdsClusterMonitoringAspect = RdsClusterMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
RdsClusterMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.RdsClusterMonitoringAspect", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmRzLWNsdXN0ZXItbW9uaXRvcmluZy1hc3BlY3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvY29uc3RydWN0cy9tb25pdG9yaW5nL2FzcGVjdHMvcmRzLWNsdXN0ZXItbW9uaXRvcmluZy1hc3BlY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxtQ0FBbUM7QUFDbkMsNkNBQW1FO0FBRW5FLHNDQUF3QztBQUV4Qyx3Q0FBaUg7QUFvRGpIOztHQUVHO0FBQ0gsTUFBYSwwQkFBMEI7SUFXckMsWUFBNkIsZ0JBQTJDO1FBQTNDLHFCQUFnQixHQUFoQixnQkFBZ0IsQ0FBMkI7UUFWdkQscUJBQWdCLEdBQXdELEVBQUUsQ0FBQztRQUMzRSxrQkFBYSxHQUErQjtZQUMzRCx1QkFBdUIsRUFBRSxFQUFFO1lBQzNCLHVCQUF1QixFQUFFLEVBQUU7WUFDM0IsdUJBQXVCLEVBQUUsRUFBRTtZQUMzQixxQkFBcUIsRUFBRSxFQUFFO1lBQ3pCLHVCQUF1QixFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQztZQUNoRCxvQkFBb0IsRUFBRSxFQUFFO1NBQ3pCLENBQUM7SUFFeUUsQ0FBQztJQUU1RSxLQUFLLENBQUMsSUFBZ0I7UUFDcEIsSUFBSSxDQUFDLENBQUMsSUFBSSxZQUFZLHFCQUFHLENBQUMsZUFBZSxDQUFDLEVBQUUsQ0FBQztZQUMzQyxPQUFPO1FBQ1QsQ0FBQztRQUNELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDckMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ25GLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILGNBQWMsQ0FBQyxJQUF5QixFQUFFLE1BQWtDO1FBQzFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQztJQUNqRCxDQUFDO0lBRU8sVUFBVSxDQUFDLElBQXlCO1FBQzFDLE9BQU87WUFDTCxHQUFHLElBQUksQ0FBQyxhQUFhO1lBQ3JCLEdBQUcsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7U0FDakQsQ0FBQztJQUNKLENBQUM7SUFFTyxPQUFPLENBQ2IsSUFBeUIsRUFDekIsTUFBa0MsRUFDbEMsT0FBb0M7UUFFcEMsT0FBTztZQUNMLElBQUEsK0JBQXFCLEVBQUMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQzlDLElBQUksNEJBQUUsQ0FBQyxXQUFXLENBQUM7Z0JBQ2pCLEtBQUssRUFBRSxpQkFBaUI7Z0JBQ3hCLElBQUksRUFBRSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7Z0JBQzlCLFNBQVMsRUFBRSw4QkFBb0I7Z0JBQy9CLGVBQWUsRUFBRSxJQUFBLDBCQUFnQixFQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQzthQUMvRSxDQUFDO1lBQ0YsSUFBSSw0QkFBRSxDQUFDLFdBQVcsQ0FBQztnQkFDakIsS0FBSyxFQUFFLGFBQWE7Z0JBQ3BCLElBQUksRUFBRSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7Z0JBQzlCLFNBQVMsRUFBRSw4QkFBb0I7Z0JBQy9CLGVBQWUsRUFBRSxJQUFBLDBCQUFnQixFQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQzthQUMvRSxDQUFDO1lBQ0YsSUFBSSw0QkFBRSxDQUFDLFdBQVcsQ0FBQztnQkFDakIsS0FBSyxFQUFFLHVCQUF1QjtnQkFDOUIsSUFBSSxFQUFFLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDO2dCQUNwRCxTQUFTLEVBQUUsOEJBQW9CO2dCQUMvQixlQUFlLEVBQUUsSUFBQSwwQkFBZ0IsRUFBQztvQkFDaEMsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtvQkFDcEUsRUFBRSxLQUFLLEVBQUUsTUFBTSxDQUFDLHFCQUFxQixFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtpQkFDakUsQ0FBQzthQUNILENBQUM7WUFDRixJQUFJLDRCQUFFLENBQUMsV0FBVyxDQUFDO2dCQUNqQixLQUFLLEVBQUUsY0FBYztnQkFDckIsSUFBSSxFQUFFLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQztnQkFDM0IsU0FBUyxFQUFFLDhCQUFvQjtnQkFDL0IsZUFBZSxFQUFFLElBQUEsMEJBQWdCLEVBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO2FBQzVFLENBQUM7U0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVPLE1BQU0sQ0FDWixJQUF5QixFQUN6QixNQUFrQyxFQUNsQyxPQUFvQztRQUVwQyxPQUFPLElBQUEsb0JBQVcsRUFBQztZQUNqQixJQUFJO1lBQ0osY0FBYyxFQUFFLElBQUksQ0FBQyxpQkFBaUI7WUFDdEMsTUFBTSxFQUFFO2dCQUNOO29CQUNFLE9BQU8sRUFBRSwwQkFBMEI7b0JBQ25DLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYztvQkFDOUIsaUJBQWlCLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLE1BQU0sQ0FBQyx1QkFBdUI7b0JBQ3pDLGdCQUFnQixFQUFFLDJCQUEyQixJQUFJLENBQUMsaUJBQWlCLEVBQUU7aUJBQ3RFO2dCQUNEO29CQUNFLE9BQU8sRUFBRSxnQ0FBZ0M7b0JBQ3pDLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYztvQkFDOUIsaUJBQWlCLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLE1BQU0sQ0FBQyx1QkFBdUIsRUFBRSxzQ0FBc0M7b0JBQ2pGLGdCQUFnQixFQUFFLDJCQUEyQixJQUFJLENBQUMsaUJBQWlCLEVBQUU7aUJBQ3RFO2dCQUNEO29CQUNFLE9BQU8sRUFBRSxnQ0FBZ0M7b0JBQ3pDLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYztvQkFDOUIsaUJBQWlCLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLE1BQU0sQ0FBQyx1QkFBdUI7b0JBQ3pDLGtCQUFrQixFQUFFLDRCQUFFLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO29CQUM3RCxnQkFBZ0IsRUFBRSwrQkFBK0IsSUFBSSxDQUFDLGlCQUFpQixFQUFFO2lCQUMxRTtnQkFDRDtvQkFDRSxPQUFPLEVBQUUsOEJBQThCO29CQUN2QyxNQUFNLEVBQUUsT0FBTyxDQUFDLFlBQVk7b0JBQzVCLGlCQUFpQixFQUFFLENBQUM7b0JBQ3BCLFNBQVMsRUFBRSxNQUFNLENBQUMscUJBQXFCO29CQUN2QyxrQkFBa0IsRUFBRSw0QkFBRSxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQjtvQkFDN0QsZ0JBQWdCLEVBQUUsNkJBQTZCLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtpQkFDeEU7Z0JBQ0Q7b0JBQ0UsT0FBTyxFQUFFLGdDQUFnQztvQkFDekMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxjQUFjO29CQUM5QixpQkFBaUIsRUFBRSxFQUFFO29CQUNyQixTQUFTLEVBQUUsTUFBTSxDQUFDLHVCQUF1QixFQUFFLE9BQU8sRUFBRTtvQkFDcEQsa0JBQWtCLEVBQUUsNEJBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7b0JBQzdELGdCQUFnQixFQUFFLDBCQUEwQixJQUFJLENBQUMsaUJBQWlCLEVBQUU7aUJBQ3JFO2dCQUNEO29CQUNFLE9BQU8sRUFBRSw2QkFBNkI7b0JBQ3RDLE1BQU0sRUFBRSxPQUFPLENBQUMsV0FBVztvQkFDM0IsaUJBQWlCLEVBQUUsQ0FBQztvQkFDcEIsU0FBUyxFQUFFLE1BQU0sQ0FBQyxvQkFBb0I7b0JBQ3RDLGtCQUFrQixFQUFFLDRCQUFFLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO29CQUNoRSxnQkFBZ0IsRUFBRSx3QkFBd0IsSUFBSSxDQUFDLGlCQUFpQixFQUFFO2lCQUNuRTthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLE9BQU8sQ0FBQyxJQUF5QjtRQUN2QyxPQUFPO1lBQ0wsY0FBYyxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQztnQkFDeEMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzthQUNoQyxDQUFDO1lBQ0YsY0FBYyxFQUFFLElBQUksQ0FBQyx5QkFBeUIsQ0FBQztnQkFDN0MsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzthQUNoQyxDQUFDO1lBQ0YsY0FBYyxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQztnQkFDeEMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzthQUNoQyxDQUFDO1lBQ0YsV0FBVyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFO2dCQUN0QyxNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2dCQUMvQixTQUFTLEVBQUUsS0FBSzthQUNqQixDQUFDO1lBQ0YsWUFBWSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsZUFBZSxFQUFFO2dCQUN6QyxNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2FBQ2hDLENBQUM7WUFDRixjQUFjLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRTtnQkFDN0MsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzthQUNoQyxDQUFDO1NBQ0gsQ0FBQztJQUNKLENBQUM7O0FBN0pILGdFQThKQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBhd3NfY2xvdWR3YXRjaCBhcyBjdywgYXdzX3JkcyBhcyByZHMgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBidWlsZEFsYXJtcyB9IGZyb20gJy4uL2FsYXJtcyc7XG5pbXBvcnQgeyBJQ29uZGVuc2VNb25pdG9yaW5nRmFjYWRlIH0gZnJvbSAnLi4vaW50ZXJmYWNlcyc7XG5pbXBvcnQgeyBhbGVydEFubm90YXRpb25zLCBkYXNoYm9hcmRHZW5lcmljQXhpcywgZGFzaGJvYXJkUGVyY2VudEF4aXMsIGRhc2hib2FyZFNlY3Rpb25UaXRsZSB9IGZyb20gJy4uL3dpZGdldHMnO1xuXG4vKipcbiAqIFRoZSBSZHNDbHVzdGVyTW9uaXRvcmluZ0NvbmZpZyBkZWZpbmVzIHRoZSB0aHJlc2hvbGRzIGZvciB0aGUgUkRTIGNsdXN0ZXIgbW9uaXRvcmluZy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZHNDbHVzdGVyTW9uaXRvcmluZ0NvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgQ1BVIFV0aWxpemF0aW9uICglKSB0aHJlc2hvbGQuXG4gICAqIEBkZWZhdWx0IDkwXG4gICAqL1xuICByZWFkb25seSBjcHVVdGlsaXphdGlvblRocmVzaG9sZD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIE1heCBDb25uZWN0aW9ucyB0aHJlc2hvbGQuXG4gICAqIEBkZWZhdWx0IDUwXG4gICAqL1xuICByZWFkb25seSBtYXhDb25uZWN0aW9uc1RocmVzaG9sZD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEVCUyBCeXRlIEJhbGFuY2UgKCUpIHRocmVzaG9sZC5cbiAgICogQGRlZmF1bHQgMTBcbiAgICovXG4gIHJlYWRvbmx5IGVic0J5dGVCYWxhbmNlVGhyZXNob2xkPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgRUJTIElPIEJhbGFuY2UgKCUpIHRocmVzaG9sZC5cbiAgICogQGRlZmF1bHQgMTBcbiAgICovXG4gIHJlYWRvbmx5IGVic0lvQmFsYW5jZVRocmVzaG9sZD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEZyZWVhYmxlIE1lbW9yeSB0aHJlc2hvbGQuXG4gICAqIEBkZWZhdWx0IDEwMCBNaUJcbiAgICovXG4gIHJlYWRvbmx5IGZyZWVhYmxlTWVtb3J5VGhyZXNob2xkPzogY2RrLlNpemU7XG5cbiAgLyoqXG4gICAqIFRoZSBSZWFkIExhdGVuY3kgdGhyZXNob2xkLlxuICAgKiBAZGVmYXVsdCAyMFxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZExhdGVuY3lUaHJlc2hvbGQ/OiBudW1iZXI7XG59XG5cbmludGVyZmFjZSBSZHNDbHVzdGVyTW9uaXRvcmluZ01ldHJpY3Mge1xuICByZWFkb25seSBjcHVVdGlsaXphdGlvbjogY3cuSU1ldHJpYztcbiAgcmVhZG9ubHkgbWF4Q29ubmVjdGlvbnM6IGN3LklNZXRyaWM7XG4gIHJlYWRvbmx5IGZyZWVhYmxlTWVtb3J5OiBjdy5JTWV0cmljO1xuICByZWFkb25seSByZWFkTGF0ZW5jeTogY3cuSU1ldHJpYztcbiAgcmVhZG9ubHkgZWJzSU9CYWxhbmNlOiBjdy5JTWV0cmljO1xuICByZWFkb25seSBlYnNCeXRlQmFsYW5jZTogY3cuSU1ldHJpYztcbn1cblxuLyoqXG4gKiBUaGUgUmRzQ2x1c3Rlck1vbml0b3JpbmdBc3BlY3QgaXRlcmF0ZXMgb3ZlciB0aGUgUkRTIGNsdXN0ZXJzIGFuZCBhZGRzIG1vbml0b3Jpbmcgd2lkZ2V0cyBhbmQgYWxhcm1zLlxuICovXG5leHBvcnQgY2xhc3MgUmRzQ2x1c3Rlck1vbml0b3JpbmdBc3BlY3QgaW1wbGVtZW50cyBjZGsuSUFzcGVjdCB7XG4gIHByaXZhdGUgcmVhZG9ubHkgb3ZlcnJpZGRlbkNvbmZpZzogUmVjb3JkPHN0cmluZywgUGFydGlhbDxSZHNDbHVzdGVyTW9uaXRvcmluZ0NvbmZpZz4+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgZGVmYXVsdENvbmZpZzogUmRzQ2x1c3Rlck1vbml0b3JpbmdDb25maWcgPSB7XG4gICAgY3B1VXRpbGl6YXRpb25UaHJlc2hvbGQ6IDkwLFxuICAgIG1heENvbm5lY3Rpb25zVGhyZXNob2xkOiA1MCxcbiAgICBlYnNCeXRlQmFsYW5jZVRocmVzaG9sZDogMTAsXG4gICAgZWJzSW9CYWxhbmNlVGhyZXNob2xkOiAxMCxcbiAgICBmcmVlYWJsZU1lbW9yeVRocmVzaG9sZDogY2RrLlNpemUubWViaWJ5dGVzKDEwMCksXG4gICAgcmVhZExhdGVuY3lUaHJlc2hvbGQ6IDIwLFxuICB9O1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgbW9uaXRvcmluZ0ZhY2FkZTogSUNvbmRlbnNlTW9uaXRvcmluZ0ZhY2FkZSkge31cblxuICB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgaWYgKCEobm9kZSBpbnN0YW5jZW9mIHJkcy5EYXRhYmFzZUNsdXN0ZXIpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IGNvbmZpZyA9IHRoaXMucmVhZENvbmZpZyhub2RlKTtcbiAgICBjb25zdCBtZXRyaWNzID0gdGhpcy5tZXRyaWNzKG5vZGUpO1xuICAgIHRoaXMubW9uaXRvcmluZ0ZhY2FkZS5kYXNoYm9hcmQuYWRkV2lkZ2V0cyguLi50aGlzLndpZGdldHMobm9kZSwgY29uZmlnLCBtZXRyaWNzKSk7XG4gICAgdGhpcy5hbGFybXMobm9kZSwgY29uZmlnLCBtZXRyaWNzKS5mb3JFYWNoKChhKSA9PiB0aGlzLm1vbml0b3JpbmdGYWNhZGUuYWRkQWxhcm0oYSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIE92ZXJyaWRlcyB0aGUgZGVmYXVsdCBjb25maWd1cmF0aW9uIGZvciBhIHNwZWNpZmljIFJEUyBjbHVzdGVyLlxuICAgKiBAcGFyYW0gbm9kZSBUaGUgUkRTIGNsdXN0ZXIgdG8gbW9uaXRvci5cbiAgICogQHBhcmFtIGNvbmZpZyBUaGUgY29uZmlndXJhdGlvbiB0byBhcHBseS5cbiAgICovXG4gIG92ZXJyaWRlQ29uZmlnKG5vZGU6IHJkcy5EYXRhYmFzZUNsdXN0ZXIsIGNvbmZpZzogUmRzQ2x1c3Rlck1vbml0b3JpbmdDb25maWcpIHtcbiAgICB0aGlzLm92ZXJyaWRkZW5Db25maWdbbm9kZS5ub2RlLnBhdGhdID0gY29uZmlnO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkQ29uZmlnKG5vZGU6IHJkcy5EYXRhYmFzZUNsdXN0ZXIpOiBSZHNDbHVzdGVyTW9uaXRvcmluZ0NvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLnRoaXMuZGVmYXVsdENvbmZpZyxcbiAgICAgIC4uLih0aGlzLm92ZXJyaWRkZW5Db25maWdbbm9kZS5ub2RlLnBhdGhdIHx8IHt9KSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSB3aWRnZXRzKFxuICAgIG5vZGU6IHJkcy5EYXRhYmFzZUNsdXN0ZXIsXG4gICAgY29uZmlnOiBSZHNDbHVzdGVyTW9uaXRvcmluZ0NvbmZpZyxcbiAgICBtZXRyaWNzOiBSZHNDbHVzdGVyTW9uaXRvcmluZ01ldHJpY3MsXG4gICk6IGN3LklXaWRnZXRbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIGRhc2hib2FyZFNlY3Rpb25UaXRsZShgUkRTICR7bm9kZS5ub2RlLnBhdGh9YCksXG4gICAgICBuZXcgY3cuR3JhcGhXaWRnZXQoe1xuICAgICAgICB0aXRsZTogJ0NQVSBVdGlsaXphdGlvbicsXG4gICAgICAgIGxlZnQ6IFttZXRyaWNzLmNwdVV0aWxpemF0aW9uXSxcbiAgICAgICAgbGVmdFlBeGlzOiBkYXNoYm9hcmRQZXJjZW50QXhpcyxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBhbGVydEFubm90YXRpb25zKFt7IHZhbHVlOiBjb25maWcuY3B1VXRpbGl6YXRpb25UaHJlc2hvbGQgfV0pLFxuICAgICAgfSksXG4gICAgICBuZXcgY3cuR3JhcGhXaWRnZXQoe1xuICAgICAgICB0aXRsZTogJ0Nvbm5lY3Rpb25zJyxcbiAgICAgICAgbGVmdDogW21ldHJpY3MubWF4Q29ubmVjdGlvbnNdLFxuICAgICAgICBsZWZ0WUF4aXM6IGRhc2hib2FyZEdlbmVyaWNBeGlzLFxuICAgICAgICBsZWZ0QW5ub3RhdGlvbnM6IGFsZXJ0QW5ub3RhdGlvbnMoW3sgdmFsdWU6IGNvbmZpZy5tYXhDb25uZWN0aW9uc1RocmVzaG9sZCB9XSksXG4gICAgICB9KSxcbiAgICAgIG5ldyBjdy5HcmFwaFdpZGdldCh7XG4gICAgICAgIHRpdGxlOiAnRUJTIEJ5dGUgJiBJTyBCYWxhbmNlJyxcbiAgICAgICAgbGVmdDogW21ldHJpY3MuZWJzQnl0ZUJhbGFuY2UsIG1ldHJpY3MuZWJzSU9CYWxhbmNlXSxcbiAgICAgICAgbGVmdFlBeGlzOiBkYXNoYm9hcmRQZXJjZW50QXhpcyxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBhbGVydEFubm90YXRpb25zKFtcbiAgICAgICAgICB7IHZhbHVlOiBjb25maWcuZWJzQnl0ZUJhbGFuY2VUaHJlc2hvbGQsIGxhYmVsOiAnRUJTIEJ5dGUgQmFsYW5jZScgfSxcbiAgICAgICAgICB7IHZhbHVlOiBjb25maWcuZWJzSW9CYWxhbmNlVGhyZXNob2xkLCBsYWJlbDogJ0VCUyBJTyBCYWxhbmNlJyB9LFxuICAgICAgICBdKSxcbiAgICAgIH0pLFxuICAgICAgbmV3IGN3LkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6ICdSZWFkIExhdGVuY3knLFxuICAgICAgICBsZWZ0OiBbbWV0cmljcy5yZWFkTGF0ZW5jeV0sXG4gICAgICAgIGxlZnRZQXhpczogZGFzaGJvYXJkR2VuZXJpY0F4aXMsXG4gICAgICAgIGxlZnRBbm5vdGF0aW9uczogYWxlcnRBbm5vdGF0aW9ucyhbeyB2YWx1ZTogY29uZmlnLnJlYWRMYXRlbmN5VGhyZXNob2xkIH1dKSxcbiAgICAgIH0pLFxuICAgIF07XG4gIH1cblxuICBwcml2YXRlIGFsYXJtcyhcbiAgICBub2RlOiByZHMuRGF0YWJhc2VDbHVzdGVyLFxuICAgIGNvbmZpZzogUmRzQ2x1c3Rlck1vbml0b3JpbmdDb25maWcsXG4gICAgbWV0cmljczogUmRzQ2x1c3Rlck1vbml0b3JpbmdNZXRyaWNzLFxuICApOiBjdy5BbGFybVtdIHtcbiAgICByZXR1cm4gYnVpbGRBbGFybXMoe1xuICAgICAgbm9kZSxcbiAgICAgIG5vZGVJZGVudGlmaWVyOiBub2RlLmNsdXN0ZXJJZGVudGlmaWVyLFxuICAgICAgYWxhcm1zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhbGFybUlkOiAnUmRzQ2x1c3Rlci1DcHVVc2FnZUFsYXJtJyxcbiAgICAgICAgICBtZXRyaWM6IG1ldHJpY3MuY3B1VXRpbGl6YXRpb24sXG4gICAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDUsXG4gICAgICAgICAgdGhyZXNob2xkOiBjb25maWcuY3B1VXRpbGl6YXRpb25UaHJlc2hvbGQsXG4gICAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogYENQVSBVdGlsaXphdGlvbiBoaWdoIG9uICR7bm9kZS5jbHVzdGVySWRlbnRpZmllcn1gLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgYWxhcm1JZDogJ1Jkc0NsdXN0ZXItTWF4Q29ubmVjdGlvbnNBbGFybScsXG4gICAgICAgICAgbWV0cmljOiBtZXRyaWNzLm1heENvbm5lY3Rpb25zLFxuICAgICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiA1LFxuICAgICAgICAgIHRocmVzaG9sZDogY29uZmlnLm1heENvbm5lY3Rpb25zVGhyZXNob2xkLCAvLyBGSVhNRTogVGhpcyBkZXBlbmRzIG9uIHRoZSBkYXRhYmFzZVxuICAgICAgICAgIGFsYXJtRGVzY3JpcHRpb246IGBNYXggQ29ubmVjdGlvbnMgaGlnaCBvbiAke25vZGUuY2x1c3RlcklkZW50aWZpZXJ9YCxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIGFsYXJtSWQ6ICdSZHNDbHVzdGVyLUVCU0J5dGVCYWxhbmNlQWxhcm0nLFxuICAgICAgICAgIG1ldHJpYzogbWV0cmljcy5lYnNCeXRlQmFsYW5jZSxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMyxcbiAgICAgICAgICB0aHJlc2hvbGQ6IGNvbmZpZy5lYnNCeXRlQmFsYW5jZVRocmVzaG9sZCxcbiAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGN3LkNvbXBhcmlzb25PcGVyYXRvci5MRVNTX1RIQU5fVEhSRVNIT0xELFxuICAgICAgICAgIGFsYXJtRGVzY3JpcHRpb246IGBFQlMgQnl0ZSBCYWxhbmNlIHRvbyBsb3cgb24gJHtub2RlLmNsdXN0ZXJJZGVudGlmaWVyfWAsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBhbGFybUlkOiAnUmRzQ2x1c3Rlci1FQlNJT0JhbGFuY2VBbGFybScsXG4gICAgICAgICAgbWV0cmljOiBtZXRyaWNzLmVic0lPQmFsYW5jZSxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMyxcbiAgICAgICAgICB0aHJlc2hvbGQ6IGNvbmZpZy5lYnNJb0JhbGFuY2VUaHJlc2hvbGQsXG4gICAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjdy5Db21wYXJpc29uT3BlcmF0b3IuTEVTU19USEFOX1RIUkVTSE9MRCxcbiAgICAgICAgICBhbGFybURlc2NyaXB0aW9uOiBgRUJTIElPIEJhbGFuY2UgdG9vIGxvdyBvbiAke25vZGUuY2x1c3RlcklkZW50aWZpZXJ9YCxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIGFsYXJtSWQ6ICdSZHNDbHVzdGVyLUZyZWVhYmxlTWVtb3J5QWxhcm0nLFxuICAgICAgICAgIG1ldHJpYzogbWV0cmljcy5mcmVlYWJsZU1lbW9yeSxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMTUsXG4gICAgICAgICAgdGhyZXNob2xkOiBjb25maWcuZnJlZWFibGVNZW1vcnlUaHJlc2hvbGQ/LnRvQnl0ZXMoKSxcbiAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGN3LkNvbXBhcmlzb25PcGVyYXRvci5MRVNTX1RIQU5fVEhSRVNIT0xELFxuICAgICAgICAgIGFsYXJtRGVzY3JpcHRpb246IGBGcmVlYWJsZSBNZW1vcnkgbG93IG9uICR7bm9kZS5jbHVzdGVySWRlbnRpZmllcn1gLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgYWxhcm1JZDogJ1Jkc0NsdXN0ZXItUmVhZExhdGVuY3lBbGFybScsXG4gICAgICAgICAgbWV0cmljOiBtZXRyaWNzLnJlYWRMYXRlbmN5LFxuICAgICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiA1LFxuICAgICAgICAgIHRocmVzaG9sZDogY29uZmlnLnJlYWRMYXRlbmN5VGhyZXNob2xkLFxuICAgICAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY3cuQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9USFJFU0hPTEQsXG4gICAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogYFJlYWQgTGF0ZW5jeSBoaWdoIG9uICR7bm9kZS5jbHVzdGVySWRlbnRpZmllcn1gLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgbWV0cmljcyhub2RlOiByZHMuRGF0YWJhc2VDbHVzdGVyKTogUmRzQ2x1c3Rlck1vbml0b3JpbmdNZXRyaWNzIHtcbiAgICByZXR1cm4ge1xuICAgICAgY3B1VXRpbGl6YXRpb246IG5vZGUubWV0cmljQ1BVVXRpbGl6YXRpb24oe1xuICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgfSksXG4gICAgICBtYXhDb25uZWN0aW9uczogbm9kZS5tZXRyaWNEYXRhYmFzZUNvbm5lY3Rpb25zKHtcbiAgICAgICAgcGVyaW9kOiBjZGsuRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIH0pLFxuICAgICAgZnJlZWFibGVNZW1vcnk6IG5vZGUubWV0cmljRnJlZWFibGVNZW1vcnkoe1xuICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgfSksXG4gICAgICByZWFkTGF0ZW5jeTogbm9kZS5tZXRyaWMoJ1JlYWRMYXRlbmN5Jywge1xuICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgICBzdGF0aXN0aWM6ICdwOTAnLFxuICAgICAgfSksXG4gICAgICBlYnNJT0JhbGFuY2U6IG5vZGUubWV0cmljKCdFQlNJT0JhbGFuY2UlJywge1xuICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgfSksXG4gICAgICBlYnNCeXRlQmFsYW5jZTogbm9kZS5tZXRyaWMoJ0VCU0J5dGVCYWxhbmNlJScsIHtcbiAgICAgICAgcGVyaW9kOiBjZGsuRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIH0pLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==