"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheClusterMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The CacheClusterMonitoringAspect iterates over the Elasticache clusters and adds monitoring widgets and alarms.
 */
class CacheClusterMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            cpuUtilizationThreshold: 90,
            maxConnectionsThreshold: 60000,
            memoryUsageThreshold: 90,
            engineCpuUtilizationThreshold: 95,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_elasticache.CfnCacheCluster)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for a specific Elasticache cluster.
     * @param node The elasticache cluster to monitor
     * @param config The configuration to apply
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        const cpuWidth = Math.max(Math.round(24 / metrics.cpuUtilization.length), 3);
        const connectionsWidth = Math.max(Math.round(24 / metrics.maxConnections.length), 3);
        return [
            (0, widgets_1.dashboardSectionTitle)(`Elasticache ${node.node.path}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Memory Usage',
                left: [metrics.memoryUsage],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.memoryUsageThreshold }]),
                width: 8,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'CPU Utilization',
                left: [metrics.engineCpuUtilization],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.engineCpuUtilizationThreshold }]),
                width: 8,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Replication Lag',
                left: [metrics.replicationLag],
                leftYAxis: widgets_1.dashboardMillisecondsAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.replicationLagThreshold?.toMilliseconds() }]),
                width: 8,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.TextWidget({
                markdown: '## Node Metrics',
                width: 24,
            }),
            ...metrics.cpuUtilization.map((metric, i) => new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: `CPU Utilization Node ${i}`,
                left: [metric],
                leftYAxis: widgets_1.dashboardPercentAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.cpuUtilizationThreshold }]),
                width: cpuWidth,
            })),
            ...metrics.maxConnections.map((metric, i) => new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: `Connections Node ${i}`,
                left: [metric],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.maxConnectionsThreshold }]),
                width: connectionsWidth,
            })),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.ref,
            alarms: [
                ...metrics.cpuUtilization.map((metric, i) => ({
                    alarmId: `CacheCluster-CpuUsageAlarm-${i}`,
                    alarmName: `CacheCluster-CpuUsageAlarm-${node.ref}-${i + 1}`,
                    metric,
                    evaluationPeriods: 5,
                    threshold: config.cpuUtilizationThreshold,
                    alarmDescription: `CPU Utilization high on ${node.ref}, node ${i + 1}`,
                })),
                ...metrics.maxConnections.map((metric, i) => ({
                    alarmId: `CacheCluster-MaxConnectionsAlarm-${i}`,
                    alarmName: `CacheCluster-MaxConnectionsAlarm-${node.ref}-${i + 1}`,
                    metric,
                    evaluationPeriods: 10,
                    threshold: config.maxConnectionsThreshold,
                    alarmDescription: `Max Connections high on ${node.ref}, node ${i + 1}`,
                })),
                {
                    alarmId: 'CacheCluster-MemoryUsageAlarm',
                    metric: metrics.memoryUsage,
                    evaluationPeriods: 5,
                    threshold: config.memoryUsageThreshold,
                    alarmDescription: `Memory Usage high on ${node.ref}`,
                },
                {
                    alarmId: 'CacheCluster-EngineCpuUsageAlarm',
                    metric: metrics.engineCpuUtilization,
                    evaluationPeriods: 5,
                    threshold: config.engineCpuUtilizationThreshold,
                    alarmDescription: `Engine CPU Utilization high on ${node.ref}`,
                },
                {
                    alarmId: 'CacheCluster-ReplicationLagAlarm',
                    metric: metrics.replicationLag,
                    evaluationPeriods: 15,
                    threshold: config.replicationLagThreshold?.toMilliseconds(),
                    alarmDescription: `Replication Lag high on ${node.ref}`,
                },
            ],
        });
    }
    metrics(node) {
        const cacheNodeIds = this.getCacheNodeIds(node);
        const cpuUtilization = cacheNodeIds.map((cacheNodeId) => new aws_cdk_lib_1.aws_cloudwatch.Metric({
            metricName: 'CPUUtilization',
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                CacheClusterId: node.ref,
                CacheNodeId: cacheNodeId,
            },
            period: cdk.Duration.minutes(1),
        }));
        const maxConnections = cacheNodeIds.map((cacheNodeId) => new aws_cdk_lib_1.aws_cloudwatch.Metric({
            metricName: 'CurrConnections',
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                CacheClusterId: node.ref,
                CacheNodeId: cacheNodeId,
            },
            period: cdk.Duration.minutes(1),
        }));
        const memoryUsage = new aws_cdk_lib_1.aws_cloudwatch.Metric({
            metricName: 'DatabaseMemoryUsagePercentage',
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                CacheClusterId: node.ref,
            },
            period: cdk.Duration.minutes(1),
        });
        const engineCpuUtilization = new aws_cdk_lib_1.aws_cloudwatch.Metric({
            metricName: 'EngineCPUUtilization',
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                CacheClusterId: node.ref,
            },
            period: cdk.Duration.minutes(1),
        });
        const replicationLag = new aws_cdk_lib_1.aws_cloudwatch.Metric({
            metricName: 'ReplicationLag',
            namespace: 'AWS/ElastiCache',
            dimensionsMap: {
                CacheClusterId: node.ref,
            },
            period: cdk.Duration.minutes(1),
        });
        return {
            cpuUtilization,
            maxConnections,
            memoryUsage,
            engineCpuUtilization,
            replicationLag,
        };
    }
    getCacheNodeId(idx) {
        return (idx + 1).toString().padStart(4, '0');
    }
    getCacheNodeIds(node) {
        const cacheNodeIds = [];
        for (let i = 0; i < node.numCacheNodes; i++) {
            cacheNodeIds.push(this.getCacheNodeId(i));
        }
        return cacheNodeIds;
    }
}
exports.CacheClusterMonitoringAspect = CacheClusterMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
CacheClusterMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.CacheClusterMonitoringAspect", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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