"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancerMonitoringAspect = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alarms_1 = require("../alarms");
const widgets_1 = require("../widgets");
/**
 * The ApplicationLoadBalancerMonitoringAspect iterates over the Application Load Balancers and adds monitoring widgets and alarms.
 */
class ApplicationLoadBalancerMonitoringAspect {
    constructor(monitoringFacade) {
        this.monitoringFacade = monitoringFacade;
        this.overriddenConfig = {};
        this.defaultConfig = {
            redirectUrlLimitExceededThreshold: 0,
            rejectedConnectionsThreshold: 0,
            targetConnectionErrorsThreshold: 0,
        };
    }
    visit(node) {
        if (!(node instanceof aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer)) {
            return;
        }
        const config = this.readConfig(node);
        const metrics = this.metrics(node);
        this.monitoringFacade.dashboard.addWidgets(...this.widgets(node, config, metrics));
        this.alarms(node, config, metrics).forEach((a) => this.monitoringFacade.addAlarm(a));
    }
    /**
     * Overrides the default configuration for a specific Application Load Balancer.
     * @param node The Application Load Balancer to monitor.
     * @param config The configuration to apply.
     */
    overrideConfig(node, config) {
        this.overriddenConfig[node.node.path] = config;
    }
    readConfig(node) {
        return {
            ...this.defaultConfig,
            ...(this.overriddenConfig[node.node.path] || {}),
        };
    }
    widgets(node, config, metrics) {
        return [
            (0, widgets_1.dashboardSectionTitle)(`Load Balancer ${node.loadBalancerName}`),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Response Time',
                left: [metrics.responseTime],
                leftYAxis: widgets_1.dashboardSecondsAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([
                    {
                        value: config.responseTimeThreshold?.toSeconds({ integral: false }),
                    },
                ]),
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Redirect URL Limit Exceeded',
                left: [metrics.redirectUrlLimitExceeded],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.redirectUrlLimitExceededThreshold }]),
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Connections',
                left: [metrics.activeConnections, metrics.rejectedConnections],
                stacked: true,
                leftYAxis: widgets_1.dashboardGenericAxis,
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: 'Target Connection Errors',
                left: [metrics.targetConnectionErrors],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.targetConnectionErrorsThreshold }]),
                width: 12,
            }),
            new aws_cdk_lib_1.aws_cloudwatch.GraphWidget({
                title: '5xx Errors',
                left: [metrics.target5xxErrors],
                leftYAxis: widgets_1.dashboardGenericAxis,
                leftAnnotations: (0, widgets_1.alertAnnotations)([{ value: config.target5xxErrorsThreshold }]),
                width: 12,
            }),
        ];
    }
    alarms(node, config, metrics) {
        return (0, alarms_1.buildAlarms)({
            node,
            nodeIdentifier: node.loadBalancerName,
            alarms: [
                {
                    alarmId: 'LoadBalancer-ResponseTimeAlarm',
                    metric: metrics.responseTime,
                    evaluationPeriods: 5,
                    threshold: config.responseTimeThreshold?.toSeconds({ integral: false }),
                    alarmDescription: `Response time is too high on ${node.loadBalancerName}`,
                },
                {
                    alarmId: 'LoadBalancer-RedirectUrlLimitExceeded',
                    metric: metrics.redirectUrlLimitExceeded,
                    evaluationPeriods: 5,
                    threshold: config.redirectUrlLimitExceededThreshold,
                    alarmDescription: `Some redirect actions couldn't be completed on ${node.loadBalancerName}`,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                },
                {
                    alarmId: 'LoadBalancer-RejectedConnectionsAlarm',
                    metric: metrics.rejectedConnections,
                    evaluationPeriods: 5,
                    threshold: config.rejectedConnectionsThreshold,
                    alarmDescription: `Rejected connections on ${node.loadBalancerName}`,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                },
                {
                    alarmId: 'LoadBalancer-TargetConnectionErrorsAlarm',
                    metric: metrics.targetConnectionErrors,
                    evaluationPeriods: 5,
                    threshold: config.targetConnectionErrorsThreshold,
                    alarmDescription: `Target connection errors on ${node.loadBalancerName}`,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                },
                {
                    alarmId: 'LoadBalancer-Target5xxErrorsAlarm',
                    metric: metrics.target5xxErrors,
                    evaluationPeriods: 5,
                    threshold: config.target5xxErrorsThreshold,
                    alarmDescription: `5xx errors on ${node.loadBalancerName}`,
                    comparisonOperator: aws_cdk_lib_1.aws_cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                },
            ],
        });
    }
    metrics(node) {
        return {
            responseTime: node.metrics.targetResponseTime({
                period: cdk.Duration.minutes(1),
            }),
            redirectUrlLimitExceeded: node.metrics.httpRedirectUrlLimitExceededCount({
                period: cdk.Duration.minutes(1),
            }),
            rejectedConnections: node.metrics.rejectedConnectionCount({
                period: cdk.Duration.minutes(1),
            }),
            activeConnections: node.metrics.activeConnectionCount({
                period: cdk.Duration.minutes(1),
            }),
            targetConnectionErrors: node.metrics.targetConnectionErrorCount({
                period: cdk.Duration.minutes(1),
            }),
            target5xxErrors: node.metrics.httpCodeTarget(aws_cdk_lib_1.aws_elasticloadbalancingv2.HttpCodeTarget.TARGET_5XX_COUNT, {
                period: cdk.Duration.minutes(1),
            }),
        };
    }
}
exports.ApplicationLoadBalancerMonitoringAspect = ApplicationLoadBalancerMonitoringAspect;
_a = JSII_RTTI_SYMBOL_1;
ApplicationLoadBalancerMonitoringAspect[_a] = { fqn: "@condensetech/cdk-constructs.ApplicationLoadBalancerMonitoringAspect", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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