"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Entrypoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_route53_targets_1 = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const application_listener_priority_allocator_1 = require("./application-listener-priority-allocator");
class EntrypointBase extends constructs_1.Construct {
    referenceListener(scope, id) {
        return aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListener.fromApplicationListenerAttributes(scope, id, {
            listenerArn: this.listener.listenerArn,
            securityGroup: aws_cdk_lib_1.aws_ec2.SecurityGroup.fromSecurityGroupId(scope, `${id}-SG`, this.securityGroup.securityGroupId),
        });
    }
    allocateListenerRule(scope, id, props) {
        const listener = this.referenceListener(scope, `${id}-Listener`);
        const priority = this.priorityAllocator.allocatePriority(scope, `${id}-Priority`, {
            priority: props.priority,
        });
        return new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListenerRule(scope, id, {
            ...props,
            listener,
            priority,
        });
    }
}
/**
 * The Entrypoint construct creates an Application Load Balancer (ALB) that serves as the centralized entry point for all applications.
 * This ALB is shared across multiple applications, primarily to optimize infrastructure costs by reducing the need for multiple load balancers.
 * It implements the IEntrypoint interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It creates an ALB with:
 * - an HTTP listener that redirects all traffic to HTTPS.
 * - an HTTPS listener that returns a 403 Forbidden response by default.
 * - a custom security group. This allows to expose the security group as a property of the entrypoint construct, making it easier to reference it in other constructs.
 * Finally, it creates the Route 53 A and AAAA record that point to the ALB.
 *
 * When hostedZoneProps is provided, by default this construct creates an HTTPS certificate, bound to the domain name and all subdomains (unless wildcardCertificate is set to false).
 * You can also provide an existing certificate ARN through certificate.certificateArn.
 *
 * When an `entrypointName` is provided, this is used as the name of the ALB and as the prefix for the security group.
 * It is also used to add an additional "Name" tag to the load balancer.
 * This helps to use [ApplicationLoadBalancer#lookup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_elasticloadbalancingv2.ApplicationLoadBalancer.html#static-fromwbrlookupscope-id-options) to find the load balancer by name.
 */
class Entrypoint extends EntrypointBase {
    static fromAttributes(scope, id, props) {
        class Import extends EntrypointBase {
            constructor() {
                super(scope, id);
                this.domainName = props.domainName;
                this.securityGroup = aws_cdk_lib_1.aws_ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId);
                this.alb = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(this, 'Alb', {
                    loadBalancerArn: props.loadBalancerArn,
                    securityGroupId: props.securityGroupId,
                });
                this.listener = aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationListener.fromApplicationListenerAttributes(this, 'Listener', {
                    listenerArn: props.listenerArn,
                    securityGroup: this.securityGroup,
                });
                this.priorityAllocator = application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator.fromServiceToken(this, 'PriorityAllocator', props.priorityAllocatorServiceToken);
            }
        }
        return new Import();
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.hostedZone = props.hostedZoneProps
            ? aws_cdk_lib_1.aws_route53.HostedZone.fromHostedZoneAttributes(this, 'HostedZone', props.hostedZoneProps)
            : undefined;
        this.domainName = props.domainName;
        const albCertificates = (props.certificates || [props.certificate ?? {}]).map(this.createCertificate);
        this.securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.networking.vpc,
            allowAllOutbound: true,
            securityGroupName: props.securityGroupName ??
                props.entrypointSecurityGroupName ??
                (props.entrypointName ? `${props.entrypointName}-sg` : undefined),
        });
        const alb = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, 'Alb', {
            vpc: props.networking.vpc,
            internetFacing: true,
            securityGroup: this.securityGroup,
            loadBalancerName: props.entrypointName,
        });
        if (props.entrypointName) {
            cdk.Tags.of(alb).add('Name', props.entrypointName);
        }
        if (props.logsBucket) {
            alb.logAccessLogs(props.logsBucket);
        }
        this.alb = alb;
        this.alb.addListener('HTTP', {
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.redirect({
                port: '443',
                protocol: 'HTTPS',
            }),
        });
        this.listener = this.alb.addListener('HTTPS', {
            protocol: aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTPS,
            certificates: albCertificates,
            defaultAction: aws_cdk_lib_1.aws_elasticloadbalancingv2.ListenerAction.fixedResponse(403, {
                messageBody: 'Forbidden',
            }),
        });
        if (this.hostedZone) {
            new aws_cdk_lib_1.aws_route53.ARecord(this, 'AlbRecord', {
                target: aws_cdk_lib_1.aws_route53.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(this.alb)),
                zone: this.hostedZone,
                recordName: this.domainName.replace(`.${this.hostedZone.zoneName}`, ''),
            });
            new aws_cdk_lib_1.aws_route53.AaaaRecord(this, 'AlbRecordIPv6', {
                target: aws_cdk_lib_1.aws_route53.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(this.alb)),
                zone: this.hostedZone,
                recordName: this.domainName.replace(`.${this.hostedZone.zoneName}`, ''),
            });
        }
        this.priorityAllocator = new application_listener_priority_allocator_1.ApplicationListenerPriorityAllocator(this, 'PriorityAllocator', {
            listener: this.listener,
            ...props.priorityAllocator,
        });
    }
    createCertificate(props) {
        if (props.certificate) {
            if (props.certificateArn) {
                cdk.Annotations.of(this).addError(`Both certificate (${props.certificate}) and certificateArn (${props.certificateArn}) are provided. Choose one.`);
            }
            if (props.wildcardCertificate) {
                cdk.Annotations.of(this).addError(`wildcardCertificate cannot be set when a certificate (${props.certificate}) is provided.`);
            }
            return props.certificate;
        }
        if (props.certificateArn) {
            if (props.wildcardCertificate) {
                cdk.Annotations.of(this).addError(`wildcardCertificate cannot be set when a certificateArn (${props.certificateArn}) is provided.`);
            }
            return aws_cdk_lib_1.aws_certificatemanager.Certificate.fromCertificateArn(this, 'Certificate', props.certificateArn);
        }
        if (this.hostedZone) {
            const subjectAlternativeNames = props.wildcardCertificate === false ? undefined : [`*.${this.hostedZone.zoneName}`];
            return new aws_cdk_lib_1.aws_certificatemanager.Certificate(this, 'Certificate', {
                domainName: this.domainName,
                subjectAlternativeNames,
                validation: aws_cdk_lib_1.aws_certificatemanager.CertificateValidation.fromDns(this.hostedZone),
            });
        }
        throw new Error('Hosted Zone Props are required when certificate must be automatically provisioned.');
    }
}
exports.Entrypoint = Entrypoint;
_a = JSII_RTTI_SYMBOL_1;
Entrypoint[_a] = { fqn: "@condensetech/cdk-constructs.Entrypoint", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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