"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The DatabaseInstance construct creates an RDS database instance.
 * Under the hood, it creates a [rds.DatabaseInstance](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#starting-an-instance-database) construct.
 * It implements the IDatabase interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It also applies the following changes to the default behavior:
 * - A [rds.ParameterGroup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#parameter-groups) specific for the cluster is always defined.
 *   By using a custom parameter group instead of relying on the default one, a later change in the parameter group's parameters wouldn't require a replace of the cluster.
 * - The credentials secret name is created after the construct's path. This way, the secret name is more readable and, when working with multiple stacks, can be easily inferred without having to rely on Cloudformation exports.
 * - It defaults the storage type to GP3 when not specified.
 * - It defaults the allocated storage to the minimum storage of 20 GB when not specified.
 * - The default instance type is set to t3.small.
 * - The storage is always encrypted.
 * - If the networking configuration includes a bastion host, the database allows connections from the bastion host.
 * - The default security group name is `${construct.node.path}-sg`. This allows for easier lookups when working with multiple stacks.
 */
class DatabaseInstance extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const removalPolicy = props.removalPolicy ?? cdk.RemovalPolicy.RETAIN;
        const parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, 'ParameterGroup', {
            engine: props.engine,
            description: this.node.path,
            removalPolicy: [cdk.RemovalPolicy.DESTROY, cdk.RemovalPolicy.RETAIN].includes(removalPolicy)
                ? removalPolicy
                : cdk.RemovalPolicy.DESTROY,
        });
        const instanceType = props.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE3, aws_cdk_lib_1.aws_ec2.InstanceSize.SMALL);
        const credentials = aws_cdk_lib_1.aws_rds.Credentials.fromUsername(props.credentialsUsername ?? 'db_user', {
            secretName: props.credentialsSecretName ?? `${this.node.path}/secret`,
        });
        const securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.networking.vpc,
            allowAllOutbound: true,
            securityGroupName: props.securityGroupName ?? `${this.node.path.replace(/\//g, '-')}-sg`,
        });
        this.resource = new aws_cdk_lib_1.aws_rds.DatabaseInstance(this, 'DB', {
            instanceIdentifier: props.instanceIdentifier,
            vpc: props.networking.vpc,
            vpcSubnets: props.networking.isolatedSubnets,
            engine: props.engine,
            databaseName: props.databaseName,
            credentials,
            parameterGroup: parameterGroup,
            instanceType,
            allocatedStorage: props.allocatedStorage ?? 20,
            storageType: props.storageType ?? aws_cdk_lib_1.aws_rds.StorageType.GP3,
            multiAz: props.multiAz ?? false,
            securityGroups: [securityGroup],
            storageEncrypted: true,
            backupRetention: props.backupRetention,
            removalPolicy,
        });
        if (props.networking.bastionHost) {
            this.resource.connections.allowDefaultPortFrom(props.networking.bastionHost);
        }
        this.endpoint = this.resource.instanceEndpoint;
    }
    get connections() {
        return this.resource.connections;
    }
    fetchSecret(scope, id = 'DatabaseSecret') {
        return aws_cdk_lib_1.aws_secretsmanager.Secret.fromSecretNameV2(scope, id, `${this.node.path}/secret`);
    }
}
exports.DatabaseInstance = DatabaseInstance;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_a] = { fqn: "@condensetech/cdk-constructs.DatabaseInstance", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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