import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_secretsmanager as sm, aws_rds as rds } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IDatabase, INetworking } from '../interfaces';
/**
 * Properties for the DatabaseInstance construct.
 */
export interface DatabaseInstanceProps {
    /**
     * The networking configuration for the database instance.
     */
    readonly networking: INetworking;
    /**
     * The engine of the database instance.
     */
    readonly engine: rds.IInstanceEngine;
    /**
     * The identifier of the database instance.
     * @default - No identifier is specified.
     */
    readonly instanceIdentifier?: string;
    /**
     * The name of the security group.
     * @default - `${construct.node.path}-sg`.
     */
    readonly securityGroupName?: string;
    /**
     * The name of the database.
     * @default - No default database is created.
     */
    readonly databaseName?: string;
    /**
     * The name of the secret that stores the credentials of the database.
     * @default `${construct.node.path}/secret`
     */
    readonly credentialsSecretName?: string;
    /**
     * The username of the database.
     * @default db_user
     */
    readonly credentialsUsername?: string;
    /**
     * The instance type of the database instance.
     * @default - db.t3.small.
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The storage type of the database instance.
     * @default rds.StorageType.GP3
     */
    readonly storageType?: rds.StorageType;
    /**
     * The allocated storage of the database instance.
     * @default 20
     */
    readonly allocatedStorage?: number;
    /**
     * If the database instance is multi-AZ.
     * @default false
     */
    readonly multiAz?: boolean;
    /**
     * The backup retention period.
     * @default - It uses the default applied by [rds.DatabaseInstanceProps#backupRetention]https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds.DatabaseInstanceProps.html#backupretention).
     */
    readonly backupRetention?: cdk.Duration;
    /**
     * The removal policy to apply when the cluster is removed.
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: cdk.RemovalPolicy;
}
/**
 * The DatabaseInstance construct creates an RDS database instance.
 * Under the hood, it creates a [rds.DatabaseInstance](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#starting-an-instance-database) construct.
 * It implements the IDatabase interface so that it can be used in other constructs and stacks without requiring to access to the underlying construct.
 *
 * It also applies the following changes to the default behavior:
 * - A [rds.ParameterGroup](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_rds-readme.html#parameter-groups) specific for the cluster is always defined.
 *   By using a custom parameter group instead of relying on the default one, a later change in the parameter group's parameters wouldn't require a replace of the cluster.
 * - The credentials secret name is created after the construct's path. This way, the secret name is more readable and, when working with multiple stacks, can be easily inferred without having to rely on Cloudformation exports.
 * - It defaults the storage type to GP3 when not specified.
 * - It defaults the allocated storage to the minimum storage of 20 GB when not specified.
 * - The default instance type is set to t3.small.
 * - The storage is always encrypted.
 * - If the networking configuration includes a bastion host, the database allows connections from the bastion host.
 * - The default security group name is `${construct.node.path}-sg`. This allows for easier lookups when working with multiple stacks.
 */
export declare class DatabaseInstance extends Construct implements IDatabase {
    /**
     * The underlying RDS database instance.
     */
    readonly resource: rds.IDatabaseInstance;
    readonly endpoint: rds.Endpoint;
    constructor(scope: Construct, id: string, props: DatabaseInstanceProps);
    get connections(): ec2.Connections;
    fetchSecret(scope: Construct, id?: string): sm.ISecret;
}
