/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.filter.SamRecordFilter;
import htsjdk.samtools.filter.SecondaryAlignmentFilter;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.util.AbstractLocusInfo;
import htsjdk.samtools.util.AbstractLocusIterator;
import htsjdk.samtools.util.AbstractRecordAndOffset;
import htsjdk.samtools.util.EdgeReadIterator;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityUtil;
import htsjdk.samtools.util.SamLocusIterator;
import htsjdk.samtools.util.SequenceUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.ArgumentCollection;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.analysis.AbstractWgsMetricsCollector;
import picard.analysis.FastWgsMetricsCollector;
import picard.analysis.MergeableMetricBase;
import picard.analysis.TheoreticalSensitivity;
import picard.analysis.TheoreticalSensitivityMetrics;
import picard.analysis.WgsMetricsProcessorImpl;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.argumentcollections.IntervalArgumentCollection;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.filter.CountingDuplicateFilter;
import picard.filter.CountingFilter;
import picard.filter.CountingMapQFilter;
import picard.filter.CountingPairedFilter;
import picard.util.MathUtil;

@CommandLineProgramProperties(summary="Collect metrics about coverage and performance of whole genome sequencing (WGS) experiments.<p>This tool collects metrics about the fractions of reads that pass base- and mapping-quality filters as well as coverage (read-depth) levels for WGS analyses. Both minimum base- and mapping-quality values as well as the maximum read depths (coverage cap) are user defined.</p><p>Note: Metrics labeled as percentages are actually expressed as fractions!</p><h4>Usage Example:</h4><pre>java -jar picard.jar CollectWgsMetrics \\<br />       I=input.bam \\<br />       O=collect_wgs_metrics.txt \\<br />       R=reference_sequence.fasta </pre>Please see <a href='https://broadinstitute.github.io/picard/picard-metric-definitions.html#CollectWgsMetrics.WgsMetrics'>CollectWgsMetrics</a> for detailed explanations of the output metrics.<hr />", oneLineSummary="Collect metrics about coverage and performance of whole genome sequencing (WGS) experiments.", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CollectWgsMetrics
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Collect metrics about coverage and performance of whole genome sequencing (WGS) experiments.";
    static final String USAGE_DETAILS = "<p>This tool collects metrics about the fractions of reads that pass base- and mapping-quality filters as well as coverage (read-depth) levels for WGS analyses. Both minimum base- and mapping-quality values as well as the maximum read depths (coverage cap) are user defined.</p><p>Note: Metrics labeled as percentages are actually expressed as fractions!</p><h4>Usage Example:</h4><pre>java -jar picard.jar CollectWgsMetrics \\<br />       I=input.bam \\<br />       O=collect_wgs_metrics.txt \\<br />       R=reference_sequence.fasta </pre>Please see <a href='https://broadinstitute.github.io/picard/picard-metric-definitions.html#CollectWgsMetrics.WgsMetrics'>CollectWgsMetrics</a> for detailed explanations of the output metrics.<hr />";
    @Argument(shortName="I", doc="Input SAM or BAM file.")
    public File INPUT;
    @Argument(shortName="O", doc="Output metrics file.")
    public File OUTPUT;
    @Argument(shortName="MQ", doc="Minimum mapping quality for a read to contribute coverage.")
    public int MINIMUM_MAPPING_QUALITY = 20;
    @Argument(shortName="Q", doc="Minimum base quality for a base to contribute coverage. N bases will be treated as having a base quality of negative infinity and will therefore be excluded from coverage regardless of the value of this parameter.")
    public int MINIMUM_BASE_QUALITY = 20;
    @Argument(shortName="CAP", doc="Treat positions with coverage exceeding this value as if they had coverage at this value (but calculate the difference for PCT_EXC_CAPPED).")
    public int COVERAGE_CAP = 250;
    @Argument(doc="At positions with coverage exceeding this value, completely ignore reads that accumulate beyond this value (so that they will not be considered for PCT_EXC_CAPPED).  Used to keep memory consumption in check, but could create bias if set too low")
    public int LOCUS_ACCUMULATION_CAP = 100000;
    @Argument(doc="For debugging purposes, stop after processing this many genomic bases.")
    public long STOP_AFTER = -1L;
    @Argument(doc="Determines whether to include the base quality histogram in the metrics file.")
    public boolean INCLUDE_BQ_HISTOGRAM = false;
    @Argument(doc="If true, count unpaired reads, and paired reads with one end unmapped")
    public boolean COUNT_UNPAIRED = false;
    @Argument(doc="Sample Size used for Theoretical Het Sensitivity sampling. Default is 10000.", optional=true)
    public int SAMPLE_SIZE = 10000;
    @ArgumentCollection
    protected IntervalArgumentCollection intervalArugmentCollection = this.makeIntervalArgumentCollection();
    @Argument(doc="Output for Theoretical Sensitivity metrics.", optional=true)
    public File THEORETICAL_SENSITIVITY_OUTPUT;
    @Argument(doc="Allele fraction for which to calculate theoretical sensitivity.", optional=true)
    public List<Double> ALLELE_FRACTION = new ArrayList<Double>(Arrays.asList(0.001, 0.005, 0.01, 0.02, 0.05, 0.1, 0.2, 0.3, 0.5));
    @Argument(doc="If true, fast algorithm is used.")
    public boolean USE_FAST_ALGORITHM = false;
    @Argument(doc="Average read length in the file. Default is 150.", optional=true)
    public int READ_LENGTH = 150;
    protected File INTERVALS = null;
    private SAMFileHeader header = null;
    private final Log log = Log.getInstance(CollectWgsMetrics.class);
    private static final double LOG_ODDS_THRESHOLD = 3.0;

    @Override
    protected boolean requiresReference() {
        return true;
    }

    protected IntervalArgumentCollection makeIntervalArgumentCollection() {
        return new CollectWgsMetricsIntervalArgumentCollection();
    }

    protected SamReader getSamReader() {
        SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        this.header = in.getFileHeader();
        return in;
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.INPUT);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        IOUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        this.INTERVALS = this.intervalArugmentCollection.getIntervalFile();
        if (this.INTERVALS != null) {
            IOUtil.assertFileIsReadable(this.INTERVALS);
        }
        if (this.THEORETICAL_SENSITIVITY_OUTPUT != null) {
            IOUtil.assertFileIsWritable(this.THEORETICAL_SENSITIVITY_OUTPUT);
        }
        if (this.LOCUS_ACCUMULATION_CAP < this.COVERAGE_CAP) {
            this.log.warn("Setting the LOCUS_ACCUMULATION_CAP to be equal to the COVERAGE_CAP (" + this.COVERAGE_CAP + ") because it should not be lower");
            this.LOCUS_ACCUMULATION_CAP = this.COVERAGE_CAP;
        }
        ProgressLogger progress = new ProgressLogger(this.log, 10000000, "Processed", "loci");
        ReferenceSequenceFileWalker refWalker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);
        SamReader in = this.getSamReader();
        AbstractLocusIterator iterator = this.getLocusIterator(in);
        ArrayList<SamRecordFilter> filters = new ArrayList<SamRecordFilter>();
        CountingMapQFilter mapqFilter = new CountingMapQFilter(this.MINIMUM_MAPPING_QUALITY);
        CountingDuplicateFilter dupeFilter = new CountingDuplicateFilter();
        CountingPairedFilter pairFilter = new CountingPairedFilter();
        filters.add(new SecondaryAlignmentFilter());
        filters.add(mapqFilter);
        filters.add(dupeFilter);
        if (!this.COUNT_UNPAIRED) {
            filters.add(pairFilter);
        }
        iterator.setSamFilters(filters);
        iterator.setMappingQualityScoreCutoff(0);
        iterator.setIncludeNonPfReads(false);
        AbstractWgsMetricsCollector collector = this.getCollector(this.COVERAGE_CAP, this.getIntervalsToExamine());
        WgsMetricsProcessorImpl processor = this.getWgsMetricsProcessor(progress, refWalker, iterator, collector);
        processor.processFile();
        MetricsFile<WgsMetrics, Integer> out = this.getMetricsFile();
        processor.addToMetricsFile(out, this.INCLUDE_BQ_HISTOGRAM, dupeFilter, mapqFilter, pairFilter);
        out.write(this.OUTPUT);
        if (this.THEORETICAL_SENSITIVITY_OUTPUT != null) {
            MetricsFile theoreticalSensitivityMetrics = this.getMetricsFile();
            this.log.info("Calculating theoretical sentitivity at " + this.ALLELE_FRACTION.size() + " allele fractions.");
            List<TheoreticalSensitivityMetrics> tsm = TheoreticalSensitivity.calculateSensitivities(this.SAMPLE_SIZE, collector.getUnfilteredDepthHistogram(), collector.getUnfilteredBaseQHistogram(), this.ALLELE_FRACTION);
            theoreticalSensitivityMetrics.addAllMetrics(tsm);
            theoreticalSensitivityMetrics.write(this.THEORETICAL_SENSITIVITY_OUTPUT);
        }
        return 0;
    }

    private <T extends AbstractRecordAndOffset> WgsMetricsProcessorImpl<T> getWgsMetricsProcessor(ProgressLogger progress, ReferenceSequenceFileWalker refWalker, AbstractLocusIterator<T, AbstractLocusInfo<T>> iterator, AbstractWgsMetricsCollector<T> collector) {
        return new WgsMetricsProcessorImpl<T>(iterator, refWalker, collector, progress);
    }

    protected IntervalList getIntervalsToExamine() {
        IntervalList intervals;
        if (this.INTERVALS != null) {
            IOUtil.assertFileIsReadable(this.INTERVALS);
            intervals = IntervalList.fromFile(this.INTERVALS);
        } else {
            intervals = new IntervalList(this.header);
            for (SAMSequenceRecord rec : this.header.getSequenceDictionary().getSequences()) {
                Interval interval = new Interval(rec.getSequenceName(), 1, rec.getSequenceLength());
                intervals.add(interval);
            }
        }
        return intervals;
    }

    protected SAMFileHeader getSamFileHeader() {
        if (this.header == null) {
            throw new IllegalStateException("getSamFileHeader() was called but this.header is null");
        }
        return this.header;
    }

    protected WgsMetrics generateWgsMetrics(IntervalList intervals, Histogram<Integer> highQualityDepthHistogram, Histogram<Integer> unfilteredDepthHistogram, double pctExcludedByMapq, double pctExcludedByDupes, double pctExcludedByPairing, double pctExcludedByBaseq, double pctExcludedByOverlap, double pctExcludedByCapping, double pctTotal, int coverageCap, Histogram<Integer> unfilteredBaseQHistogram, int theoreticalHetSensitivitySampleSize) {
        return new WgsMetrics(intervals, highQualityDepthHistogram, unfilteredDepthHistogram, pctExcludedByMapq, pctExcludedByDupes, pctExcludedByPairing, pctExcludedByBaseq, pctExcludedByOverlap, pctExcludedByCapping, pctTotal, coverageCap, unfilteredBaseQHistogram, theoreticalHetSensitivitySampleSize);
    }

    WgsMetrics generateWgsMetrics(IntervalList intervals, Histogram<Integer> highQualityDepthHistogram, Histogram<Integer> unfilteredDepthHistogram, long basesExcludedByMapq, long basesExcludedByDupes, long basesExcludedByPairing, long basesExcludedByBaseq, long basesExcludedByOverlap, long basesExcludedByCapping, int coverageCap, Histogram<Integer> unfilteredBaseQHistogram, int theoreticalHetSensitivitySampleSize) {
        double total = highQualityDepthHistogram.getSum();
        double totalWithExcludes = total + (double)basesExcludedByDupes + (double)basesExcludedByMapq + (double)basesExcludedByPairing + (double)basesExcludedByBaseq + (double)basesExcludedByOverlap + (double)basesExcludedByCapping;
        double pctExcludedByMapq = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByMapq / totalWithExcludes;
        double pctExcludedByDupes = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByDupes / totalWithExcludes;
        double pctExcludedByPairing = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByPairing / totalWithExcludes;
        double pctExcludedByBaseq = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByBaseq / totalWithExcludes;
        double pctExcludedByOverlap = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByOverlap / totalWithExcludes;
        double pctExcludedByCapping = totalWithExcludes == 0.0 ? 0.0 : (double)basesExcludedByCapping / totalWithExcludes;
        double pctTotal = totalWithExcludes == 0.0 ? 0.0 : (totalWithExcludes - total) / totalWithExcludes;
        return this.generateWgsMetrics(intervals, highQualityDepthHistogram, unfilteredDepthHistogram, pctExcludedByMapq, pctExcludedByDupes, pctExcludedByPairing, pctExcludedByBaseq, pctExcludedByOverlap, pctExcludedByCapping, pctTotal, coverageCap, unfilteredBaseQHistogram, theoreticalHetSensitivitySampleSize);
    }

    protected long getBasesExcludedBy(CountingFilter filter) {
        return filter.getFilteredBases();
    }

    protected AbstractLocusIterator getLocusIterator(SamReader in) {
        if (this.USE_FAST_ALGORITHM) {
            return this.INTERVALS != null ? new EdgeReadIterator(in, IntervalList.fromFile(this.INTERVALS)) : new EdgeReadIterator(in);
        }
        SamLocusIterator iterator = this.INTERVALS != null ? new SamLocusIterator(in, IntervalList.fromFile(this.INTERVALS)) : new SamLocusIterator(in);
        iterator.setMaxReadsToAccumulatePerLocus(this.LOCUS_ACCUMULATION_CAP);
        iterator.setEmitUncoveredLoci(true);
        iterator.setQualityScoreCutoff(0);
        return iterator;
    }

    protected AbstractWgsMetricsCollector getCollector(int coverageCap, IntervalList intervals) {
        return this.USE_FAST_ALGORITHM ? new FastWgsMetricsCollector(this, coverageCap, intervals) : new WgsMetricsCollector(this, coverageCap, intervals);
    }

    protected static class WgsMetricsCollector
    extends AbstractWgsMetricsCollector<SamLocusIterator.RecordAndOffset> {
        public WgsMetricsCollector(CollectWgsMetrics metrics, int coverageCap, IntervalList intervals) {
            super(metrics, coverageCap, intervals);
        }

        @Override
        public void addInfo(AbstractLocusInfo<SamLocusIterator.RecordAndOffset> info, ReferenceSequence ref, boolean referenceBaseN) {
            if (referenceBaseN) {
                return;
            }
            HashSet<String> readNames = new HashSet<String>(info.getRecordAndOffsets().size());
            int pileupSize = 0;
            int unfilteredDepth = 0;
            for (SamLocusIterator.RecordAndOffset recs : info.getRecordAndOffsets()) {
                if (recs.getBaseQuality() <= 2) {
                    ++this.basesExcludedByBaseq;
                    continue;
                }
                if (unfilteredDepth < this.coverageCap) {
                    byte by = recs.getRecord().getBaseQualities()[recs.getOffset()];
                    this.unfilteredBaseQHistogramArray[by] = this.unfilteredBaseQHistogramArray[by] + 1L;
                    ++unfilteredDepth;
                }
                if (recs.getBaseQuality() < this.collectWgsMetrics.MINIMUM_BASE_QUALITY || SequenceUtil.isNoCall(recs.getReadBase())) {
                    ++this.basesExcludedByBaseq;
                    continue;
                }
                if (!readNames.add(recs.getRecord().getReadName())) {
                    ++this.basesExcludedByOverlap;
                    continue;
                }
                ++pileupSize;
            }
            int highQualityDepth = Math.min(pileupSize, this.coverageCap);
            if (highQualityDepth < pileupSize) {
                this.basesExcludedByCapping += (long)(pileupSize - this.coverageCap);
            }
            int n = highQualityDepth;
            this.highQualityDepthHistogramArray[n] = this.highQualityDepthHistogramArray[n] + 1L;
            int n2 = unfilteredDepth;
            this.unfilteredDepthHistogramArray[n2] = this.unfilteredDepthHistogramArray[n2] + 1L;
        }
    }

    public static class WgsMetrics
    extends MergeableMetricBase {
        @MergeableMetricBase.MergingIsManual
        protected IntervalList intervals;
        @MergeableMetricBase.MergingIsManual
        protected final Histogram<Integer> highQualityDepthHistogram;
        @MergeableMetricBase.MergingIsManual
        protected final Histogram<Integer> unfilteredDepthHistogram;
        @MergeableMetricBase.MergingIsManual
        protected final Histogram<Integer> unfilteredBaseQHistogram;
        @MergeableMetricBase.MergeByAssertEquals
        protected final int coverageCap;
        @MergeableMetricBase.NoMergingKeepsValue
        protected final int theoreticalHetSensitivitySampleSize;
        @MergeableMetricBase.NoMergingIsDerived
        public long GENOME_TERRITORY;
        @MergeableMetricBase.NoMergingIsDerived
        public double MEAN_COVERAGE;
        @MergeableMetricBase.NoMergingIsDerived
        public double SD_COVERAGE;
        @MergeableMetricBase.NoMergingIsDerived
        public double MEDIAN_COVERAGE;
        @MergeableMetricBase.NoMergingIsDerived
        public double MAD_COVERAGE;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_MAPQ;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_DUPE;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_UNPAIRED;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_BASEQ;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_OVERLAP;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_CAPPED;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_EXC_TOTAL;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_1X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_5X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_10X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_15X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_20X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_25X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_30X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_40X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_50X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_60X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_70X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_80X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_90X;
        @MergeableMetricBase.NoMergingIsDerived
        public double PCT_100X;
        @MergeableMetricBase.NoMergingIsDerived
        public double HET_SNP_SENSITIVITY;
        @MergeableMetricBase.NoMergingIsDerived
        public double HET_SNP_Q;

        public WgsMetrics() {
            this.intervals = null;
            this.highQualityDepthHistogram = null;
            this.unfilteredDepthHistogram = null;
            this.unfilteredBaseQHistogram = null;
            this.theoreticalHetSensitivitySampleSize = -1;
            this.coverageCap = -1;
        }

        public WgsMetrics(IntervalList intervals, Histogram<Integer> highQualityDepthHistogram, Histogram<Integer> unfilteredDepthHistogram, double pctExcludedByMapq, double pctExcludedByDupes, double pctExcludedByPairing, double pctExcludedByBaseq, double pctExcludedByOverlap, double pctExcludedByCapping, double pctExcludeTotal, int coverageCap, Histogram<Integer> unfilteredBaseQHistogram, int theoreticalHetSensitivitySampleSize) {
            this.intervals = intervals.uniqued();
            this.highQualityDepthHistogram = highQualityDepthHistogram;
            this.unfilteredDepthHistogram = unfilteredDepthHistogram;
            this.unfilteredBaseQHistogram = unfilteredBaseQHistogram;
            this.coverageCap = coverageCap;
            this.theoreticalHetSensitivitySampleSize = theoreticalHetSensitivitySampleSize;
            this.PCT_EXC_MAPQ = pctExcludedByMapq;
            this.PCT_EXC_DUPE = pctExcludedByDupes;
            this.PCT_EXC_UNPAIRED = pctExcludedByPairing;
            this.PCT_EXC_BASEQ = pctExcludedByBaseq;
            this.PCT_EXC_OVERLAP = pctExcludedByOverlap;
            this.PCT_EXC_CAPPED = pctExcludedByCapping;
            this.PCT_EXC_TOTAL = pctExcludeTotal;
            this.calculateDerivedFields();
        }

        @Override
        public MergeableMetricBase merge(MergeableMetricBase other) {
            WgsMetrics otherMetric = (WgsMetrics)other;
            if (this.highQualityDepthHistogram == null || otherMetric.highQualityDepthHistogram == null || this.unfilteredDepthHistogram == null || otherMetric.unfilteredDepthHistogram == null) {
                throw new PicardException("Depth histogram is required when deriving metrics.");
            }
            long genomeTerritory = this.intervals.getBaseCount() + otherMetric.intervals.getBaseCount();
            this.intervals.addall(otherMetric.intervals.getIntervals());
            this.intervals = this.intervals.uniqued();
            if (this.intervals.getBaseCount() != genomeTerritory) {
                throw new PicardException("Trying to merge WgsMetrics calculated on intervals that overlap.");
            }
            long thisMetricTotal = (long)this.highQualityDepthHistogram.getSum();
            long otherMetricTotal = (long)otherMetric.highQualityDepthHistogram.getSum();
            long total = thisMetricTotal + otherMetricTotal;
            long thisTotalWithExcludes = (long)((double)thisMetricTotal / (1.0 - this.PCT_EXC_TOTAL));
            long otherTotalWithExcludes = (long)((double)otherMetricTotal / (1.0 - otherMetric.PCT_EXC_TOTAL));
            double totalWithExcludes = thisTotalWithExcludes + otherTotalWithExcludes;
            if (0.0 < totalWithExcludes) {
                this.PCT_EXC_DUPE = (this.PCT_EXC_DUPE * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_DUPE * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_MAPQ = (this.PCT_EXC_MAPQ * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_MAPQ * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_UNPAIRED = (this.PCT_EXC_UNPAIRED * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_UNPAIRED * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_BASEQ = (this.PCT_EXC_BASEQ * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_BASEQ * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_OVERLAP = (this.PCT_EXC_OVERLAP * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_OVERLAP * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_CAPPED = (this.PCT_EXC_CAPPED * (double)thisTotalWithExcludes + otherMetric.PCT_EXC_CAPPED * (double)otherTotalWithExcludes) / totalWithExcludes;
                this.PCT_EXC_TOTAL = (totalWithExcludes - (double)total) / totalWithExcludes;
            }
            super.merge(other);
            this.highQualityDepthHistogram.addHistogram(otherMetric.highQualityDepthHistogram);
            this.unfilteredDepthHistogram.addHistogram(otherMetric.unfilteredDepthHistogram);
            if (this.unfilteredBaseQHistogram != null && otherMetric.unfilteredBaseQHistogram != null) {
                this.unfilteredBaseQHistogram.addHistogram(otherMetric.unfilteredBaseQHistogram);
            }
            return this;
        }

        @Override
        public void calculateDerivedFields() {
            if (this.highQualityDepthHistogram == null || this.unfilteredDepthHistogram == null) {
                throw new PicardException("Depth histogram is required when deriving metrics.");
            }
            if (this.unfilteredBaseQHistogram != null && this.theoreticalHetSensitivitySampleSize <= 0) {
                throw new PicardException("Sample size is required when a baseQ histogram is given when deriving metrics.");
            }
            long[] depthHistogramArray = new long[this.coverageCap + 1];
            for (Histogram.Bin<Integer> bin : this.highQualityDepthHistogram.values()) {
                int depth;
                int n = depth = Math.min((int)bin.getIdValue(), this.coverageCap);
                depthHistogramArray[n] = (long)((double)depthHistogramArray[n] + bin.getValue());
            }
            this.GENOME_TERRITORY = (long)this.highQualityDepthHistogram.getSumOfValues();
            this.MEAN_COVERAGE = this.highQualityDepthHistogram.getMean();
            this.SD_COVERAGE = this.highQualityDepthHistogram.getStandardDeviation();
            this.MEDIAN_COVERAGE = this.highQualityDepthHistogram.getMedian();
            this.MAD_COVERAGE = this.highQualityDepthHistogram.getMedianAbsoluteDeviation();
            this.PCT_1X = (double)MathUtil.sum(depthHistogramArray, 1, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_5X = (double)MathUtil.sum(depthHistogramArray, 5, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_10X = (double)MathUtil.sum(depthHistogramArray, 10, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_15X = (double)MathUtil.sum(depthHistogramArray, 15, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_20X = (double)MathUtil.sum(depthHistogramArray, 20, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_25X = (double)MathUtil.sum(depthHistogramArray, 25, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_30X = (double)MathUtil.sum(depthHistogramArray, 30, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_40X = (double)MathUtil.sum(depthHistogramArray, 40, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_50X = (double)MathUtil.sum(depthHistogramArray, 50, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_60X = (double)MathUtil.sum(depthHistogramArray, 60, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_70X = (double)MathUtil.sum(depthHistogramArray, 70, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_80X = (double)MathUtil.sum(depthHistogramArray, 80, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_90X = (double)MathUtil.sum(depthHistogramArray, 90, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            this.PCT_100X = (double)MathUtil.sum(depthHistogramArray, 100, depthHistogramArray.length) / (double)this.GENOME_TERRITORY;
            if (this.unfilteredBaseQHistogram != null && this.unfilteredDepthHistogram != null) {
                double[] depthDoubleArray = TheoreticalSensitivity.normalizeHistogram(this.unfilteredDepthHistogram);
                double[] baseQDoubleArray = TheoreticalSensitivity.normalizeHistogram(this.unfilteredBaseQHistogram);
                this.HET_SNP_SENSITIVITY = TheoreticalSensitivity.hetSNPSensitivity(depthDoubleArray, baseQDoubleArray, this.theoreticalHetSensitivitySampleSize, 3.0);
                this.HET_SNP_Q = QualityUtil.getPhredScoreFromErrorProbability(1.0 - this.HET_SNP_SENSITIVITY);
            }
        }
    }

    public static class CollectWgsMetricsIntervalArgumentCollection
    implements IntervalArgumentCollection {
        @Argument(doc="An interval list file that contains the positions to restrict the assessment. Please note that all bases of reads that overlap these intervals will be considered, even if some of those bases extend beyond the boundaries of the interval. The ideal use case for this argument is to use it to restrict the calculation to a subset of (whole) contigs.", optional=true)
        public File INTERVALS;

        @Override
        public File getIntervalFile() {
            return this.INTERVALS;
        }
    }
}

