# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['protobuf_to_pydantic', 'protobuf_to_pydantic.contrib']

package_data = \
{'': ['*']}

install_requires = \
['grpcio-tools>=1.47.0,<2.0.0', 'pydantic>=1.7.3,<2.0.0']

setup_kwargs = {
    'name': 'protobuf-to-pydantic',
    'version': '0.0.1',
    'description': 'Convert Protobuf-generated Python objects to Pydantic.BaseModel objects with parameter checksum',
    'long_description': '# protobuf_to_pydantic\nConvert Protobuf-generated Python objects to Pydantic.BaseModel objects with parameter checksum\n\n> NOTE: Only support proto3\n\n# 1.installation\n```bash\npip install protobuf_to_pydantic\n```\n\n# 2.Quick Start\n## 2.1gen runtime obj\n`protobuf_to_pydantic` can generate the corresponding `pydantic.BaseModel` object at runtime based on the `Message` object.\n\nFor example, the `User Message` in the following Protobuf file:\n```protobuf\nsyntax = "proto3";\npackage user;\n\nenum SexType {\n  man = 0;\n  women = 1;\n}\n\nmessage UserMessage {\n  string uid=1;\n  int32 age=2;\n  float height=3;\n  SexType sex=4;\n  bool is_adult=5;\n  string user_name=6;\n}\n```\n`protobuf to pydantic` can read the corresponding `Python` object at runtime and convert it to the corresponding `pydantic.Base Model`:\n```Python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(demo_pb2.UserMessage)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n\n# output\n# {\n#   \'uid\': FieldInfo(default=\'\', extra={}), \n#   \'age\': FieldInfo(default=0, extra={}), \n#   \'height\': FieldInfo(default=0.0, extra={}), \n#   \'sex\': FieldInfo(default=0, extra={}),\n#   \'is_adult\': FieldInfo(default=False, extra={}), \n#   \'user_name\': FieldInfo(default=\'\', extra={})\n#  }\n```\n## 2.2.validate\nThe message generated according to protobuf carries very little information, and there is not enough information to make the generated `pydantic.Base Model` with parameter verification function.\n\nAt present, `protobuf to pydantic` obtains other information of Message in three ways, so that the generated `pydantic.Base Model` object has the function of parameter verification.\n\n\n### 2.2.1.Text Comment\n使用者可以在protobuf文件中为每个字段编写符合`protobuf_to_pydantic`要求的注释来为`protobuf_to_pydantic`提供参数校验信息，比如下面这个例子\n```protobuf\nsyntax = "proto3";\npackage user;\n\nenum SexType {\n  man = 0;\n  women = 1;\n}\n\n// user info\nmessage UserMessage {\n  // p2p: {"miss_default": true, "example": "10086"}\n  // p2p: {"title": "UID"}\n  string uid=1; // p2p: {"description": "user union id"}\n  // p2p: {"example": 18, "title": "use age", "ge": 0}\n  int32 age=2;\n  // p2p: {"ge": 0, "le": 2.5}\n  float height=3;\n  SexType sex=4;\n  bool is_adult=5;\n  // p2p: {"description": "user name"}\n  // p2p: {"default": "", "min_length": 1, "max_length": "10", "example": "so1n"}\n  string user_name=6;\n}\n```\n在这个例子中，每个可以被`protobuf_to_pydantic`使用的注释都是以`p2p:`开头，并在后面跟着一个完整的Json字符串，这个字符串里面的都是对应字段的校验信息。\n比如`UserMessage`中的uid附带了如下4个信息：\n- miss_default: 表示生成的字段不带有默认值\n- example: 表示生成的字段的示例值为10086\n- title: 表示字段的名称为UID\n- description: 表示字段的文档描述为`user union id`\n\n> Note: 目前只支持单行注释且一个完整的Json数据不能换行。\n\n当拥有这些信息后，`protobuf_to_pydantic`在把Message生成对应的`Pydantic.BaseModel`对象时都会为每个字段带上对应的信息，如下:\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(demo_pb2.UserMessage, parse_msg_desc_method=demo_pb2)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'uid\': FieldInfo(default=PydanticUndefined, title=\'UID\', description=\'user union id\', extra={\'example\': \'10086\'}), \n#   \'age\': FieldInfo(default=0, title=\'use age\', ge=0, extra={\'example\': 18}), \n#   \'height\': FieldInfo(default=0.0, ge=0, le=2, extra={}), \n#   \'sex\': FieldInfo(default=0, extra={}), \n#   \'is_adult\': FieldInfo(default=False, extra={}), \n#   \'user_name\': FieldInfo(default=\'\', description=\'user name\', min_length=1, max_length=10, extra={\'example\': \'so1n\'})\n# }\n\n```\n\n### 2.2.1.By pyi file\n由于`Python`中负责把protobuf文件专为`Python`代码时并不会把Message的注释带到生成的`Python`代码中，所以在上面的示例通过把Message对象所属的模块传入`parse_msg_desc_method`中，\n使得`protobuf_to_pydantic`可以通过读取Message对应的pyi文件的注释来获取Message对象的附加信息。\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(demo_pb2.UserMessage, parse_msg_desc_method=demo_pb2)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'uid\': FieldInfo(default=PydanticUndefined, title=\'UID\', description=\'user union id\', extra={\'example\': \'10086\'}), \n#   \'age\': FieldInfo(default=0, title=\'use age\', ge=0, extra={\'example\': 18}), \n#   \'height\': FieldInfo(default=0.0, ge=0, le=2, extra={}), \n#   \'sex\': FieldInfo(default=0, extra={}), \n#   \'is_adult\': FieldInfo(default=False, extra={}), \n#   \'user_name\': FieldInfo(default=\'\', description=\'user name\', min_length=1, max_length=10, extra={\'example\': \'so1n\'})\n# }\n```\n\n> 注：该功能需要在通过Protobuf文件生成对应的`Python`代码时有使用[mypy-protobuf](https://github.com/nipunn1313/mypy-protobuf)插件，且指定的pyi文件输出路径与生成的`Python`代码路径时，才能生效\n\n### 2.2.2.By Protobuf file\n\n> NOTE: 需要提前安装lark库\n\n如果生成Message的原始Protobuf文件存在与项目中， 那么可以设置`parse_msg_desc_method`的值为Message生成时指定的根目录路径，\n这样`protobuf_to_pydantic`就可以通过Protobuf生成对应`Python`对象时指定的路径来获取到Message对象的protobuf文件路径，再通过解析protobuf文件获取到Message的附带信息。\n\n比如`protobuf_to_pydantic`的项目结构如下:\n```bash\n.protobuf_to_pydantic/\n├── example/\n│ ├── python_example_proto_code/\n│ └── example_proto/\n├── protobuf_to_pydantic/\n└── / \n```\n其中protobuf文件存放在`example/example_proto`文件中，然后在`example`目录下通过如下命令生成protobuf对应的`Python`代码文件:\n```bash\npython -m grpc_tools.protoc\n  --python_out=./python_example_proto_code \\\n  --grpc_python_out=./python_example_proto_code \\\n  -I. \\\n```\n那么此时需要填写的路径就是`./protobuf_to_pydantic/example`。\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(\n    demo_pb2.UserMessage, parse_msg_desc_method="./protobuf_to_pydantic/example"\n)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'uid\': FieldInfo(default=PydanticUndefined, title=\'UID\', description=\'user union id\', extra={\'example\': \'10086\'}), \n#   \'age\': FieldInfo(default=0, title=\'use age\', ge=0, extra={\'example\': 18}), \n#   \'height\': FieldInfo(default=0.0, ge=0, le=2, extra={}), \n#   \'sex\': FieldInfo(default=0, extra={}), \n#   \'is_adult\': FieldInfo(default=False, extra={}), \n#   \'user_name\': FieldInfo(default=\'\', description=\'user name\', min_length=1, max_length=10, extra={\'example\': \'so1n\'})\n# }\n```\n\n### 2.2.2.Protobuf Field Option(PGV)\n\n### 2.2.3.By PGV\n> Note 正在开发中...\n\n这是最推荐的方式，该方式参考了[protoc-gen-validate](https://github.com/envoyproxy/protoc-gen-validate)项目，大多数Protobuf文件API参考了[protoc-gen-validate](https://github.com/envoyproxy/protoc-gen-validate)项目\n\n### 2.2.4.其它参数支持\n`protobuf_to_pydantic`除了支持`FieldInfo`的参数外，还支持下面几种参数:\n- miss_default：默认情况下，生成对应`pydantic.BaseModel`对象中每个字段的默认值与Message中每个字段的默认值是一样的。不过可以通过设置`miss_default`为`true`来取消默认值的设置，需要注意的是在设置`miss_default`为`true`的情况下，`default`参数将失效。\n- enable: 默认情况下， `pydantic.BaseModel`会把Message中的每个字段都进行转换，如果有些字段不想被转换，可以设置`enable`为`false`\n\n> Note `FieldInfo`支持的参数见:https://pydantic-docs.helpmanual.io/usage/types/#constrained-types\n\n\n## 2.3.gen python code\nIn addition to generating the corresponding `pydantic.Base Model` object at runtime, `protobuf to pydantic` supports converting the runtime `pydantic.Base Model` object to `Python` code text (only applicable to `protobuf to pydantic` generated `pydantic.Base Model` object),\n\nAmong them, `protobuf_to_pydantic.pydantic_model_to_py_code` is used to generate code text, `protobuf_to_pydantic.pydantic_model_to_py_file` is used to generate code files, and the example of `protobuf_to_pydantic.pydantic_model_to_py_file` is as follows:\n```Python\nfrom protobuf_to_pydantic import msg_to_pydantic_model, pydantic_model_to_py_file\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\n# gen code: url: https://github.com/so1n/protobuf_to_pydantic/blob/master/example/demo_gen_code.py\npydantic_model_to_py_file(\n    "./demo_gen_code.py",\n    msg_to_pydantic_model(demo_pb2.NestedMessage),\n)\n\n# gen code: url: https://github.com/so1n/protobuf_to_pydantic/blob/master/example/demo_gen_code_by_pyi.py \npydantic_model_to_py_file(\n    "./demo_gen_code_by_pyi.py",\n    msg_to_pydantic_model(demo_pb2.NestedMessage, parse_msg_desc_method=demo_pb2),\n)\n\n# gen code: url: https://github.com/so1n/protobuf_to_pydantic/blob/master/example/demo_gen_code_by_protobuf_field.py\npydantic_model_to_py_file(\n    "./demo_gen_code_by_protobuf_field.py",\n    msg_to_pydantic_model(\n        demo_pb2.NestedMessage, parse_msg_desc_method="/home/so1n/github/protobuf_to_pydantic/example"\n    ),\n)\n```\nThe specific generated code can be viewed through the corresponding url. It should be noted that if `proto buf to pydantic` checks that the current environment is installed with `isort` and `autoflake`, it will format the code through them by default.\n',
    'author': 'So1n',
    'author_email': 'so1n897046026@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/so1n/protobuf_to_pydantic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
