"""
The asset server. All assets are loaded on startup and served from
memory, thus allowing blazing fast serving.
"""

import os
import re
import logging
import pkg_resources

import jinja2
import pscript
import markdown

from .. import __version__


versionstring = "v" + __version__


logger = logging.getLogger("asgineer")

image_exts = ".png", ".jpg", ".gif", ".ico", ".mp4", ".svg"
font_exts = ".ttf", ".otf", ".woff", ".woff2"

re_fas = re.compile(r"\>(\\uf[0-9a-fA-F][0-9a-fA-F][0-9a-fA-F])\<")

default_template = (
    open(pkg_resources.resource_filename("timetagger.client", "_template.html"), "rb")
    .read()
    .decode()
)
style_embed = (
    open(pkg_resources.resource_filename("timetagger.client", "_style_embed.css"), "rb")
    .read()
    .decode()
)


def md2html(text, template):
    title = description = ""
    if text.startswith("%"):
        title, text = text.split("\n", 1)
        title = title.strip("% \t\r\n")
    if text.startswith("%"):
        description, text = text.split("\n", 1)
        description = description.strip("% \t\r\n")
    title = title or "TimeTagger"
    description = description or title
    assert '"' not in description
    # Convert font-awesome codepoints to Unicode chars
    for match in reversed(list(re_fas.finditer(text))):
        text = (
            text[: match.start(1)]
            + eval("'" + match.group(1) + "'")
            + text[match.end(1) :]
        )
    # Some per-line tweaks (turn some headers into anchors, e.g. in support page)
    lines = text.splitlines()
    for i, line in enumerate(lines):
        if line.startswith(("## ", "### ")) and "|" in line:
            pre, header = line.split(" ", 1)
            linkname, header = header.split("|", 1)
            pre, linkname, line = pre.strip(), linkname.strip(), header.strip()
            line = f"<a name='{linkname}' href='#{linkname}'>{header}</a>"
            line = f"<h{len(pre)}>{line}</h{len(pre)}>"
            lines[i] = line
    text = "\n".join(lines)
    # Turn md into html and store
    main = markdown.markdown(text, extensions=[])

    return template.render(
        title=title,
        description=description,
        main=main,
        embedded_script="",
        embedded_style=style_embed,
        versionstring=versionstring,
    )


def create_assets_from_dir(dirname):
    """Get a dictionary of assets from a directory."""

    assets = {}

    thtml = default_template
    if os.path.isfile(os.path.join(dirname, "_template.html")):
        thtml = open(os.path.join(dirname, "_template.html"), "rb").read().decode()
    template = jinja2.Template(thtml)

    for fname in os.listdir(dirname):

        if fname.startswith("_"):
            continue
        elif fname.endswith(".md"):
            # Turn markdown into HTML
            text = open(os.path.join(dirname, fname), "rb").read().decode()
            html = md2html(text, template)
            name, ext = os.path.splitext(fname)
            assets["" if name == "index" else name] = html
        elif fname.endswith(".html"):
            # Raw HTML
            text = open(os.path.join(dirname, fname), "rb").read().decode()
            assets[fname[:-5]] = text
        elif fname.endswith(".py"):
            # Turn Python into JS
            name, ext = os.path.splitext(fname)
            filename = os.path.join(dirname, fname)
            # Compile
            pycode = open(filename, "rb").read().decode()
            parser = pscript.Parser(pycode, filename)
            jscode = "/* Do not edit, autogenerated by pscript */\n\n" + parser.dump()
            # Wrap in module
            exports = [
                name for name in parser.vars.get_defined() if not name.startswith("_")
            ]
            jscode = pscript.create_js_module(name, jscode, [], exports, "simple")
            assets[fname[:-2] + "js"] = jscode.encode()
            logger.info(f"Compiled pscript from {fname}")
        elif fname.endswith((".txt", ".js", ".css", ".json")):
            # Text assets
            assets[fname] = open(os.path.join(dirname, fname), "rb").read().decode()
        elif fname.endswith(image_exts + font_exts):
            # Binary assets
            assets[fname] = open(os.path.join(dirname, fname), "rb").read()
        else:
            continue  # Skip unknown extensions

    logger.info(f"Collected {len(assets)} assets from {dirname}")
    return assets
