import os
import time

import pytest
import sh

from .architecture.dvcs.git import Git
from .architecture.dvcs.hg import Hg
from .architecture.forge.gitea import Gitea
from .architecture.forge.gitlab import GitLab
from .domain.fedeproxy import Fedeproxy

forge_params = [
    (Gitea, f"http://{os.environ.get('MY_IP', '0.0.0.0')}:8781"),
    (GitLab, f"http://{os.environ.get('MY_IP', '0.0.0.0')}:8181"),
]
forge_ids = [p[0].__name__ for p in forge_params]


def skip_excluded_forges(request):
    marker = request.node.get_closest_marker("forges")
    (Forge, url) = request.param
    if marker is not None:
        forges = marker.args[0]
        if Forge.__name__ not in forges:
            pytest.skip(f"the test has a 'mark' excluding {Forge.__name__} ")


@pytest.fixture(params=forge_params, ids=forge_ids)
def fedeproxy(request, tmpdir, make_user, testuser, password, testproject, make_project):
    skip_excluded_forges(request)
    (Forge, url) = request.param
    f = Fedeproxy(Forge, url, f"{tmpdir}/fedeproxy")

    make_user(f.own, f.own.owner)
    f.own.authenticate(username=f.own.owner, password=password)
    make_project(f.own, f.own.namespace, "fedeproxy")

    f.init()

    make_user(f.forge, testuser)
    f.forge.authenticate(username=testuser, password=password)
    make_project(f.forge, testuser, testproject)

    return f


@pytest.fixture(params=forge_params, ids=forge_ids)
def forge(request):
    skip_excluded_forges(request)
    (Forge, url) = request.param
    return Forge(url)


@pytest.fixture(params=forge_params, ids=forge_ids)
def forge_factory(request):
    return request.param


@pytest.fixture
def password():
    return "Wrobyak4"


@pytest.fixture
def testuser():
    return "testuser1"


@pytest.fixture
def testproject():
    return "testproject"


@pytest.fixture
def make_user(password):

    contexts = []

    def _make_user(forge, username):
        forge.authenticate(username="root", password="Wrobyak4")
        forge.users.delete(username)
        email = f"{username}@example.com"
        user = forge.users.create(username, password, email)
        contexts.append((forge, username))
        return user

    yield _make_user

    for (forge, username) in contexts:
        forge.authenticate(username="root", password="Wrobyak4")
        forge.users.delete(username)


@pytest.fixture
def make_project(password):

    contexts = []

    def _make_project(forge, username, project):
        forge.project_delete(username, project)
        p = forge.project_create(username, project)
        contexts.append((forge, username, project))
        return p

    yield _make_project

    for (forge, username, project) in contexts:
        forge.authenticate(username="root", password=password)
        forge.project_delete(username, project)


class Repository(object):
    def __init__(self, d):
        self.d = d

    def populate(self):
        open(f"{self.d}/README.md", "w").write("# testrepo ")
        open(f"{self.d}/info.txt", "w").write("# someinfo")
        open(f"{self.d}/time.txt", "w").write(f"time {time.time()}")
        return self


class RepositoryGit(Repository):
    def commit(self):
        sh.git("-C", self.d, "add", "README.md", "info.txt", "time.txt")
        sh.git("-C", self.d, "commit", "-m", "initial")
        return self

    def prepare_for_test(self):
        sh.git.init(self.d)
        self.populate()
        self.commit()
        sh.git("-C", self.d, "checkout", "-b", "otherbranch")


class RepositoryHg(Repository):
    def commit(self):
        sh.hg("--cwd", self.d, "add", "README.md", "info.txt", "time.txt")
        sh.hg("--cwd", self.d, "commit", "-m", "initial")
        return self

    def prepare_for_test(self):
        sh.hg.init(self.d)
        open(f"{self.d}/.hgkeep", "w").write("# init")
        sh.hg("--cwd", self.d, "add", ".hgkeep")
        sh.hg("--cwd", self.d, "commit", "-m", "initial")

        sh.hg("--cwd", self.d, "branch", "master")
        self.populate()
        self.commit()

        sh.hg("--cwd", self.d, "branch", "otherbranch")


@pytest.fixture(
    params=[
        (Git, RepositoryGit),
        (Hg, RepositoryHg),
    ],
    ids=[
        "git",
        "hg",
    ],
)
def dvcs_factory(request):
    return request.param
