"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unionOf = exports.diffKeyedEntities = exports.deepEqual = void 0;
/**
 * Compares two objects for equality, deeply. The function handles arguments that are
 * +null+, +undefined+, arrays and objects. For objects, the function will not take the
 * object prototype into account for the purpose of the comparison, only the values of
 * properties reported by +Object.keys+.
 *
 * If both operands can be parsed to equivalent numbers, will return true.
 * This makes diff consistent with CloudFormation, where a numeric 10 and a literal "10"
 * are considered equivalent.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function deepEqual(lvalue, rvalue) {
    if (lvalue === rvalue) {
        return true;
    }
    // CloudFormation allows passing strings into boolean-typed fields
    if (((typeof lvalue === 'string' && typeof rvalue === 'boolean') ||
        (typeof lvalue === 'boolean' && typeof rvalue === 'string')) &&
        lvalue.toString() === rvalue.toString()) {
        return true;
    }
    // allows a numeric 10 and a literal "10" to be equivalent;
    // this is consistent with CloudFormation.
    if ((typeof lvalue === 'string' || typeof rvalue === 'string') &&
        safeParseFloat(lvalue) === safeParseFloat(rvalue)) {
        return true;
    }
    if (typeof lvalue !== typeof rvalue) {
        return false;
    }
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        return false;
    }
    if (Array.isArray(lvalue) /* && Array.isArray(rvalue) */) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            if (!deepEqual(lvalue[i], rvalue[i])) {
                return false;
            }
        }
        return true;
    }
    if (typeof lvalue === 'object' /* && typeof rvalue === 'object' */) {
        if (lvalue === null || rvalue === null) {
            // If both were null, they'd have been ===
            return false;
        }
        const keys = Object.keys(lvalue);
        if (keys.length !== Object.keys(rvalue).length) {
            return false;
        }
        for (const key of keys) {
            if (!rvalue.hasOwnProperty(key)) {
                return false;
            }
            if (key === 'DependsOn') {
                if (!dependsOnEqual(lvalue[key], rvalue[key])) {
                    return false;
                }
                ;
                // check differences other than `DependsOn`
                continue;
            }
            if (!deepEqual(lvalue[key], rvalue[key])) {
                return false;
            }
        }
        return true;
    }
    // Neither object, nor array: I deduce this is primitive type
    // Primitive type and not ===, so I deduce not deepEqual
    return false;
}
exports.deepEqual = deepEqual;
/**
 * Compares two arguments to DependsOn for equality.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function dependsOnEqual(lvalue, rvalue) {
    // allows ['Value'] and 'Value' to be equal
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        const array = Array.isArray(lvalue) ? lvalue : rvalue;
        const nonArray = Array.isArray(lvalue) ? rvalue : lvalue;
        if (array.length === 1 && deepEqual(array[0], nonArray)) {
            return true;
        }
        return false;
    }
    // allows arrays passed to DependsOn to be equivalent irrespective of element order
    if (Array.isArray(lvalue) && Array.isArray(rvalue)) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            for (let j = 0; j < lvalue.length; j++) {
                if ((!deepEqual(lvalue[i], rvalue[j])) && (j === lvalue.length - 1)) {
                    return false;
                }
                break;
            }
        }
        return true;
    }
    return false;
}
/**
 * Produce the differences between two maps, as a map, using a specified diff function.
 *
 * @param oldValue  the old map.
 * @param newValue  the new map.
 * @param elementDiff the diff function.
 *
 * @returns a map representing the differences between +oldValue+ and +newValue+.
 */
function diffKeyedEntities(oldValue, newValue, elementDiff) {
    const result = {};
    for (const logicalId of unionOf(Object.keys(oldValue || {}), Object.keys(newValue || {}))) {
        const oldElement = oldValue && oldValue[logicalId];
        const newElement = newValue && newValue[logicalId];
        result[logicalId] = elementDiff(oldElement, newElement, logicalId);
    }
    return result;
}
exports.diffKeyedEntities = diffKeyedEntities;
/**
 * Computes the union of two sets of strings.
 *
 * @param lv the left set of strings.
 * @param rv the right set of strings.
 *
 * @returns a new array containing all elemebts from +lv+ and +rv+, with no duplicates.
 */
function unionOf(lv, rv) {
    const result = new Set(lv);
    for (const v of rv) {
        result.add(v);
    }
    return new Array(...result);
}
exports.unionOf = unionOf;
/**
 * A parseFloat implementation that does the right thing for
 * strings like '0.0.0'
 * (for which JavaScript's parseFloat() returns 0).
 */
function safeParseFloat(str) {
    const ret = parseFloat(str);
    if (ret === 0) {
        // if the str is exactly '0', that's OK;
        // but parseFloat() also returns 0 for things like '0.0';
        // in this case, return NaN, so we'll fall back to string comparison
        return str === '0' ? ret : NaN;
    }
    else {
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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