"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.annotateMatcher = exports.matcherFrom = exports.encodedJson = exports.Capture = exports.notMatching = exports.anything = exports.stringLike = exports.match = exports.failMatcher = exports.arrayWith = exports.exactValue = exports.deepObjectLike = exports.objectLike = void 0;
const have_resource_1 = require("./have-resource");
/**
 * A matcher for an object that contains at least the given fields with the given matchers (or literals)
 *
 * Only does lenient matching one level deep, at the next level all objects must declare the
 * exact expected keys again.
 */
function objectLike(pattern) {
    return _objectContaining(pattern, false);
}
exports.objectLike = objectLike;
/**
 * A matcher for an object that contains at least the given fields with the given matchers (or literals)
 *
 * Switches to "deep" lenient matching. Nested objects also only need to contain declared keys.
 */
function deepObjectLike(pattern) {
    return _objectContaining(pattern, true);
}
exports.deepObjectLike = deepObjectLike;
function _objectContaining(pattern, deep) {
    const anno = { [deep ? '$deepObjectLike' : '$objectLike']: pattern };
    return annotateMatcher(anno, (value, inspection) => {
        if (typeof value !== 'object' || !value) {
            return failMatcher(inspection, `Expect an object but got '${typeof value}'`);
        }
        const errors = new Array();
        for (const [patternKey, patternValue] of Object.entries(pattern)) {
            if (patternValue === have_resource_1.ABSENT) {
                if (value[patternKey] !== undefined) {
                    errors.push(`Field ${patternKey} present, but shouldn't be`);
                }
                continue;
            }
            if (!(patternKey in value)) {
                errors.push(`Field ${patternKey} missing`);
                continue;
            }
            // If we are doing DEEP objectLike, translate object literals in the pattern into
            // more `deepObjectLike` matchers, even if they occur in lists.
            const matchValue = deep ? deepMatcherFromObjectLiteral(patternValue) : patternValue;
            const innerInspection = { ...inspection, failureReason: '' };
            const valueMatches = match(value[patternKey], matchValue, innerInspection);
            if (!valueMatches) {
                errors.push(`Field ${patternKey} mismatch: ${innerInspection.failureReason}`);
            }
        }
        /**
         * Transform nested object literals into more deep object matchers, if applicable
         *
         * Object literals in lists are also transformed.
         */
        function deepMatcherFromObjectLiteral(nestedPattern) {
            if (isObject(nestedPattern)) {
                return deepObjectLike(nestedPattern);
            }
            if (Array.isArray(nestedPattern)) {
                return nestedPattern.map(deepMatcherFromObjectLiteral);
            }
            return nestedPattern;
        }
        if (errors.length > 0) {
            return failMatcher(inspection, errors.join(', '));
        }
        return true;
    });
}
/**
 * Match exactly the given value
 *
 * This is the default, you only need this to escape from the deep lenient matching
 * of `deepObjectLike`.
 */
function exactValue(expected) {
    const anno = { $exactValue: expected };
    return annotateMatcher(anno, (value, inspection) => {
        return matchLiteral(value, expected, inspection);
    });
}
exports.exactValue = exactValue;
/**
 * A matcher for a list that contains all of the given elements in any order
 */
function arrayWith(...elements) {
    if (elements.length === 0) {
        return anything();
    }
    const anno = { $arrayContaining: elements.length === 1 ? elements[0] : elements };
    return annotateMatcher(anno, (value, inspection) => {
        if (!Array.isArray(value)) {
            return failMatcher(inspection, `Expect an array but got '${typeof value}'`);
        }
        for (const element of elements) {
            const failure = longestFailure(value, element);
            if (failure) {
                return failMatcher(inspection, `Array did not contain expected element, closest match at index ${failure[0]}: ${failure[1]}`);
            }
        }
        return true;
        /**
         * Return 'null' if the matcher matches anywhere in the array, otherwise the longest error and its index
         */
        function longestFailure(array, matcher) {
            let fail = null;
            for (let i = 0; i < array.length; i++) {
                const innerInspection = { ...inspection, failureReason: '' };
                if (match(array[i], matcher, innerInspection)) {
                    return null;
                }
                if (fail === null || innerInspection.failureReason.length > fail[1].length) {
                    fail = [i, innerInspection.failureReason];
                }
            }
            return fail;
        }
    });
}
exports.arrayWith = arrayWith;
/**
 * Whether a value is an object
 */
function isObject(x) {
    // Because `typeof null === 'object'`.
    return x && typeof x === 'object';
}
/**
 * Helper function to make matcher failure reporting a little easier
 *
 * Our protocol is weird (change a string on a passed-in object and return 'false'),
 * but I don't want to change that right now.
 */
function failMatcher(inspection, error) {
    inspection.failureReason = error;
    return false;
}
exports.failMatcher = failMatcher;
/**
 * Match a given literal value against a matcher
 *
 * If the matcher is a callable, use that to evaluate the value. Otherwise, the values
 * must be literally the same.
 */
function match(value, matcher, inspection) {
    if (isCallable(matcher)) {
        // Custom matcher (this mostly looks very weird because our `InspectionFailure` signature is weird)
        const innerInspection = { ...inspection, failureReason: '' };
        const result = matcher(value, innerInspection);
        if (typeof result !== 'boolean') {
            return failMatcher(inspection, `Predicate returned non-boolean return value: ${result}`);
        }
        if (!result && !innerInspection.failureReason) {
            // Custom matcher neglected to return an error
            return failMatcher(inspection, 'Predicate returned false');
        }
        // Propagate inner error in case of failure
        if (!result) {
            inspection.failureReason = innerInspection.failureReason;
        }
        return result;
    }
    return matchLiteral(value, matcher, inspection);
}
exports.match = match;
/**
 * Match a literal value at the top level.
 *
 * When recursing into arrays or objects, the nested values can be either matchers
 * or literals.
 */
function matchLiteral(value, pattern, inspection) {
    if (pattern == null) {
        return true;
    }
    const errors = new Array();
    if (Array.isArray(value) !== Array.isArray(pattern)) {
        return failMatcher(inspection, 'Array type mismatch');
    }
    if (Array.isArray(value)) {
        if (pattern.length !== value.length) {
            return failMatcher(inspection, 'Array length mismatch');
        }
        // Recurse comparison for individual objects
        for (let i = 0; i < pattern.length; i++) {
            if (!match(value[i], pattern[i], { ...inspection })) {
                errors.push(`Array element ${i} mismatch`);
            }
        }
        if (errors.length > 0) {
            return failMatcher(inspection, errors.join(', '));
        }
        return true;
    }
    if ((typeof value === 'object') !== (typeof pattern === 'object')) {
        return failMatcher(inspection, 'Object type mismatch');
    }
    if (typeof pattern === 'object') {
        // Check that all fields in the pattern have the right value
        const innerInspection = { ...inspection, failureReason: '' };
        const matcher = objectLike(pattern)(value, innerInspection);
        if (!matcher) {
            inspection.failureReason = innerInspection.failureReason;
            return false;
        }
        // Check no fields uncovered
        const realFields = new Set(Object.keys(value));
        for (const key of Object.keys(pattern)) {
            realFields.delete(key);
        }
        if (realFields.size > 0) {
            return failMatcher(inspection, `Unexpected keys present in object: ${Array.from(realFields).join(', ')}`);
        }
        return true;
    }
    if (value !== pattern) {
        return failMatcher(inspection, 'Different values');
    }
    return true;
}
/**
 * Whether a value is a callable
 */
function isCallable(x) {
    return x && {}.toString.call(x) === '[object Function]';
}
/**
 * Do a glob-like pattern match (which only supports *s)
 */
function stringLike(pattern) {
    // Replace * with .* in the string, escape the rest and brace with ^...$
    const regex = new RegExp(`^${pattern.split('*').map(escapeRegex).join('.*')}$`);
    return annotateMatcher({ $stringContaining: pattern }, (value, failure) => {
        if (typeof value !== 'string') {
            failure.failureReason = `Expected a string, but got '${typeof value}'`;
            return false;
        }
        if (!regex.test(value)) {
            failure.failureReason = 'String did not match pattern';
            return false;
        }
        return true;
    });
}
exports.stringLike = stringLike;
/**
 * Matches any value
 */
function anything() {
    return annotateMatcher({ $anything: true }, () => true);
}
exports.anything = anything;
/**
 * Negate an inner matcher
 */
function notMatching(matcher) {
    return annotateMatcher({ $notMatching: matcher }, (value, failure) => {
        const result = matcherFrom(matcher)(value, failure);
        if (result) {
            failure.failureReason = 'Should not have matched, but did';
            return false;
        }
        return true;
    });
}
exports.notMatching = notMatching;
/**
 * Captures a value onto an object if it matches a given inner matcher
 *
 * @example
 *
 * const someValue = Capture.aString();
 * expect(stack).toHaveResource({
 *    // ...
 *    Value: someValue.capture(stringMatching('*a*')),
 * });
 * console.log(someValue.capturedValue);
 */
class Capture {
    constructor(typeValidator) {
        this.typeValidator = typeValidator;
        this._didCapture = false;
        this._wasInvoked = false;
    }
    /**
     * A Capture object that captures any type
     */
    static anyType() {
        return new Capture();
    }
    /**
     * A Capture object that captures a string type
     */
    static aString() {
        return new Capture((x) => {
            if (typeof x !== 'string') {
                throw new Error(`Expected to capture a string, got '${x}'`);
            }
            return true;
        });
    }
    /**
     * A Capture object that captures a custom type
     */
    // eslint-disable-next-line @typescript-eslint/no-shadow
    static a(validator) {
        return new Capture(validator);
    }
    /**
     * Capture the value if the inner matcher successfully matches it
     *
     * If no matcher is given, `anything()` is assumed.
     *
     * And exception will be thrown if the inner matcher returns `true` and
     * the value turns out to be of a different type than the `Capture` object
     * is expecting.
     */
    capture(matcher) {
        if (matcher === undefined) {
            matcher = anything();
        }
        return annotateMatcher({ $capture: matcher }, (value, failure) => {
            this._wasInvoked = true;
            const result = matcherFrom(matcher)(value, failure);
            if (result) {
                if (this.typeValidator && !this.typeValidator(value)) {
                    throw new Error(`Value not of the expected type: ${value}`);
                }
                this._didCapture = true;
                this._value = value;
            }
            return result;
        });
    }
    /**
     * Whether a value was successfully captured
     */
    get didCapture() {
        return this._didCapture;
    }
    /**
     * Return the value that was captured
     *
     * Throws an exception if now value was captured
     */
    get capturedValue() {
        // When this module is ported to jsii, the type parameter will obviously
        // have to be dropped and this will have to turn into an `any`.
        if (!this.didCapture) {
            throw new Error(`Did not capture a value: ${this._wasInvoked ? 'inner matcher failed' : 'never invoked'}`);
        }
        return this._value;
    }
}
exports.Capture = Capture;
/**
 * Match on the innards of a JSON string, instead of the complete string
 */
function encodedJson(matcher) {
    return annotateMatcher({ $encodedJson: matcher }, (value, failure) => {
        if (typeof value !== 'string') {
            failure.failureReason = `Expected a string, but got '${typeof value}'`;
            return false;
        }
        let decoded;
        try {
            decoded = JSON.parse(value);
        }
        catch (e) {
            failure.failureReason = `String is not JSON: ${e}`;
            return false;
        }
        return matcherFrom(matcher)(decoded, failure);
    });
}
exports.encodedJson = encodedJson;
function escapeRegex(s) {
    return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
}
/**
 * Make a matcher out of the given argument if it's not a matcher already
 *
 * If it's not a matcher, it will be treated as a literal.
 */
function matcherFrom(matcher) {
    return isCallable(matcher) ? matcher : exactValue(matcher);
}
exports.matcherFrom = matcherFrom;
/**
 * Annotate a matcher with toJSON
 *
 * We will JSON.stringify() values if we have a match failure, but for matchers this
 * would show (in traditional JS fashion) something like '[function Function]', or more
 * accurately nothing at all since functions cannot be JSONified.
 *
 * We override to JSON() in order to produce a readadable version of the matcher.
 */
function annotateMatcher(how, matcher) {
    matcher.toJSON = () => how;
    return matcher;
}
exports.annotateMatcher = annotateMatcher;
//# sourceMappingURL=data:application/json;base64,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