# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ovld']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'ovld',
    'version': '0.2.11',
    'description': 'Overloading Python functions',
    'long_description': '\n# Ovld\n\nMultiple dispatch in Python, with some extra features.\n\nWith ovld, you can write a version of the same function for every type signature using annotations instead of writing an awkward sequence of `isinstance` statements. Unlike Python `singledispatch`, it works for multiple arguments.\n\nOther features of `ovld`:\n\n* Multiple dispatch for methods (with `metaclass=ovld.OvldMC`)\n* Create variants of functions\n* Built-in support for extensible, stateful recursion\n* Function wrappers\n* Function postprocessors\n* Nice stack traces\n\n## Example\n\nHere\'s a function that adds lists, tuples and dictionaries:\n\n```python\nfrom ovld import ovld\n\n@ovld\ndef add(x: list, y: list):\n    return [add(a, b) for a, b in zip(x, y)]\n\n@ovld\ndef add(x: tuple, y: tuple):\n    return tuple(add(a, b) for a, b in zip(x, y))\n\n@ovld\ndef add(x: dict, y: dict):\n    return {k: add(v, y[k]) for k, v in x.items()}\n\n@ovld\ndef add(x: object, y: object):\n    return x + y\n```\n\n## Bootstrapping and variants\n\nNow, there is another way to do this using ovld\'s *auto-bootstrapping*. Simply list `self` as the first argument to the function, and `self` will be bound to the function itself, so you can call `self(x, y)` for the recursion instead of `add(x, y)`:\n\n\n```python\n@ovld\ndef add(self, x: list, y: list):\n    return [self(a, b) for a, b in zip(x, y)]\n\n@ovld\ndef add(self, x: tuple, y: tuple):\n    return tuple(self(a, b) for a, b in zip(x, y))\n\n@ovld\ndef add(self, x: dict, y: dict):\n    return {k: self(v, y[k]) for k, v in x.items()}\n\n@ovld\ndef add(self, x: object, y: object):\n    return x + y\n```\n\nWhy is this useful, though? Observe:\n\n```python\n@add.variant\ndef mul(self, x: object, y: object):\n    return x * y\n\nassert add([1, 2], [3, 4]) == [4, 6]\nassert mul([1, 2], [3, 4]) == [3, 8]\n```\n\nA `variant` of a function is a copy which inherits all of the original\'s implementations but may define new ones. And because `self` is bound to the function that\'s called at the top level, the implementations for `list`, `tuple` and `dict` will bind `self` to `add` or `mul` depending on which one was called. You may also call `self.super(*args)` to invoke the parent implementation for that type.\n\n## State\n\nYou can pass `initial_state` to `@ovld` or `variant`. The initial state must be a function that takes no arguments. Its return value will be available in `self.state`. The state is initialized at the top level call, but recursive calls to `self` will preserve it.\n\nIn other words, you can do something like this:\n\n```python\n@add.variant(initial_state=lambda: 0)\ndef count(self, x, y):\n    self.state += 1\n    return (f"#{self.state}", x + y)\n\nassert count([1, 2, 3], [4, 5, 6]) == [("#1", 5), ("#2", 7), ("#3", 9)]\n```\n\nThe initial_state function can return any object and you can use the state to any purpose (e.g. cache or memoization).\n\n## Custom dispatch\n\nYou can define your own dispatching function. The dispatcher\'s first argument is always `self`.\n\n* `self.resolve(x, y)` to get the right function for the types of x and y\n* `self[type(x), type(y)]` will also return the right function for these types, but it works directly with the types.\n\nFor example, here is how you might define a function such that f(x) <=> f(x, x):\n\n```python\n@ovld.dispatch\ndef add_default(self, x, y=None):\n    if y is None:\n        y = x\n    return self.resolve(x, y)(x, y)\n\n@ovld\ndef add_default(x: int, y: int):\n    return x + y\n\n@ovld\ndef add_default(x: str, y: str):\n    return x + y\n\n@ovld\ndef add_default(xs: list, ys: list):\n    return [add_default(x, y) for x, y in zip(xs, ys)]\n\nassert add_default([1, 2, "alouette"]) == [2, 4, "alouettealouette"]\n```\n\nThere are other uses for this feature, e.g. memoization.\n\nThe normal functions may also have a `self`, which works the same as bootstrapping, and you can give an `initial_state` to `@ovld.dispatch` as well.\n\n## Postprocess\n\n`@ovld`, `@ovld.dispatch`, etc. take a `postprocess` argument which should be a function of one argument. That function will be called with the result of the call and must return the final result of the call.\n\nNote that intermediate, bootstrapped recursive calls (recursive calls using `self()`) will **not** be postprocessed (if you want to wrap these calls, you can do so otherwise, like defining a custom dispatch). Only the result of the top level call is postprocessed.\n\n## Methods\n\nUse the `OvldMC` metaclass to use multiple dispatch on methods. In this case there is no bootstrapping as described above and `self` is simply bound to the class instance.\n\n```python\nfrom ovld import OvldMC, ovld\n\nclass Cat(metaclass=OvldMC):\n    @ovld\n    def interact(self, x: Mouse):\n        return "catch"\n\n    def interact(self, x: Food):\n        return "devour"\n\n    def interact(self, x: PricelessVase):\n        return "destroy"\n```\n\nSubclasses of `Cat` will inherit the overloaded `interact` and it may define additional overloaded methods which will only be valid for the subclass.\n\n**Note:** It is possible to use `ovld.dispatch` on methods, but in this case be aware that the first argument for the dispatch method will not be the usual `self` but an `OvldCall` object. The `self` can be retrived as `ovldcall.obj`. Here\'s an example to make it all clear:\n\n```python\nclass Stuff(metaclass=OvldMC):\n    def __init__(self, mul):\n        self.mul = mul\n\n    @ovld.dispatch\n    def calc(ovldcall, x):\n        # Wraps every call to self.calc, but we receive ovldcall instead of self\n        # ovldcall[type(x)] returns the right method to call\n        # ovldcall.obj is the self (the actual instance of Stuff)\n        return ovldcall[type(x)](x) * ovldcall.obj.mul\n\n    def calc(self, x: int):\n        return x + 1\n\n    def calc(self, xs: list):\n        return [self.calc(x) for x in xs]\n\nprint(Stuff(2).calc([1, 2, 3]))  # [4, 6, 8, 4, 6, 8]\n```\n\n## Ambiguous calls\n\nThe following definitions will cause a TypeError at runtime when called with two ints, because it is unclear which function is the right match:\n\n```python\n@ovld\ndef ambig(x: int, y: object):\n    print("io")\n\n@ovld\ndef ambig(x: object, y: int):\n    print("oi")\n\nambig(8, 8)  # ???\n```\n\nYou may define an additional function with signature (int, int) to disambiguate:\n\n```python\n@ovld\ndef ambig(x: int, y: int):\n    print("ii")\n```\n\n## Other features\n\n### meta\n\nTo test arbitrary conditions, you can use `meta`:\n\n```python\nfrom ovld import ovld, meta\n\n@meta\ndef StartsWithT(cls):\n    return cls.__name__.startswith("T")\n\n@ovld\ndef f(x: StartsWithT):\n    return "T"\n\nassert f(TypeError("xyz")) == "T"\n\n\n# Or: a useful example, since dataclasses have no common superclass:\n\nfrom dataclasses import dataclass, is_dataclass\n\n@dataclass\nclass Point:\n    x: int\n    y: int\n\n@ovld\ndef f(x: meta(is_dataclass)):\n    return "dataclass"\n\nassert f(Point(1, 2)) == "dataclass"\n```\n\n\n### deferred\n\nYou may define overloads for certain classes from external packages without\nhaving to import them:\n\n\n```python\nfrom ovld import ovld, deferred\n\n@ovld\ndef f(x: deferred("numpy.ndarray")):\n    return "ndarray"\n\n# numpy is not imported\nassert "numpy" not in sys.modules\n\n# But once we import it, the ovld works:\nimport numpy\nassert f(numpy.arange(10)) == "ndarray"\n```\n\n\n### Tracebacks\n\n`ovld` automagically renames functions so that the stack trace is more informative:\n\n```python\n@add.variant\ndef bad(self, x: object, y: object):\n    raise Exception("Bad.")\n\nbad([1], [2])\n\n"""\n  File "/Users/breuleuo/code/ovld/ovld/core.py", line 148, in bad.entry\n    res = ovc(*args, **kwargs)\n  File "/Users/breuleuo/code/ovld/ovld/core.py", line 182, in bad.dispatch\n    return method(self.bind_to, *args, **kwargs)\n  File "example.py", line 6, in bad[list, list]\n    return [self(a, b) for a, b in zip(x, y)]\n  File "example.py", line 6, in <listcomp>\n    return [self(a, b) for a, b in zip(x, y)]\n  File "/Users/breuleuo/code/ovld/ovld/core.py", line 182, in bad.dispatch\n    return method(self.bind_to, *args, **kwargs)\n  File "example.py", line 26, in bad[*, *]\n    raise Exception("Bad.")\n  Exception: Bad.\n"""\n```\n\nThe functions on the stack have names like `bad.entry`, `bad.dispatch`, `bad[list, list]` and `bad[*, *]` (`*` stands for `object`), which lets you better understand what happened just from the stack trace.\n\nThis also means profilers will be able to differentiate between these paths and between variants, even if they share code paths.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/ovld',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
