"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A database of regional information.
 */
class Fact {
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        return Object.keys(this.database);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        regionFacts[fact.name] = fact.value;
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
}
exports.Fact = Fact;
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
}
exports.FactName = FactName;
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZhY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQTs7R0FFRztBQUNILE1BQWEsSUFBSTtJQWtFYjtRQUNJLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBbkVEOzs7T0FHRztJQUNJLE1BQU0sS0FBSyxPQUFPO1FBQ3JCLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQWMsRUFBRSxJQUFZO1FBQzNDLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDMUMsT0FBTyxXQUFXLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUFDLE1BQWMsRUFBRSxJQUFZO1FBQ2xELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDWixNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsSUFBSSwrQkFBK0IsTUFBTSxjQUFjLElBQUksRUFBRSxDQUFDLENBQUM7U0FDN0Y7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNyQixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQVcsRUFBRSxjQUFjLEdBQUcsS0FBSztRQUN0RCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3BGLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxXQUFXLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3RGLE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLENBQUMsTUFBTSx1QkFBdUIsSUFBSSxDQUFDLElBQUksZ0JBQWdCLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ2xIO1FBQ0QsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3hDLENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQWMsRUFBRSxJQUFZLEVBQUUsS0FBYztRQUNqRSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNoRCxJQUFJLElBQUksSUFBSSxXQUFXLElBQUksS0FBSyxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxLQUFLLEVBQUU7WUFDN0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsSUFBSSxTQUFTLE1BQU0sZUFBZSxLQUFLLDZCQUE2QixXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ25JO1FBQ0QsT0FBTyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDN0IsQ0FBQzs7QUE1REwsb0JBcUVDO0FBUjJCLGFBQVEsR0FJNUIsRUFBRSxDQUFDO0FBc0JYOztHQUVHO0FBQ0gsTUFBYSxRQUFRO0lBNEJqQjs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsT0FBZTtRQUMxQyxPQUFPLHFCQUFxQixPQUFPLENBQUMsT0FBTyxDQUFDLDBCQUEwQixFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUM7SUFDbEYsQ0FBQzs7QUFyQ0wsNEJBc0NDO0FBckNHOztHQUVHO0FBQ29CLGtCQUFTLEdBQUcsV0FBVyxDQUFDO0FBQy9DOztHQUVHO0FBQ29CLHNCQUFhLEdBQUcsY0FBYyxDQUFDO0FBQ3REOzs7R0FHRztBQUNvQix3Q0FBK0IsR0FBRyxpQ0FBaUMsQ0FBQztBQUMzRjs7R0FFRztBQUNvQixtQ0FBMEIsR0FBRyw0QkFBNEIsQ0FBQztBQUNqRjs7R0FFRztBQUNvQixpREFBd0MsR0FBRywyQ0FBMkMsQ0FBQztBQUM5Rzs7OztHQUlHO0FBQ29CLHlDQUFnQyxHQUFHLDhCQUE4QixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBBIGRhdGFiYXNlIG9mIHJlZ2lvbmFsIGluZm9ybWF0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgRmFjdCB7XG4gICAgLyoqXG4gICAgICogQHJldHVybnMgdGhlIGxpc3Qgb2YgbmFtZXMgb2YgQVdTIHJlZ2lvbnMgZm9yIHdoaWNoIHRoZXJlIGlzIGF0IGxlYXN0IG9uZSByZWdpc3RlcmVkIGZhY3QuIFRoaXNcbiAgICAgKiAgICAgICAgICBtYXkgbm90IGJlIGFuIGV4aGF1c3RpdmUgbGlzdCBvZiBhbGwgYXZhaWxhYmxlIEFXUyByZWdpb25zLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZ2V0IHJlZ2lvbnMoKTogc3RyaW5nW10ge1xuICAgICAgICByZXR1cm4gT2JqZWN0LmtleXModGhpcy5kYXRhYmFzZSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHJpZXZlcyBhIGZhY3QgZnJvbSB0aGlzIEZhY3QgZGF0YWJhc2UuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gcmVnaW9uIHRoZSBuYW1lIG9mIHRoZSByZWdpb24gKGUuZzogYHVzLWVhc3QtMWApXG4gICAgICogQHBhcmFtIG5hbWUgICB0aGUgbmFtZSBvZiB0aGUgZmFjdCBiZWluZyBsb29rZWQgdXAgKHNlZSB0aGUgYEZhY3ROYW1lYCBjbGFzcyBmb3IgZGV0YWlscylcbiAgICAgKlxuICAgICAqIEByZXR1cm5zIHRoZSBmYWN0IHZhbHVlIGlmIGl0IGlzIGtub3duLCBhbmQgYHVuZGVmaW5lZGAgb3RoZXJ3aXNlLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZmluZChyZWdpb246IHN0cmluZywgbmFtZTogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgY29uc3QgcmVnaW9uRmFjdHMgPSB0aGlzLmRhdGFiYXNlW3JlZ2lvbl07XG4gICAgICAgIHJldHVybiByZWdpb25GYWN0cyAmJiByZWdpb25GYWN0c1tuYW1lXTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0cmlldmUgYSBmYWN0IGZyb20gdGhlIEZhY3QgZGF0YWJhc2UuIChyZXRyaWV2YWwgd2lsbCBmYWlsIGlmIHRoZSBzcGVjaWZpZWQgcmVnaW9uIG9yXG4gICAgICogZmFjdCBuYW1lIGRvZXMgbm90IGV4aXN0LilcbiAgICAgKlxuICAgICAqIEBwYXJhbSByZWdpb24gdGhlIG5hbWUgb2YgdGhlIHJlZ2lvbiAoZS5nOiBgdXMtZWFzdC0xYClcbiAgICAgKiBAcGFyYW0gbmFtZSB0aGUgbmFtZSBvZiB0aGUgZmFjdCBiZWluZyBsb29rZWQgdXAgKHNlZSB0aGUgYEZhY3ROYW1lYCBjbGFzcyBmb3IgZGV0YWlscylcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlcXVpcmVGYWN0KHJlZ2lvbjogc3RyaW5nLCBuYW1lOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgICAgICBjb25zdCBmb3VuZEZhY3QgPSB0aGlzLmZpbmQocmVnaW9uLCBuYW1lKTtcbiAgICAgICAgaWYgKCFmb3VuZEZhY3QpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgTm8gZmFjdCAke25hbWV9IGNvdWxkIGJlIGZvdW5kIGZvciByZWdpb246ICR7cmVnaW9ufSBhbmQgbmFtZTogJHtuYW1lfWApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBmb3VuZEZhY3Q7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlZ2lzdGVycyBhIG5ldyBmYWN0IGluIHRoaXMgRmFjdCBkYXRhYmFzZS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBmYWN0ICAgICAgICAgICB0aGUgbmV3IGZhY3QgdG8gYmUgcmVnaXN0ZXJlZC5cbiAgICAgKiBAcGFyYW0gYWxsb3dSZXBsYWNpbmcgd2hldGhlciBuZXcgZmFjdHMgY2FuIHJlcGxhY2UgZXhpc3RpbmcgZmFjdHMgb3Igbm90LlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVnaXN0ZXIoZmFjdDogSUZhY3QsIGFsbG93UmVwbGFjaW5nID0gZmFsc2UpOiB2b2lkIHtcbiAgICAgICAgY29uc3QgcmVnaW9uRmFjdHMgPSB0aGlzLmRhdGFiYXNlW2ZhY3QucmVnaW9uXSB8fCAodGhpcy5kYXRhYmFzZVtmYWN0LnJlZ2lvbl0gPSB7fSk7XG4gICAgICAgIGlmIChmYWN0Lm5hbWUgaW4gcmVnaW9uRmFjdHMgJiYgcmVnaW9uRmFjdHNbZmFjdC5uYW1lXSAhPT0gZmFjdC52YWx1ZSAmJiAhYWxsb3dSZXBsYWNpbmcpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUmVnaW9uICR7ZmFjdC5yZWdpb259IGFscmVhZHkgaGFzIGEgZmFjdCAke2ZhY3QubmFtZX0sIHdpdGggdmFsdWUgJHtyZWdpb25GYWN0c1tmYWN0Lm5hbWVdfWApO1xuICAgICAgICB9XG4gICAgICAgIHJlZ2lvbkZhY3RzW2ZhY3QubmFtZV0gPSBmYWN0LnZhbHVlO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZW1vdmVzIGEgZmFjdCBmcm9tIHRoZSBkYXRhYmFzZS5cbiAgICAgKiBAcGFyYW0gcmVnaW9uIHRoZSByZWdpb24gZm9yIHdoaWNoIHRoZSBmYWN0IGlzIHRvIGJlIHJlbW92ZWQuXG4gICAgICogQHBhcmFtIG5hbWUgICB0aGUgbmFtZSBvZiB0aGUgZmFjdCB0byByZW1vdmUuXG4gICAgICogQHBhcmFtIHZhbHVlICB0aGUgdmFsdWUgdGhhdCBzaG91bGQgYmUgcmVtb3ZlZCAocmVtb3ZhbCB3aWxsIGZhaWwgaWYgdGhlIHZhbHVlIGlzIHNwZWNpZmllZCwgYnV0IGRvZXMgbm90IG1hdGNoIHRoZVxuICAgICAqICAgICAgICAgICAgICAgY3VycmVudCBzdG9yZWQgdmFsdWUpLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgdW5yZWdpc3RlcihyZWdpb246IHN0cmluZywgbmFtZTogc3RyaW5nLCB2YWx1ZT86IHN0cmluZyk6IHZvaWQge1xuICAgICAgICBjb25zdCByZWdpb25GYWN0cyA9IHRoaXMuZGF0YWJhc2VbcmVnaW9uXSB8fCB7fTtcbiAgICAgICAgaWYgKG5hbWUgaW4gcmVnaW9uRmFjdHMgJiYgdmFsdWUgJiYgcmVnaW9uRmFjdHNbbmFtZV0gIT09IHZhbHVlKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEF0dGVtcHRlZCB0byByZW1vdmUgJHtuYW1lfSBmcm9tICR7cmVnaW9ufSB3aXRoIHZhbHVlICR7dmFsdWV9LCBidXQgdGhlIGZhY3QncyB2YWx1ZSBpcyAke3JlZ2lvbkZhY3RzW25hbWVdfWApO1xuICAgICAgICB9XG4gICAgICAgIGRlbGV0ZSByZWdpb25GYWN0c1tuYW1lXTtcbiAgICB9XG4gICAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgZGF0YWJhc2U6IHtcbiAgICAgICAgW3JlZ2lvbjogc3RyaW5nXToge1xuICAgICAgICAgICAgW25hbWU6IHN0cmluZ106IHN0cmluZztcbiAgICAgICAgfTtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVc2UgdGhlIHN0YXRpYyBtZXRob2RzIG9mIEZhY3QgaW5zdGVhZCEnKTtcbiAgICB9XG59XG4vKipcbiAqIEEgZmFjdCB0aGF0IGNhbiBiZSByZWdpc3RlcmVkIGFib3V0IGEgcGFydGljdWxhciByZWdpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUZhY3Qge1xuICAgIC8qKlxuICAgICAqIFRoZSByZWdpb24gZm9yIHdoaWNoIHRoaXMgZmFjdCBhcHBsaWVzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlZ2lvbjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoaXMgZmFjdC4gU3RhbmRhcmRpemVkIHZhbHVlcyBhcmUgcHJvdmlkZWQgYnkgdGhlIGBGYWN0c2AgY2xhc3MuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSB2YWx1ZSBvZiB0aGlzIGZhY3QuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbn1cbi8qKlxuICogQWxsIHN0YW5kYXJkaXplZCBmYWN0IG5hbWVzLlxuICovXG5leHBvcnQgY2xhc3MgRmFjdE5hbWUge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBwYXJ0aXRpb24gZm9yIGEgcmVnaW9uIChlLmc6ICdhd3MnLCAnYXdzLWNuJywgLi4uKVxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUEFSVElUSU9OID0gJ3BhcnRpdGlvbic7XG4gICAgLyoqXG4gICAgICogVGhlIGRvbWFpbiBzdWZmaXggZm9yIGEgcmVnaW9uIChlLmc6ICdhbWF6b25hd3MuY29tYClcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IERPTUFJTl9TVUZGSVggPSAnZG9tYWluU3VmZml4JztcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRoZSBBV1M6OkNESzo6TWV0YWRhdGEgQ2xvdWRGb3JtYXRpb24gUmVzb3VyY2UgaXMgYXZhaWxhYmxlIGluLXJlZ2lvbiBvciBub3QuIFRoZSB2YWx1ZSBpcyBhIGJvb2xlYW5cbiAgICAgKiBtb2RlbGxlZCBhcyBgWUVTYCBvciBgTk9gLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ0RLX01FVEFEQVRBX1JFU09VUkNFX0FWQUlMQUJMRSA9ICdjZGs6bWV0YWRhdGEtcmVzb3VyY2U6YXZhaWxhYmxlJztcbiAgICAvKipcbiAgICAgKiBUaGUgZW5kcG9pbnQgdXNlZCBmb3IgaG9zdGluZyBTMyBzdGF0aWMgd2Vic2l0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFMzX1NUQVRJQ19XRUJTSVRFX0VORFBPSU5UID0gJ3MzLXN0YXRpYy13ZWJzaXRlOmVuZHBvaW50JztcbiAgICAvKipcbiAgICAgKiBUaGUgZW5kcG9pbnQgdXNlZCBmb3IgYWxpYXNpbmcgUzMgc3RhdGljIHdlYnNpdGVzIGluIFJvdXRlIDUzXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBTM19TVEFUSUNfV0VCU0lURV9aT05FXzUzX0hPU1RFRF9aT05FX0lEID0gJ3MzLXN0YXRpYy13ZWJzaXRlOnJvdXRlLTUzLWhvc3RlZC16b25lLWlkJztcbiAgICAvKipcbiAgICAgKiBUaGUgcHJlZml4IGZvciBWUEMgRW5kcG9pbnQgU2VydmljZSBuYW1lcyxcbiAgICAgKiBjbi5jb20uYW1hem9uYXdzLnZwY2UgZm9yIENoaW5hIHJlZ2lvbnMsXG4gICAgICogY29tLmFtYXpvbmF3cy52cGNlIG90aGVyd2lzZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZQQ19FTkRQT0lOVF9TRVJWSUNFX05BTUVfUFJFRklYID0gJ3ZwY0VuZHBvaW50U2VydmljZU5hbWVQcmVmaXgnO1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSByZWdpb25hbCBzZXJ2aWNlIHByaW5jaXBhbCBmb3IgYSBnaXZlbiBzZXJ2aWNlLlxuICAgICAqXG4gICAgICogQHBhcmFtIHNlcnZpY2UgdGhlIHNlcnZpY2UgbmFtZSwgZWl0aGVyIHNpbXBsZSAoZS5nOiBgczNgLCBgY29kZWRlcGxveWApIG9yIHF1YWxpZmllZCAoZS5nOiBgczMuYW1hem9uYXdzLmNvbWApLlxuICAgICAqICAgICAgICAgICAgICAgIFRoZSBgLmFtYXpvbmF3cy5jb21gIGFuZCBgLmFtYXpvbmF3cy5jb20uY25gIGRvbWFpbnMgYXJlIHN0cmlwcGVkIGZyb20gc2VydmljZSBuYW1lcywgc28gdGhleSBhcmVcbiAgICAgKiAgICAgICAgICAgICAgICBjYW5vbmljYWxpemVkIGluIHRoYXQgcmVzcGVjdC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHNlcnZpY2VQcmluY2lwYWwoc2VydmljZTogc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiBgc2VydmljZS1wcmluY2lwYWw6JHtzZXJ2aWNlLnJlcGxhY2UoL1xcLmFtYXpvbmF3c1xcLmNvbShcXC5jbik/JC8sICcnKX1gO1xuICAgIH1cbn1cbiJdfQ==