"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const consts = require("./runtime/consts");
const util_1 = require("./util");
const waiter_state_machine_1 = require("./waiter-state-machine");
const RUNTIME_HANDLER_PATH = path.join(__dirname, 'runtime');
const FRAMEWORK_HANDLER_TIMEOUT = core_1.Duration.minutes(15); // keep it simple for now
/**
 * Defines an AWS CloudFormation custom resource provider.
 */
class Provider extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.isCompleteHandler && (props.queryInterval || props.totalTimeout)) {
            throw new Error('"queryInterval" and "totalTimeout" can only be configured if "isCompleteHandler" is specified. '
                + 'Otherwise, they have no meaning');
        }
        this.onEventHandler = props.onEventHandler;
        this.isCompleteHandler = props.isCompleteHandler;
        const onEventFunction = this.createFunction(consts.FRAMEWORK_ON_EVENT_HANDLER_NAME);
        if (this.isCompleteHandler) {
            const isCompleteFunction = this.createFunction(consts.FRAMEWORK_IS_COMPLETE_HANDLER_NAME);
            const timeoutFunction = this.createFunction(consts.FRAMEWORK_ON_TIMEOUT_HANDLER_NAME);
            const retry = util_1.calculateRetryPolicy(props);
            const waiterStateMachine = new waiter_state_machine_1.WaiterStateMachine(this, 'waiter-state-machine', {
                isCompleteHandler: isCompleteFunction,
                timeoutHandler: timeoutFunction,
                backoffRate: retry.backoffRate,
                interval: retry.interval,
                maxAttempts: retry.maxAttempts,
            });
            // the on-event entrypoint is going to start the execution of the waiter
            onEventFunction.addEnvironment(consts.WAITER_STATE_MACHINE_ARN_ENV, waiterStateMachine.stateMachineArn);
            waiterStateMachine.grantStartExecution(onEventFunction);
        }
        this.entrypoint = onEventFunction;
        this.serviceToken = this.entrypoint.functionArn;
    }
    /**
     * Called by `CustomResource` which uses this provider.
     * @deprecated use `provider.serviceToken` instead
     */
    bind(_) {
        return {
            serviceToken: this.entrypoint.functionArn,
        };
    }
    createFunction(entrypoint) {
        const fn = new lambda.Function(this, `framework-${entrypoint}`, {
            code: lambda.Code.fromAsset(RUNTIME_HANDLER_PATH),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: `framework.${entrypoint}`,
            timeout: FRAMEWORK_HANDLER_TIMEOUT,
        });
        fn.addEnvironment(consts.USER_ON_EVENT_FUNCTION_ARN_ENV, this.onEventHandler.functionArn);
        this.onEventHandler.grantInvoke(fn);
        if (this.isCompleteHandler) {
            fn.addEnvironment(consts.USER_IS_COMPLETE_FUNCTION_ARN_ENV, this.isCompleteHandler.functionArn);
            this.isCompleteHandler.grantInvoke(fn);
        }
        return fn;
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,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