"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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