"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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