"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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