"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Class that keeps track of the logical IDs that are assigned to resources
 *
 * Supports renaming the generated IDs.
 */
class LogicalIDs {
    constructor() {
        /**
         * The rename table (old to new)
         */
        this.renames = {};
        /**
         * All assigned names (new to old, may be identical)
         *
         * This is used to ensure that:
         *
         * - No 2 resources end up with the same final logical ID, unless they were the same to begin with.
         * - All renames have been used at the end of renaming.
         */
        this.reverse = {};
    }
    /**
     * Rename a logical ID from an old ID to a new ID
     */
    addRename(oldId, newId) {
        if (oldId in this.renames) {
            throw new Error(`A rename has already been registered for '${oldId}'`);
        }
        this.renames[oldId] = newId;
    }
    /**
     * Return the renamed version of an ID or the original ID.
     */
    applyRename(oldId) {
        let newId = oldId;
        if (oldId in this.renames) {
            newId = this.renames[oldId];
        }
        // If this newId has already been used, it must have been with the same oldId
        if (newId in this.reverse && this.reverse[newId] !== oldId) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Two objects have been assigned the same Logical ID: '${this.reverse[newId]}' and '${oldId}' are now both named '${newId}'.`);
        }
        this.reverse[newId] = oldId;
        validateLogicalId(newId);
        return newId;
    }
    /**
     * Throw an error if not all renames have been used
     *
     * This is to assure that users didn't make typoes when registering renames.
     */
    assertAllRenamesApplied() {
        const keys = new Set();
        Object.keys(this.renames).forEach(keys.add.bind(keys));
        Object.keys(this.reverse).map(newId => {
            keys.delete(this.reverse[newId]);
        });
        if (keys.size !== 0) {
            const unusedRenames = Array.from(keys.values());
            throw new Error(`The following Logical IDs were attempted to be renamed, but not found: ${unusedRenames.join(', ')}`);
        }
    }
}
exports.LogicalIDs = LogicalIDs;
const VALID_LOGICALID_REGEX = /^[A-Za-z][A-Za-z0-9]{1,254}$/;
/**
 * Validate logical ID is valid for CloudFormation
 */
function validateLogicalId(logicalId) {
    if (!VALID_LOGICALID_REGEX.test(logicalId)) {
        throw new Error(`Logical ID must adhere to the regular expression: ${VALID_LOGICALID_REGEX.toString()}, got '${logicalId}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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