"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_resource_1 = require("./cfn-resource");
const removal_policy_1 = require("./removal-policy");
const resource_1 = require("./resource");
const token_1 = require("./token");
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class CustomResource extends resource_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const pascalCaseProperties = (_a = props.pascalCaseProperties) !== null && _a !== void 0 ? _a : false;
        const properties = pascalCaseProperties ? uppercaseProperties(props.properties || {}) : (props.properties || {});
        this.resource = new cfn_resource_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.serviceToken,
                ...properties,
            },
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: removal_policy_1.RemovalPolicy.DESTROY,
        });
    }
    /**
     * The physical name of this custom resource.
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        return token_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return 'AWS::CloudFormation::CustomResource';
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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