"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const asset = stack.addFileAsset({
            fileName: staging.stagedPath,
            sourceHash: staging.sourceHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        const timeout = (_a = props.timeout) !== null && _a !== void 0 ? _a : duration_1.Duration.minutes(15);
        const memory = (_b = props.memorySize) !== null && _b !== void 0 ? _b : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: 'nodejs12.x',
            },
        });
        handler.addDependsOn(role);
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        var _a;
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_a = stack.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new CustomResourceProvider(stack, id, props);
        return provider.serviceToken;
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,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