"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_element_1 = require("./cfn-element");
const util_1 = require("./util");
/**
 * The Rules that define template constraints in an AWS Service Catalog portfolio describe when
 * end users can use the template and which values they can specify for parameters that are declared
 * in the AWS CloudFormation template used to create the product they are attempting to use. Rules
 * are useful for preventing end users from inadvertently specifying an incorrect value.
 * For example, you can add a rule to verify whether end users specified a valid subnet in a
 * given VPC or used m1.small instance types for test environments. AWS CloudFormation uses
 * rules to validate parameter values before it creates the resources for the product.
 *
 * A rule can include a RuleCondition property and must include an Assertions property.
 * For each rule, you can define only one rule condition; you can define one or more asserts within the Assertions property.
 * You define a rule condition and assertions by using rule-specific intrinsic functions.
 *
 * @link https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html
 */
class CfnRule extends cfn_element_1.CfnRefElement {
    /**
     * Creates and adds a rule.
     * @param scope The parent construct.
     * @param props The rule props.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.ruleCondition = props && props.ruleCondition;
        this.assertions = props && props.assertions;
    }
    /**
     * Adds an assertion to the rule.
     * @param condition The expression to evaluation.
     * @param description The description of the assertion.
     */
    addAssertion(condition, description) {
        if (!this.assertions) {
            this.assertions = [];
        }
        this.assertions.push({
            assert: condition,
            assertDescription: description,
        });
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Rules: {
                [this.logicalId]: {
                    RuleCondition: this.ruleCondition,
                    Assertions: util_1.capitalizePropertyNames(this, this.assertions),
                },
            },
        };
    }
}
exports.CfnRule = CfnRule;
//# sourceMappingURL=data:application/json;base64,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