"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 */
class Aws {
    constructor() { }
}
exports.Aws = Aws;
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
Aws.PARTITION = pseudoString(AWS_PARTITION);
Aws.REGION = pseudoString(AWS_REGION);
Aws.STACK_ID = pseudoString(AWS_STACKID);
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 */
class ScopedAws {
    constructor(scope) {
        this.scope = scope;
    }
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS,
        });
    }
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    get region() {
        return this.asString(AWS_REGION);
    }
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    return token_1.Token.asString({ Ref: name }, { displayHint: name });
}
//# sourceMappingURL=data:application/json;base64,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