"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_compat_1 = require("./construct-compat");
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which
 * resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences
     * this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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