"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs");
const path = require("path");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        this.sourceHash = fs_1.FileSystem.fingerprint(this.sourcePath, props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = this.sourcePath;
        }
        else {
            this.relativePath = 'asset.' + this.sourceHash + path.extname(this.sourcePath);
            this.stagedPath = this.relativePath; // always relative to outdir
        }
    }
    synthesize(session) {
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // asset already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
}
exports.AssetStaging = AssetStaging;
//# sourceMappingURL=data:application/json;base64,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